import { UserDao } from '../../user/userDao';
import { TileRow } from './tileRow';
import { TileGrid } from '../tileGrid';
import { TileMatrix } from '../matrix/tileMatrix';
import { BoundingBox } from '../../boundingBox';
import { SpatialReferenceSystem } from '../../core/srs/spatialReferenceSystem';
import { TileMatrixSet } from '../matrixset/tileMatrixSet';
import { GeoPackage } from '../../geoPackage';
import { TileTable } from './tileTable';
import { DataTypes } from '../../db/dataTypes';
import { DBValue } from '../../db/dbAdapter';
/**
 * `TileDao` is a {@link module:dao/dao~Dao} subclass for reading
 * [user tile tables]{@link module:tiles/user/tileTable~TileTable}.
 *
 * @class TileDao
 * @extends UserDao
 * @param  {GeoPackageConnection} connection
 * @param  {TileTable} table
 * @param  {TileMatrixSet} tileMatrixSet
 * @param  {TileMatrix[]} tileMatrices
 */
export declare class TileDao extends UserDao<TileRow> {
    table: TileTable;
    tileMatrixSet: TileMatrixSet;
    tileMatrices: TileMatrix[];
    zoomLevelToTileMatrix: TileMatrix[];
    widths: number[];
    heights: number[];
    minZoom: number;
    maxZoom: number;
    srs: SpatialReferenceSystem;
    projection: string;
    minWebMapZoom: number;
    maxWebMapZoom: number;
    webZoomToGeoPackageZooms: Record<number, number>;
    constructor(geoPackage: GeoPackage, table: TileTable, tileMatrixSet: TileMatrixSet, tileMatrices: TileMatrix[]);
    initialize(): void;
    webZoomToGeoPackageZoom(webZoom: number): number;
    setWebMapZoomLevels(): void;
    determineGeoPackageZoomLevel(webMercatorBoundingBox: BoundingBox, zoom: number): number;
    /**
     * Get the bounding box of tiles at the zoom level
     * @param  {Number} zoomLevel zoom level
     * @return {BoundingBox}           bounding box of the zoom level, or null if no tiles
     */
    getBoundingBoxWithZoomLevel(zoomLevel: number): BoundingBox;
    getBoundingBox(): BoundingBox;
    queryForTileGridWithZoomLevel(zoomLevel: number): TileGrid;
    /**
     * Get the tile grid of the zoom level
     * @param  {Number} zoomLevel zoom level
     * @return {TileGrid}           tile grid at zoom level, null if no tile matrix at zoom level
     */
    getTileGridWithZoomLevel(zoomLevel: number): TileGrid;
    /**
     * get the tile table
     * @return {TileTable} tile table
     */
    getTileTable(): TileTable;
    /**
     * Create a new tile row with the column types and values
     * @param  {Array} columnTypes column types
     * @param  {Array} values      values
     * @return {TileRow}             tile row
     */
    newRowWithColumnTypes(columnTypes: {
        [key: string]: DataTypes;
    }, values: Record<string, DBValue>): TileRow;
    /**
     * Create a new tile row
     * @return {TileRow} tile row
     */
    newRow(): TileRow;
    /**
     * Adjust the tile matrix lengths if needed. Check if the tile matrix width
     * and height need to expand to account for pixel * number of pixels fitting
     * into the tile matrix lengths
     */
    adjustTileMatrixLengths(): void;
    /**
     * Get the tile matrix at the zoom level
     * @param  {Number} zoomLevel zoom level
     * @returns {TileMatrix}           tile matrix
     */
    getTileMatrixWithZoomLevel(zoomLevel: number): TileMatrix;
    /**
     * Query for a tile
     * @param  {Number} column    column
     * @param  {Number} row       row
     * @param  {Number} zoomLevel zoom level
     */
    queryForTile(column: number, row: number, zoomLevel: number): TileRow;
    queryForTilesWithZoomLevel(zoomLevel: number): IterableIterator<TileRow>;
    /**
     * Query for Tiles at a zoom level in descending row and column order
     * @param  {Number} zoomLevel    zoom level
     * @returns {IterableIterator<TileRow>}
     */
    queryForTilesDescending(zoomLevel: number): IterableIterator<TileRow>;
    /**
     * Query for tiles at a zoom level and column
     * @param  {Number} column       column
     * @param  {Number} zoomLevel    zoom level
     * @returns {IterableIterator<TileRow>}
     */
    queryForTilesInColumn(column: number, zoomLevel: number): IterableIterator<TileRow>;
    /**
     * Query for tiles at a zoom level and row
     * @param  {Number} row       row
     * @param  {Number} zoomLevel    zoom level
     */
    queryForTilesInRow(row: number, zoomLevel: number): IterableIterator<TileRow>;
    /**
     * Query by tile grid and zoom level
     * @param  {TileGrid} tileGrid  tile grid
     * @param  {Number} zoomLevel zoom level
     * @returns {IterableIterator<any>}
     */
    queryByTileGrid(tileGrid: TileGrid, zoomLevel: number): IterableIterator<TileRow>;
    /**
     * count by tile grid and zoom level
     * @param  {TileGrid} tileGrid  tile grid
     * @param  {Number} zoomLevel zoom level
     * @returns {Number} count of tiles
     */
    countByTileGrid(tileGrid: TileGrid, zoomLevel: number): number;
    deleteTile(column: number, row: number, zoomLevel: number): number;
    getSrs(): SpatialReferenceSystem;
    dropTable(): boolean;
    rename(newName: string): void;
}
