import { Dao } from '../dao/dao';
import { GeoPackage } from '../geoPackage';
import { UserMappingTable } from '../extension/relatedTables/userMappingTable';
import { UserRow } from './userRow';
import { RelationType } from '../extension/relatedTables/relationType';
import { UserTable } from './userTable';
import { MediaRow } from '../extension/relatedTables/mediaRow';
import { SimpleAttributesRow } from '../extension/relatedTables/simpleAttributesRow';
import { FeatureRow } from '../features/user/featureRow';
import { ExtendedRelation } from '../extension/relatedTables/extendedRelation';
import { DBValue } from '../db/dbAdapter';
import { DataTypes } from '../db/dataTypes';
/**
 * Abstract User DAO for reading user tables
 * @class UserDao
 * @extends Dao
 * @param  {module:db/geoPackageConnection~GeoPackageConnection} geoPackage        connection
 * @param  {string} table table name
 */
export declare class UserDao<T extends UserRow> extends Dao<UserRow> {
    table: UserTable;
    table_name: string;
    columns: string[];
    constructor(geoPackage: GeoPackage, table: UserTable);
    /**
     * Creates a UserRow
     * @param  {Object} [results] results to create the row from if not specified, an empty row is created
     * @return {module:user/userRow~UserRow}
     */
    createObject(results: any): UserRow;
    /**
     * Create a new user row
     */
    newRow(): UserRow;
    /**
     * Sets the value in the row
     * @param  {module:user/userRow~UserRow} object      user row
     * @param  {Number} columnIndex index
     * @param  {Object} value       value
     */
    setValueInObject(object: T, columnIndex: number, value: any): void;
    /**
     * Get a user row from the current results
     * @param  {Object} results result to create the row from
     * @return {module:user/userRow~UserRow}         the user row
     */
    getRow(results: Record<string, DBValue>): UserRow;
    /**
     * Get the table for this dao
     * @return {module:user/userTable~UserTable}
     */
    getTable(): UserTable;
    /**
     * Create a user row
     * @param  {module:db/dataTypes[]} columnTypes  column types
     * @param  {module:dao/columnValues~ColumnValues[]} values      values
     * @return {module:user/userRow~UserRow}             user row
     */
    newRowWithColumnTypes(columnTypes: {
        [key: string]: DataTypes;
    }, values: Record<string, DBValue>): UserRow;
    /**
     * Links related rows together
     * @param  {module:user/userRow~UserRow} userRow             user row
     * @param  {module:user/userRow~UserRow} relatedRow          related row
     * @param  {string} relationType        relation type
     * @param  {string|UserMappingTable} [mappingTable]        mapping table
     * @param  {module:dao/columnValues~ColumnValues} [mappingColumnValues] column values
     * @return {Promise}
     */
    linkRelatedRow(userRow: UserRow, relatedRow: UserRow, relationType: RelationType, mappingTable?: string | UserMappingTable, mappingColumnValues?: Record<string, any>): Promise<number>;
    /**
     * Links a user row to a feature row
     * @param  {module:user/userRow~UserRow} userRow             user row
     * @param  {module:features/user/featureRow~FeatureRow} featureRow          feature row
     * @param  {string|UserMappingTable} [mappingTable]        mapping table
     * @param  {module:dao/columnValues~ColumnValues} [mappingColumnValues] column values
     * @return {Promise}
     */
    linkFeatureRow(userRow: UserRow, featureRow: FeatureRow, mappingTable?: string | UserMappingTable, mappingColumnValues?: Record<string, any>): Promise<number>;
    /**
     * Links a user row to a media row
     * @param  {module:user/userRow~UserRow} userRow             user row
     * @param  {module:extension/relatedTables~MediaRow} mediaRow          media row
     * @param  {string|UserMappingTable} [mappingTable]        mapping table
     * @param  {module:dao/columnValues~ColumnValues} [mappingColumnValues] column values
     * @return {Promise}
     */
    linkMediaRow(userRow: UserRow, mediaRow: MediaRow, mappingTable?: string | UserMappingTable, mappingColumnValues?: Record<string, any>): Promise<number>;
    /**
     * Links a user row to a simpleAttributes row
     * @param  {module:user/userRow~UserRow} userRow             user row
     * @param  {module:extension/relatedTables~SimpleAttributesRow} simpleAttributesRow          simple attributes row
     * @param  {string|UserMappingTable} [mappingTable]        mapping table
     * @param  {module:dao/columnValues~ColumnValues} [mappingColumnValues] column values
     * @return {Promise}
     */
    linkSimpleAttributesRow(userRow: UserRow, simpleAttributesRow: SimpleAttributesRow, mappingTable?: string | UserMappingTable, mappingColumnValues?: Record<string, any>): Promise<number>;
    /**
     * Get all media rows that are linked to this user row
     * @param  {module:user/userRow~UserRow} userRow user row
     * @return {module:extension/relatedTables~MediaRow[]}
     */
    getLinkedMedia(userRow: UserRow): MediaRow[];
    /**
     * Get all simple attribute rows that are linked to this user row
     * @param  {module:user/userRow~UserRow} userRow user row
     * @return {module:extension/relatedTables~SimpleAttributeRow[]}
     */
    getLinkedSimpleAttributes(userRow: UserRow): SimpleAttributesRow[];
    /**
     * Get all feature rows that are linked to this user row
     * @param  {module:user/userRow~UserRow} userRow user row
     * @return {module:features/user/featureRow~FeatureRow[]}
     */
    getLinkedFeatures(userRow: UserRow): FeatureRow[];
    /**
     * Get all simple attribute relations to this table
     * @return {Object[]}
     */
    getSimpleAttributesRelations(): ExtendedRelation[];
    /**
     * Get all feature relations to this table
     * @return {Object[]}
     */
    getFeatureRelations(): ExtendedRelation[];
    /**
     * Get all media relations to this table
     * @return {Object[]}
     */
    getMediaRelations(): ExtendedRelation[];
    /**
     * Get all relations to this table with the specified name
     * @param {string} name
     * @return {Object[]}
     */
    getRelationsWithName(name: string): ExtendedRelation[];
    /**
     * Get all relations to this table
     * @return {Object[]}
     */
    getRelations(): ExtendedRelation[];
    /**
     * Gets the rows in this table by id
     * @param  {Number[]} ids ids to query for
     * @return {Object[]}
     */
    getRows(ids: number[]): T[];
    /**
     *  Get the approximate zoom level of where the bounding box of the user data fits into the world
     *
     *  @return zoom level
     */
    getZoomLevel(): number;
    /**
     * Get count of all rows in this table
     * @return {Number}
     */
    getCount(): number;
    /**
     * Reads the table specified from the geopackage
     * @param  {module:geoPackage~GeoPackage} geoPackage      geopackage object
     * @param  {string} tableName       table name
     * @return {module:user/userDao~UserDao}
     */
    static readTable(geoPackage: GeoPackage, tableName: string): UserDao<UserRow>;
}
