/**
 * UserRow module.
 * @module user/userRow
 */
import { UserTable } from './userTable';
import { DataTypes } from '../db/dataTypes';
import { DBValue } from '../db/dbAdapter';
import { UserColumn } from './userColumn';
export declare class UserRow {
    table: UserTable;
    columnTypes?: {
        [key: string]: DataTypes;
    };
    values?: Record<string, DBValue>;
    /**
     * User Row containing the values from a single result row
     * @param table User Table
     * @param columnTypes Column types of this row, based upon the data values
     * @param values Array of the row values
     */
    constructor(table: UserTable, columnTypes?: {
        [key: string]: DataTypes;
    }, values?: Record<string, DBValue>);
    /**
     * Get the column count
     * @return {number} column count
     */
    columnCount(): number;
    /**
     * Get the column names
     * @return {Array} column names
     */
    getColumnNames(): string[];
    /**
     * Get the column name at the index
     * @param  {Number} index index
     * @return {string}       column name
     */
    getColumnNameWithIndex(index: number): string;
    /**
     * Get the column index of the column name
     * @param  {string} columnName column name
     * @return {Number}            column index
     */
    getColumnIndexWithColumnName(columnName: string): number;
    /**
     * Get the value at the index
     * @param  {Number} index index
     * @return {object}       value
     */
    getValueWithIndex(index: number): any;
    /**
     * Get the value of the column name
     * @param  {string} columnName column name
     * @return {Object}            value
     */
    getValueWithColumnName(columnName: string): any;
    /**
     * Get the value from the database as an object based on the column
     * @param index column index
     * @param value value from the database
     */
    toObjectValue(index: number, value: DBValue): any;
    /**
     * Get the value which will be persisted to the database based on the column
     * @param columnName name of the column
     */
    toDatabaseValue(columnName: string): DBValue;
    /**
     * Get the row column type at the index
     * @param  {Number} index index
     * @return {Number}       row column type
     */
    getRowColumnTypeWithIndex(index: number): number;
    /**
     * Get the row column type of the column name
     * @param  {string} columnName column name
     * @return {Number}            row column type
     */
    getRowColumnTypeWithColumnName(columnName: string): number;
    /**
     * Get the column at the index
     * @param  {Number} index index
     * @return {UserColumn}       column
     */
    getColumnWithIndex(index: number): UserColumn;
    /**
     * Get the column of the column name
     * @param  {string} columnName column name
     * @return {UserColumn}            column
     */
    getColumnWithColumnName(columnName: string): UserColumn;
    /**
     * Get the id value, which is the value of the primary key
     * @return {Number} id value
     */
    get id(): number;
    /**
     * Get the primary key column Index
     * @return {Number} pk index
     */
    getPkColumnIndex(): number;
    /**
     * Get the primary key column
     * @return {UserColumn} pk column
     */
    getPkColumn(): UserColumn;
    /**
     * Set the value at the index
     * @param {Number} index index
     * @param {object} value value
     */
    setValueWithIndex(index: number, value: any): void;
    /**
     * Set the value at the index without validation
     * @param {Number} index index
     * @param {Object} value value
     */
    setValueNoValidationWithIndex(index: number, value: any): void;
    /**
     * Set the value of the column name
     * @param {string} columnName column name
     * @param {Object} value      value
     */
    setValueWithColumnName(columnName: string, value: any): void;
    hasIdColumn(): boolean;
    hasId(): boolean;
    /**
     * Set the primary key id value
     * @param {Number} id id
     */
    setId(id: number): void;
    /**
     * Clears the id so the row can be used as part of an insert or create
     */
    resetId(): void;
    /**
     * Validate the value and its actual value types against eh column data type class
     * @param  {UserColumn} column     column
     * @param  {Object} value      value
     * @param  {Array} valueTypes value types
     */
    validateValueWithColumn(column: UserColumn, value: any, valueTypes: any[]): boolean;
}
