"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * userTableReader module.
 * @module user/userTableReader
 */
var userTable_1 = require("./userTable");
var userColumn_1 = require("./userColumn");
var dataTypes_1 = require("../db/dataTypes");
/**
 * @class
 */
var UserTableReader = /** @class */ (function () {
    /**
     * @param tableName name of the table
     * @param requiredColumns array of the required column nammes
     */
    function UserTableReader(table_name, requiredColumns) {
        this.table_name = table_name;
        this.requiredColumns = requiredColumns;
    }
    /**
     * Read the table
     * @param  {object} db db connection
     * @return {module:user/userTable~UserTable}
     */
    UserTableReader.prototype.readTable = function (db) {
        var columnList = [];
        var results = db.all("PRAGMA table_info('" + this.table_name + "')");
        for (var i = 0; i < results.length; i++) {
            var result = results[i];
            var index = result[UserTableReader.GPKG_UTR_CID];
            var name_1 = result[UserTableReader.GPKG_UTR_NAME];
            var type = result[UserTableReader.GPKG_UTR_TYPE];
            var notNull = result[UserTableReader.GPKG_UTR_NOT_NULL] === 1;
            var primarykey = result[UserTableReader.GPKG_UTR_PK] === 1;
            var max = undefined;
            if (type && type.lastIndexOf(')') === type.length - 1) {
                var maxStart = type.indexOf('(');
                if (maxStart > -1) {
                    var maxString = type.substring(maxStart + 1, type.length - 1);
                    if (maxString !== '') {
                        max = parseInt(maxString);
                        type = type.substring(0, maxStart);
                    }
                }
            }
            var defaultValue = undefined;
            if (result[UserTableReader.GPKG_UTR_DFLT_VALUE]) {
                defaultValue = result[UserTableReader.GPKG_UTR_DFLT_VALUE].replace(/\\'/g, '');
            }
            var column = this.createColumn(index, name_1, type, max, notNull, defaultValue, primarykey);
            columnList.push(column);
        }
        if (columnList.length === 0) {
            throw new Error('Table does not exist: ' + this.table_name);
        }
        return this.createTable(this.table_name, columnList, this.requiredColumns);
    };
    /**
     * Creates a user column
     * @param {Number} index        column index
     * @param {string} name         column name
     * @param {module:db/dataTypes~GPKGDataType} type         data type
     * @param {Number} max max value
     * @param {Boolean} notNull      not null
     * @param {Object} defaultValue default value or nil
     * @param {Boolean} primaryKey primary key
     * @return {module:user/custom~UserCustomColumn}
     */
    UserTableReader.prototype.createColumn = function (index, name, type, max, notNull, defaultValue, primaryKey) {
        var dataType = dataTypes_1.DataTypes.fromName(type);
        return new userColumn_1.UserColumn(index, name, dataType, max, notNull, defaultValue, primaryKey);
    };
    /**
     * Create the table
     * @param  {string} tableName table name
     * @param  {module:dao/columnValues~ColumnValues[]} columns   columns
     * @param  {string[]} [requiredColumns] required columns
     * @return {module:user/userTable~UserTable}           the user table
     *
     */
    UserTableReader.prototype.createTable = function (tableName, columns, requiredColumns) {
        return new userTable_1.UserTable(tableName, columns, requiredColumns);
    };
    UserTableReader.GPKG_UTR_CID = 'cid';
    UserTableReader.GPKG_UTR_NAME = 'name';
    UserTableReader.GPKG_UTR_TYPE = 'type';
    UserTableReader.GPKG_UTR_NOT_NULL = 'notnull';
    UserTableReader.GPKG_UTR_PK = 'pk';
    UserTableReader.GPKG_UTR_DFLT_VALUE = 'dflt_value';
    return UserTableReader;
}());
exports.UserTableReader = UserTableReader;
//# sourceMappingURL=userTableReader.js.map