import { GeoPackage } from '../geoPackage';
import { GeoPackageConnection } from '../db/geoPackageConnection';
import { ColumnValues } from './columnValues';
import { DBValue } from '../db/dbAdapter';
/**
 * Base DAO
 */
export declare abstract class Dao<T> {
    readonly geoPackage: GeoPackage;
    /**
     * Database connection to the sqlite file
     */
    readonly connection: GeoPackageConnection;
    /**
     * ID Columns for this DAO
     */
    idColumns: string[];
    /**
     * Name of the table within the GeoPackage
     */
    gpkgTableName: string;
    /**
     *
     * @param geoPackage GeoPackage object this dao belongs to
     */
    constructor(geoPackage: GeoPackage);
    /**
     * Creates a object from the result
     */
    abstract createObject(result: Record<string, DBValue>): T;
    /**
     * Checks if the table exists
     */
    isTableExists(): boolean;
    /**
     * Refreshes the object by id
     * @param object object to refresh
     */
    refresh(object: T): T | undefined;
    /**
     * Query for object by id
     * @param  id ID of the object to query for
     * @return object created from the raw database object
     */
    queryForId(id: DBValue): T | undefined;
    queryForSameId(object: T): T;
    getMultiId(object: T | any): any[];
    /**
     * Query for object by multi id
     * @param  idValues ColumnValues with the multi id
     * @return object created from the raw database object
     */
    queryForMultiId(idValues: DBValue[]): T;
    /**
     * Queries for all matches and returns them in the callback.  Be aware this pulls all results into memory
     * @param  {string} [where]     Optional where clause
     * @param  {object[]} [whereArgs] Optional where args array
     * @return {Object[]} raw object array from the database
     */
    queryForAll(where?: string, whereArgs?: DBValue[]): Record<string, DBValue>[];
    /**
     * Queries for all matches and returns them in the callback.  Be aware this pulls all results into memory
     * @param  {string} fieldName name of the field to query for
     * @param  {string} value     value of the like clause
     * @return {Object[]} raw object array from the database
     */
    queryForLike(fieldName: string, value: string): Record<string, DBValue>[];
    /**
     * Queries for all matches and returns them.  Only queries for the specified column name  Be aware this pulls all results into memory
     * @param {string}  columnName  name of the column to query for
     * @param {module:dao/columnValues~ColumnValues} [fieldValues] optional values to filter on
     * @return {Object[]} raw object array from the database
     */
    queryForColumns(columnName: string, fieldValues?: ColumnValues): Record<string, DBValue>[];
    /**
     * Queries for all items in the table with a page size and page number
     * @param  {number} pageSize size of the chunk to query for
     * @param  {number} page     chunk number to query for
     * @return {Object[]} raw object array from the database
     */
    queryForChunk(pageSize: number, page: number): Record<string, DBValue>[];
    /**
     * Iterate all items in the table one at a time.  If no parameters are passed, iterates the entire table.  Returns an Iterable object
     * @param  {string} [field]   field to filter on
     * @param  {Object} [value]   value to filter on
     * @param  {string} [groupBy] group by clause
     * @param  {string} [having]  having clause
     * @param  {string} [orderBy] order by clause
     * @return {IterableIterator<any>} iterable of database objects
     */
    queryForEach(field?: string, value?: DBValue, groupBy?: string, having?: string, orderBy?: string): IterableIterator<Record<string, DBValue>>;
    /**
     * Iterate all objects in thet able that match the ColumnValues passed in
     * @param  {module:dao/columnValues~ColumnValues} fieldValues ColumnValues to query for
     * @return {IterableIterator<any>}
     */
    queryForFieldValues(fieldValues: ColumnValues): IterableIterator<Record<string, DBValue>>;
    /**
     * Iterate all matching objects
     * @param  {string} join      join clause
     * @param  {string} where     where clause
     * @param  {Object[]} whereArgs array of where query values
     * @param  {string[]} columns   columns to query for
     * @return {IterableIterator<any>}
     */
    queryJoinWhereWithArgs(join: string, where?: string, whereArgs?: DBValue[], columns?: string[]): IterableIterator<Record<string, DBValue>>;
    /**
     * Count all matching objects
     * @param  {string} join      join clause
     * @param  {string} where     where clause
     * @param  {Object[]} whereArgs array of where query values
     * @return {number}
     */
    countJoinWhereWithArgs(join: string, where?: string, whereArgs?: DBValue[]): number;
    /**
     * Iterate all distinct matching rows in the table
     * @param  {string} where     where clause
     * @param  {Object[]} whereArgs array of where query values
     * @return {IterableIterator<any>}
     */
    queryWhereWithArgsDistinct(where: string, whereArgs?: DBValue[]): IterableIterator<Record<string, DBValue>>;
    /**
     * Iterate all matching rows
     * @param  {string} [where]     where clause
     * @param  {Object[]} [whereArgs] array of where query values
     * @param  {string} [groupBy]   group by clause
     * @param  {string} [having]    having clause
     * @param  {string} [orderBy]   order by clause
     * @param  {number} [limit]     limit clause
     * @return {IterableIterator<any>}
     */
    queryWhere(where?: string, whereArgs?: DBValue[], groupBy?: string, having?: string, orderBy?: string, limit?: number): IterableIterator<Record<string, DBValue>>;
    /**
     * Get the primary key where clause
     * @param  {Object|Object[]} idValue id
     * @return {string} primary key where clause
     */
    buildPkWhere(idValue: any[] | any): string;
    /**
     * Get the primary key where args
     * @param  {Object} idValue id
     * @return {Object[]} where args
     */
    buildPkWhereArgs(idValue: DBValue[] | DBValue): DBValue[];
    /**
     * Build where (or selection) LIKE statement for fields
     * @param  {module:dao/columnValues~ColumnValues} fields    columns and values
     * @param  {string} [operation] AND or OR
     * @return {string} where clause
     */
    buildWhereLike(fields: ColumnValues, operation?: string): string;
    /**
     * Build where or selection statement for fields
     * @param  fields    columns and values
     * @param  [operation=AND] AND or OR
     * @return where clause
     */
    buildWhere(fields: ColumnValues, operation?: string): string;
    /**
     * Builds a where args array
     * @param {any[]|ColumnValues|any} values argument values to push
     * @returns {any[]}
     */
    buildWhereArgs(values: DBValue[] | ColumnValues | DBValue): DBValue[] | null;
    /**
     * Builds a where args array
     * @param {any[]} values argument values to push
     * @returns {any[]}
     */
    _buildWhereArgsWithArray(values: DBValue[]): DBValue[];
    /**
     * Builds a where args array
     * @param {ColumnValues} values argument values to push
     * @returns {any[]}
     */
    _buildWhereArgsWithColumnValues(values: ColumnValues): DBValue[];
    /**
     * Builds a where clause from the field and value with an optional operation.  If the value is empty, 'is null' is added to the query for the field
     * @param  {string} field     field name
     * @param  {Object} [value]     optional value to filter on
     * @param  {string} [operation='='] optional operation
     * @return {string} where clause
     */
    buildWhereWithFieldAndValue(field: string, value: DBValue, operation?: string): string;
    /**
     * Query for all rows in the table that match
     * @param  {string} field   field to match
     * @param  {*} value   value to match
     * @param  {string} [groupBy] group by clause
     * @param  {string} [having]  having clause
     * @param  {string} [orderBy] order by clause
     * @return {Object[]} array of raw database objects
     */
    queryForAllEq(field: string, value: DBValue, groupBy?: string, having?: string, orderBy?: string): Record<string, DBValue>[];
    /**
     * Count rows in the table optionally filtered by the parameters specified
     * @param  {module:dao/columnValues~ColumnValues|string} [fields] Either a ColumnValues object or a string specifying a field name
     * @param  {Object} [value]  value to filter on if fields is a string
     * @return {number} count of objects
     */
    count(fields?: ColumnValues | string, value?: DBValue): number;
    /**
     * Count rows in the table optionally filtered by the parameters specified
     * @param  {string} where where string
     * @param  {any[]} whereArgs arguments to filter on
     * @return {number} count of objects
     */
    countWhere(where: string, whereArgs: DBValue[]): number;
    /**
     * Get the min of the column
     * @param  {string} column    column name
     * @param  {string} [where]     where clause
     * @param  {Object[]} [whereArgs] where args
     * @return {number}
     */
    minOfColumn(column: string, where?: string, whereArgs?: DBValue[]): number;
    /**
     * Get the max of the column
     * @param  {string} column    column name
     * @param  {string} [where]     where clause
     * @param  {Object[]} [whereArgs] where args
     * @return {number}
     */
    maxOfColumn(column: string, where?: string, whereArgs?: DBValue[]): number;
    /**
     * Delete the object passed in.  Object is deleted by id
     * @param  {Object} object object to delete
     * @return {number} number of objects deleted
     */
    delete(object: T | Record<string, DBValue>): number;
    /**
     * Delete the object specified by the id
     * @param  {Object} idValue id value
     * @return {number} number of objects deleted
     */
    deleteById(idValue: DBValue): number;
    /**
     * Delete the object specified by the ids
     * @param  {module:dao/columnValues~ColumnValues} idValues id values
     * @return {number} number of objects deleted
     */
    deleteByMultiId(idValues: any[]): number;
    /**
     * Delete objects that match the query
     * @param  {string} where     where clause
     * @param  {Object[]} whereArgs where arguments
     * @return {number} number of objects deleted
     */
    deleteWhere(where: string, whereArgs: DBValue[]): number;
    /**
     * Delete all objects in the table
     * @return {number} number of objects deleted
     */
    deleteAll(): number;
    /**
     * Insert the object into the table
     * @param  {Object} object object to be inserted
     * @return {number} id of the inserted object
     */
    create(object: T): number;
    /**
     * Update all rows that match the query
     * @param  {module:dao/columnValues~ColumnValues} values    values to insert
     * @param  {string} where     where clause
     * @param  {Object[]} whereArgs where arguments
     * @return {number} number of objects updated
     */
    updateWithValues(values: Record<string, DBValue>, where: string, whereArgs: DBValue[]): {
        changes: number;
        lastInsertRowid: number;
    };
    /**
     * Update the object specified
     * @param  {Object} object object with updated values
     * @return {number} number of objects updated
     */
    update(object: T): {
        changes: number;
        lastInsertRowid: number;
    };
    /**
     * Queries for the object by id, and if it exists, updates it, otherwise creates a new object
     * @param  {Object} object object to update or create
     * @return {number} number of objects modified
     */
    createOrUpdate(object: T): number;
    /**
     * Drops this table
     * @return {boolean} results of the drop
     */
    dropTable(): boolean;
    /**
     * Rename the table
     * @param {string} newName
     */
    rename(newName: string): void;
}
