/// <reference types="node" />
import { SqliteAdapter } from './sqliteAdapter';
import { SqljsAdapter } from './sqljsAdapter';
import { DBAdapter } from './dbAdapter';
/**
 * Represents a connection to the GeoPackage database
 */
export declare class GeoPackageConnection {
    filePath: string | Buffer | Uint8Array | undefined;
    adapter: DBAdapter;
    adapterCreator: typeof SqliteAdapter | typeof SqljsAdapter;
    /**
     * Construct a new connection to the GeoPackage SQLite file
     * @param filePath path to the sqlite file
     */
    constructor(filePath: string | Buffer | Uint8Array | undefined);
    /**
     * Creates a connection to the SQLite file and when connected, returns a promise that resolves the connection.
     * This will create a {module:db/sqliteAdapter~Adapter} if running in node and the FORCE_SQLJS environment variable is not set.
     * This will create a {module:db/sqljsAdapter~Adapter} if running in the browser or the FORCE_SQLJS environment variable is set
     * @return {Promise<GeoPackageConnection>}
     */
    init(): Promise<GeoPackageConnection>;
    /**
     * Close the database.
     */
    close(): void;
    /**
     * exports the GeoPackage as a file
     * @param  {Function} callback called with an err and the buffer containing the contents of the file
     */
    export(): Promise<any>;
    /**
     * Gets the raw connection to the database
     * @return {any}
     */
    getDBConnection(): any;
    /**
     * Connects to a GeoPackage database
     * @param  {any} db database to connect to
     */
    setDBConnection(db: any): void;
    /**
     * Registers the given function so that it can be used by SQL statements
     * @param  {string} name               name of function to register
     * @param  {Function} functionDefinition function to register
     * @return {DBAdapter} the adapter in use
     */
    registerFunction(name: string, functionDefinition: Function): DBAdapter;
    /**
     * Gets the first result from the query
     * @param  {string} sql    sql query to run
     * @param  {Array|Object} [params] array of substitution parameters
     * @return {any}
     */
    get(sql: string, params?: [] | Record<string, any>): any;
    /**
     * Checks if table exists in database
     * @param {string} tableName
     * @returns {Boolean}
     */
    isTableExists(tableName: string): boolean;
    /**
     * Run the given SQL and return the results.
     * @param  {string} sql    sql to run
     * @param  {Array|Object} [params] array of substitution parameters
     * @return {{changes: number, lastInsertRowid: number}} object: `{ "changes": number, "lastInsertROWID": number }`
     * * `changes`: number of rows the statement changed
     * * `lastInsertROWID`: ID of the last inserted row
     */
    run(sql: string, params?: Record<string, any> | []): {
        changes: number;
        lastInsertRowid: number;
    };
    /**
     * Executes the query and returns all results in an array
     * @param  {string} sql sql to run
     * @param  {Array|Object} [params] substitution parameters
     * @return {any[]}
     */
    all(sql: string, params?: [] | Record<string, any> | null): any[];
    /**
     * Executes the query and returns an Iterable object of results
     * @param  {string} sql    sql to run
     * @param  {Array|Object} [params] substitution parameters
     * @return {IterableIterator<Object>}
     */
    each(sql: string, params?: [] | Record<string, any>): IterableIterator<any>;
    /**
     * Gets the minimum value from the column
     * @param  {string} table     table to query
     * @param  {string} column    column to get the min value from
     * @param  {string} [where]     where clause
     * @param  {Array|Object} [whereArgs] substitution parameters
     * @return {number}
     */
    minOfColumn(table: string, column: string, where?: string, whereArgs?: [] | Record<string, any>): number;
    /**
     * Gets the maximum value from the column
     * @param  {string} table     table to query
     * @param  {string} column    column to get the max value from
     * @param  {string} [where]     where clause
     * @param  {Array|Object} [whereArgs] substitution parameters
     * @return {number}
     */
    maxOfColumn(table: string, column: string, where?: string, whereArgs?: [] | Record<string, any>): number;
    /**
     * Return the count of objects in the table
     * @param  {string} table table name
     * @param  {string} [where] where clause
     * @param  {Array|Object} [whereArgs] substitution parameters
     * @return {number}
     */
    count(table: string, where?: string, whereArgs?: [] | Record<string, any>): number;
    /**
     * Executes an insert statement and returns the last id inserted
     * @param  {string} sql    sql to insert
     * @param  {Array|Object} params substitution parameters
     * @return {Object} last row id inserted
     */
    insert(sql: string, params: [] | Record<string, any>): number;
    /**
     * Delete from the table
     * @param  {string} tableName table name to delete from
     * @param  {string} [where]     where clause
     * @param  {Array|Object} [whereArgs] substitution parameters
     * @return {number} number of rows deleted
     */
    delete(tableName: string, where?: string, whereArgs?: [] | Record<string, any>): number;
    /**
     * Drops the table specified
     * @param  {string} tableName table to drop
     * @return {Boolean} results of table drop
     */
    dropTable(tableName: string): boolean;
    /**
     * Gets information about the table specified.  If data is returned, the table exists
     * @param  {string} tableName table to check
     * @return {Object}
     */
    tableExists(tableName: string): boolean;
    /**
     * Checks if a table and column exist
     * @param  {string} tableName  table to check
     * @param  {string} columnName column to check
     * @return {Boolean}
     */
    columnAndTableExists(tableName: string, columnName: string): boolean;
    /**
     * Sets the APPLICATION_ID and user_version for GeoPackage
     */
    setApplicationId(): void;
    /**
     * gets the application_id from the sqlite file
     * @return {number}
     */
    getApplicationId(): string;
    /**
     * Convenience method
     * @see {module:db/geoPackageConnection~GeoPackageConnection}
     * @see {module:db/sqliteAdapter~Adapter}
     * @see {module:db/sqljsAdapter~Adapter}
     * @param  {string|Buffer|Uint8Array} filePath string path to an existing file or a path to where a new file will be created or a Buffer containing the contents of the file, if undefined, an in memory database is created
     * @return {Promise} that resolves
     */
    static connect(filePath: string | Buffer | Uint8Array): Promise<GeoPackageConnection>;
    /**
     * Convenience method
     * @param  {Object}   db       open database to connect to
     * @return {Promise}
     */
    static connectWithDatabase(db: any): Promise<GeoPackageConnection>;
}
