import { Dao } from '../../dao/dao';
import { GeometryIndex } from './geometryIndex';
import { FeatureDao } from '../../features/user/featureDao';
import { GeoPackage } from '../../geoPackage';
import { TableIndex } from './tableIndex';
import { Envelope } from '../../geom/envelope';
import { FeatureRow } from '../../features/user/featureRow';
import { DBValue } from '../../db/dbAdapter';
/**
 * Geometry Index Data Access Object
 * @class
 * @extends Dao
 */
export declare class GeometryIndexDao extends Dao<GeometryIndex> {
    static readonly TABLE_NAME: string;
    static readonly COLUMN_TABLE_NAME: string;
    static readonly COLUMN_GEOM_ID: string;
    static readonly COLUMN_MIN_X: string;
    static readonly COLUMN_MAX_X: string;
    static readonly COLUMN_MIN_Y: string;
    static readonly COLUMN_MAX_Y: string;
    static readonly COLUMN_MIN_Z: string;
    static readonly COLUMN_MAX_Z: string;
    static readonly COLUMN_MIN_M: string;
    static readonly COLUMN_MAX_M: string;
    readonly gpkgTableName: string;
    readonly idColumns: string[];
    featureDao: FeatureDao<FeatureRow>;
    constructor(geoPackage: GeoPackage, featureDao: FeatureDao<FeatureRow>);
    createObject(results?: Record<string, DBValue>): GeometryIndex;
    /**
     * Get the Table Index of the Geometry Index
     *
     * @param {module:extension/index~GeometryIndex} geometryIndex geometry index
     * @return {module:extension/index~TableIndex}
     */
    getTableIndex(geometryIndex: GeometryIndex): TableIndex;
    /**
     * Query by table name
     * @param  {string} tableName table name
     * @return {Iterable}
     */
    queryForTableName(tableName: string): IterableIterator<GeometryIndex>;
    /**
     * Count by table name
     * @param  {string}   tableName table name
     * @return {Number}
     */
    countByTableName(tableName: string): number;
    /**
     * Populate a new goemetry index from an envelope
     * @param  {module:extension/index~TableIndex} tableIndex TableIndex
     * @param  {Number} geometryId id of the geometry
     * @param  {Object} envelope   envelope to store
     * @return {module:extension/index~GeometryIndex}
     */
    populate(tableIndex: TableIndex, geometryId: number, envelope: Envelope): GeometryIndex;
    /**
     * Create the GeometryIndex table
     * @return {Promise}
     */
    createTable(): Promise<boolean>;
    /**
     * Query the index with an envelope
     * @param  {Object} envelope envelope
     * @param  {Number} envelope.minX min x
     * @param  {Number} envelope.maxX max x
     * @param  {Number} envelope.minY min y
     * @param  {Number} envelope.maxY max y
     * @param  {Number} envelope.minZ min z
     * @param  {Number} envelope.maxZ max z
     * @param  {Number} envelope.minM min m
     * @param  {Number} envelope.maxM max m
     * @param  {Boolean} envelope.hasM has m
     * @param  {Boolean} envelope.hasZ has z
     * @return {Object}
     */
    _generateGeometryEnvelopeQuery(envelope: Envelope): {
        join: string;
        where: string;
        whereArgs: DBValue[];
        tableNameArr: string[];
    };
    /**
     * @param  {Object} envelope envelope
     * @param  {Number} envelope.minX min x
     * @param  {Number} envelope.maxX max x
     * @param  {Number} envelope.minY min y
     * @param  {Number} envelope.maxY max y
     * @param  {Number} envelope.minZ min z
     * @param  {Number} envelope.maxZ max z
     * @param  {Number} envelope.minM min m
     * @param  {Number} envelope.maxM max m
     * @param  {Boolean} envelope.hasM has m
     * @param  {Boolean} envelope.hasZ has z
     */
    queryWithGeometryEnvelope(envelope: Envelope): IterableIterator<GeometryIndex>;
    /**
     * @param  {Object} envelope envelope
     * @param  {Number} envelope.minX min x
     * @param  {Number} envelope.maxX max x
     * @param  {Number} envelope.minY min y
     * @param  {Number} envelope.maxY max y
     * @param  {Number} envelope.minZ min z
     * @param  {Number} envelope.maxZ max z
     * @param  {Number} envelope.minM min m
     * @param  {Number} envelope.maxM max m
     * @param  {Boolean} envelope.hasM has m
     * @param  {Boolean} envelope.hasZ has z
     */
    countWithGeometryEnvelope(envelope: Envelope): number;
}
