import { TileGrid } from './tileGrid';
import { BoundingBox } from '../boundingBox';
import proj4 from 'proj4';
import { TileMatrix } from './matrix/tileMatrix';
/**
 * This module exports utility functions for [slippy map (XYZ)](https://wiki.openstreetmap.org/wiki/Slippy_map_tilenames)
 * tile calculations.
 *
 * @module tiles/tileBoundingBoxUtils
 */
export declare class TileBoundingBoxUtils {
    static readonly WEB_MERCATOR_HALF_WORLD_WIDTH: number;
    static webMercatorProjection: proj4.Converter;
    /**
     * Calculate the bounds in tile coordinates that covers the given bounding box
     * at the given zoom level.  The result object contains the keys `minX`, `maxX`,
     * `minY`, and `maxY`, which are tile column and row values in the XYZ tile
     * scheme.
     *
     * @param {BoundingBox} webMercatorBoundingBox bounds in EPSG:3857 coordinates (meters)
     * @param {number} zoom the integral zoom level
     * @returns {{minX: number, maxX: number, minY: number, maxY: number}} bounds in tile column and row coordinates
     */
    static webMercatorTileBox(webMercatorBoundingBox: BoundingBox, zoom: number): BoundingBox;
    static determinePositionAndScale(geoPackageTileBoundingBox: BoundingBox, tileHeight: number, tileWidth: number, totalBoundingBox: BoundingBox, totalHeight: number, totalWidth: number): {
        yPositionInFinalTileStart: number;
        xPositionInFinalTileStart: number;
        dx: number;
        dy: number;
        sx: number;
        sy: number;
        dWidth: number;
        dHeight: number;
        sWidth: number;
        sHeight: number;
    };
    /**
     * Calculate the bounds in EPSG:3857 coordinates of the tile at the given XYZ
     * coordinates coordinates and zoom level.
     *
     *  @param {number} x tile column
     *  @param {number} y tile row
     *  @param {number} zoom zoom level
     *  @param {*} [options] options object
     *  @return {BoundingBox} a bounding box in EPSG:3857 meters
     */
    static getWebMercatorBoundingBoxFromXYZ(x: number, y: number, zoom: number, options?: {
        tileSize?: number;
        buffer?: number;
    }): BoundingBox;
    /**
     *  Get the tile size in meters
     *
     *  @param tilesPerSide tiles per side
     *
     *  @return meters
     */
    static tileSizeWithTilesPerSide(tilesPerSide: number): number;
    /**
     *  Get the tiles per side, width and height, at the zoom level
     *
     *  @param zoom zoom level
     *
     *  @return tiles per side
     */
    static tilesPerSideWithZoom(zoom: number): number;
    /**
     *  Get the tile grid
     *  @param {BoundingBox} totalBoundingBox    web mercator total bounding box
     *  @param {Number} matrixWidth            matrix width
     *  @param {Number} matrixHeight           matrix height
     *  @param {BoundingBox} boundingBox            bounding box
     *
     *  @return tile grid
     */
    static getTileGridWithTotalBoundingBox(totalBoundingBox: BoundingBox, matrixWidth: number, matrixHeight: number, boundingBox: BoundingBox): TileGrid;
    /**
     *  Get the tile column of the longitude in degrees
     *
     *  @param {BoundingBox} webMercatorTotalBox web mercator total bounding box
     *  @param {Number} matrixWidth         matrix width
     *  @param {Number} longitude           longitude
     *  @param {Boolean} [max]
     *
     *  @return tile column
     */
    static getTileColumnWithTotalBoundingBox(webMercatorTotalBox: BoundingBox, matrixWidth: number, longitude: number, max?: boolean): number;
    /**
     *  Get the tile row of the latitude in degrees
     *
     *  @param {BoundingBox} webMercatorTotalBox web mercator total bounding box
     *  @param {Number} matrixHeight        matrix height
     *  @param {Number} latitude            latitude
     *  @param {Boolean} [max]
     *  @return tile row
     */
    static getRowWithTotalBoundingBox(webMercatorTotalBox: BoundingBox, matrixHeight: number, latitude: number, max?: boolean): number;
    /**
     *  Get the web mercator bounding box of the tile column and row in the tile
     *  matrix using the total bounding box
     *
     *  @param {BoundingBox} box web mercator total bounding box
     *  @param {TileMatrix} tileMatrix          tile matrix
     *  @param {Number} tileColumn          tile column
     *  @param {Number} tileRow             tile row
     *
     *  @return web mercator bounding box
     */
    static getTileBoundingBox(box: BoundingBox, tileMatrix: TileMatrix, tileColumn: number, tileRow: number): BoundingBox;
    static getTileGridBoundingBox(matrixSetBoundingBox: BoundingBox, tileMatrixWidth: number, tileMatrixHeight: number, tileGrid: TileGrid): BoundingBox;
    static getXPixel(width: number, boundingBox: BoundingBox, longitude: number): number;
    static getLongitudeFromPixel(width: number, boundingBox: BoundingBox, tileBoundingBox: BoundingBox, pixel: number): number;
    static getYPixel(height: number, boundingBox: BoundingBox, latitude: number): number;
    static getLatitudeFromPixel(height: number, boundingBox: BoundingBox, tileBoundingBox: BoundingBox, pixel: number): number;
    /**
     * Get the tile size in meters
     * @param tilesPerSide tiles per side
     * @return {Number} tile size
     */
    static tileSize(tilesPerSide: number): number;
    /**
     * Get the zoom level from the tile size in meters
     * @param tileSize tile size in meters
     * @return {Number} zoom level
     * @since 1.2.0
     */
    static zoomLevelOfTileSize(tileSize: number): number;
    /**
     * Get the tile width in degrees
     * @param tilesPerSide tiles per side
     * @return {Number} tile width degrees
     */
    static tileWidthDegrees(tilesPerSide: number): number;
    /**
     * Get the tile height in degrees
     * @param tilesPerSide tiles per side
     * @return {Number} tile height degrees
     */
    statictileHeightDegrees(tilesPerSide: number): number;
    /**
     * Get the tiles per side, width and height, at the zoom level
     * @param zoom zoom level
     * @return {Number} tiles per side
     */
    static tilesPerSide(zoom: number): number;
    /**
     * Get the tile size in meters at the zoom level
     * @param zoom zoom level
     * @return {Number} tile size in meters
     * @since 2.0.0
     */
    static tileSizeWithZoom(zoom: number): number;
    /**
     * Get the tolerance distance in meters for the zoom level and pixels length
     * @param zoom zoom level
     * @param pixels pixel length
     * @return {Number} tolerance distance in meters
     * @since 2.0.0
     */
    static toleranceDistance(zoom: number, pixels: number): number;
    /**
     * Get the tolerance distance in meters for the zoom level and pixels length
     * @param zoom zoom level
     * @param pixelWidth pixel width
     * @param pixelHeight pixel height
     * @return {Number} tolerance distance in meters
     * @since 2.0.0
     */
    static toleranceDistanceWidthAndHeight(zoom: number, pixelWidth: number, pixelHeight: number): number;
}
