"use strict";
/**
 * @module user/userColumn
 */
Object.defineProperty(exports, "__esModule", { value: true });
var dataTypes_1 = require("../db/dataTypes");
/**
 * A `UserColumn` is meta-data about a single column from a {@link module:/user/userTable~UserTable}.
 *
 * @class
 * @param {Number} index column index
 * @param {string} name column name
 * @param {module:db/dataTypes~GPKGDataType} dataType data type of the column
 * @param {?Number} max max value
 * @param {Boolean} notNull not null
 * @param {?Object} defaultValue default value or null
 * @param {Boolean} primaryKey `true` if this column is part of the table's primary key
 */
var UserColumn = /** @class */ (function () {
    function UserColumn(index, name, dataType, max, notNull, defaultValue, primaryKey) {
        this.index = index;
        this.name = name;
        this.dataType = dataType;
        this.max = max;
        this.notNull = notNull;
        this.defaultValue = defaultValue;
        this.primaryKey = primaryKey;
        this.validateMax();
    }
    /**
     * Gets the type name
     * @return {module:db/dataTypes~GPKGDataType}
     */
    UserColumn.prototype.getTypeName = function () {
        var type = undefined;
        if (this.dataType !== dataTypes_1.DataTypes.GEOMETRY) {
            type = dataTypes_1.DataTypes.nameFromType(this.dataType);
        }
        return type;
    };
    /**
     * Validate that if max is set, the data type is text or blob
     */
    UserColumn.prototype.validateMax = function () {
        if (this.max && this.dataType !== dataTypes_1.DataTypes.TEXT && this.dataType !== dataTypes_1.DataTypes.BLOB) {
            throw new Error('Column max is only supported for TEXT and BLOB columns. column: ' +
                this.name +
                ', max: ' +
                this.max +
                ', type: ' +
                this.dataType);
        }
        return true;
    };
    /**
     *  Create a new primary key column
     *
     *  @param {Number} index column index
     *  @param {string} name  column name
     *
     *  @return {module:user/userColumn~UserColumn} created column
     */
    UserColumn.createPrimaryKeyColumnWithIndexAndName = function (index, name) {
        return new UserColumn(index, name, dataTypes_1.DataTypes.INTEGER, undefined, true, undefined, true);
    };
    /**
     *  Create a new column
     *
     *  @param {Number} index        column index
     *  @param {string} name         column name
     *  @param {module:db/dataTypes~GPKGDataType} type         data type
     *  @param {Number} max max value
     *  @param {Boolean} notNull      not null
     *  @param {Object} defaultValue default value or nil
     *
     *  @return {module:user/userColumn~UserColumn} created column
     */
    UserColumn.createColumn = function (index, name, type, notNull, defaultValue, max) {
        if (notNull === void 0) { notNull = false; }
        return new UserColumn(index, name, type, max, notNull, defaultValue, false);
    };
    return UserColumn;
}());
exports.UserColumn = UserColumn;
//# sourceMappingURL=userColumn.js.map