import { UserColumn } from './userColumn';
import { DataTypes } from '../db/dataTypes';
/**
 * `UserTable` models optional [user data tables](https://www.geopackage.org/spec121/index.html#_options)
 * in a [GeoPackage]{@link module:geoPackage~GeoPackage}.
 *
 * @class
 * @param  {string} tableName table name
 * @param  {module:user/userColumn~UserColumn[]} columns user columns
 * @param  {string[]} [requiredColumns] required columns
 */
export declare class UserTable {
    table_name: string;
    columns: UserColumn[];
    requiredColumns?: string[];
    static readonly FEATURE_TABLE = "FEATURE";
    static readonly TILE_TABLE = "TILE";
    /**
     * Array of column names
     * @type {string[]}
     */
    columnNames: string[];
    /**
     * Mapping between column names and their index
     * @type {Object}
     */
    nameToIndex: {
        [key: string]: number;
    };
    uniqueConstraints: {
        columns: UserColumn[];
    }[];
    pkIndex: number;
    /**
     *
     * @param table_name the name of the table
     * @param columns array of columns
     * @param requiredColumns required columns
     */
    constructor(table_name: string, columns: UserColumn[], requiredColumns?: string[]);
    getTableType(): string;
    /**
     * Check for duplicate column names
     * @param  {Number} index         index
     * @param  {Number} previousIndex previous index
     * @param  {string} column        column
     * @throws Throws an error if previous index is not undefined
     */
    duplicateCheck(index: number, previousIndex?: number, column?: string): boolean;
    /**
     * Check for the expected data type
     * @param  {module:db/dataTypes~GPKGDataType} expected expected data type
     * @param  {module:user/userColumn~UserColumn} column   column
     * @throws Will throw an error if the actual column type does not match the expected column type
     */
    typeCheck(expected: DataTypes, column: UserColumn): boolean;
    /**
     * Check for missing columns
     * @param  {Number} index  index
     * @param  {string} column column
     * @throws Will throw an error if no column is found
     */
    missingCheck(index: number, column: string): boolean;
    /**
     * Get the column index of the column name
     * @param  {string} columnName column name
     * @return {Number} the column index
     * @throws Will throw an error if the column is not found in the table
     */
    getColumnIndex(columnName: string): number;
    /**
     * Check if the table has the column
     * @param  {string} columnName name of the column
     * @return {Boolean}            true if the column exists in the table
     */
    hasColumn(columnName: string): boolean;
    /**
     * Get the column name from the index
     * @param  {Number} index index
     * @return {string} the column name
     */
    getColumnNameWithIndex(index: number): string;
    /**
     * Get the column from the index
     * @param  {Number} index index
     * @return {module:user/userColumn~UserColumn} column at the index
     */
    getColumnWithIndex(index: number): UserColumn;
    /**
     * Get column with the column name
     * @param  {string} columnName column name
     * @return {module:user/userColumn~UserColumn}            column at the index
     */
    getColumnWithColumnName(columnName: string): UserColumn;
    /**
     * Get the column count
     * @return {Number} the count of the columns
     */
    columnCount(): number;
    /**
     * Get the primary key column
     * @return {module:user/userColumn~UserColumn} the primary key column
     */
    getPkColumn(): UserColumn;
    /**
     * Get the primary key id column
     * @return {module:user/userColumn~UserColumn}
     */
    getIdColumn(): UserColumn;
    /**
     * Add a unique constraint
     * @param uniqueConstraint unique constraint to add
     * @returns number of unique constraints
     */
    addUniqueConstraint(uniqueConstraint: {
        columns: UserColumn[];
    }): number;
}
