import { MediaTable } from './mediaTable';
import { UserRow } from '../../user/userRow';
import { ImageUtils } from '../../tiles/imageUtils';
import { DBValue } from '../../db/dbAdapter';
import { UserColumn } from '../../user/userColumn';
import { DataTypes } from '../../db/dataTypes';

/**
 * MediaRow module.
 * @module extension/relatedTables
 */

/**
 * User Media Row containing the values from a single result set row
 * @class
 * @extends UserRow
 * @param  {module:extension/relatedTables~MediaTable} mediaTable  media table
 * @param  {module:db/dataTypes[]} columnTypes  column types
 * @param  {module:dao/columnValues~ColumnValues[]} values      values
 */
export class MediaRow extends UserRow {
  constructor(
    public mediaTable: MediaTable,
    columnTypes?: { [key: string]: DataTypes },
    values?: Record<string, DBValue>,
  ) {
    super(mediaTable, columnTypes, values);
  }
  /**
   * Gets the id column
   * @return {module:user/userColumn~UserColumn}
   */
  getIdColumn(): UserColumn {
    return this.mediaTable.getIdColumn();
  }
  /**
   * Gets the id
   * @return {Number}
   */
  get id(): number {
    return this.getValueWithColumnName(this.getIdColumn().name);
  }
  /**
   * Get the data column
   * @return {module:user/userColumn~UserColumn}
   */
  getDataColumn(): UserColumn {
    return this.mediaTable.getDataColumn();
  }
  /**
   * Gets the data
   * @return {Buffer}
   */
  getData(): Buffer {
    return this.getValueWithColumnName(this.getDataColumn().name);
  }
  /**
   * Get the data image
   *
   * @return {Promise<Image>}
   */
  getDataImage(): Promise<any> {
    return ImageUtils.getImage(this.getData(), this.getContentType());
  }
  /**
   * Get the scaled data image
   * @param {Number} scale
   * @return {Promise<Image>}
   */
  getScaledDataImage(scale: number): Promise<any> {
    return ImageUtils.getScaledImage(this.getData(), scale);
  }
  /**
   * Sets the data for the row
   * @param  {Buffer} data data
   */
  setData(data: Buffer): void {
    this.setValueWithColumnName(this.getDataColumn().name, data);
  }
  /**
   * Get the content type column
   * @return {module:user/userColumn~UserColumn}
   */
  getContentTypeColumn(): UserColumn {
    return this.mediaTable.getContentTypeColumn();
  }
  /**
   * Gets the content type
   * @return {string}
   */
  getContentType(): string {
    return this.getValueWithColumnName(this.getContentTypeColumn().name);
  }
  /**
   * Sets the content type for the row
   * @param  {string} contentType contentType
   */
  setContentType(contentType: string): void {
    this.setValueWithColumnName(this.getContentTypeColumn().name, contentType);
  }
}
