/*
Copyright (c) 2010–2012
    Lars-Dominik Braun <lars@6xq.net>
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
export function flickerCode(newcode) {
    let code = newcode.toUpperCase().replace(/[^a-fA-F0-9]/g, '');
    const toHex = (n, minlen) => {
        let s = n.toString(16).toUpperCase();
        while (s.length < minlen) {
            s = '0' + s;
        }
        return s;
    };
    const quersumme = (n) => {
        let q = 0;
        while (n != 0) {
            q += n % 10;
            n = Math.floor(n / 10);
        }
        return q;
    };
    const getPayload = () => {
        let i = 0;
        let payload = '';
        let len = parseInt(code.slice(0, 2), 16);
        i += 2;
        while (i < code.length - 2) {
            i += 1;
            len = parseInt(code.slice(i, i + 1), 16);
            i += 1;
            payload += code.slice(i, i + len * 2);
            i += len * 2;
        }
        return payload;
    };
    const checksum = () => {
        let len = code.length / 2 - 1;
        code = toHex(len, 2) + code.substr(2);
        let luhndata = getPayload();
        let luhnsum = 0;
        for (let i = 0; i < luhndata.length; i += 2) {
            luhnsum += (1 * parseInt(luhndata[i], 16)) + quersumme(2 * parseInt(luhndata[i + 1], 16));
        }
        luhnsum = (10 - (luhnsum % 10)) % 10;
        code = code.substr(0, code.length - 2) + toHex(luhnsum, 1) + code.substr(code.length - 1);
        let xorsum = 0;
        for (let i = 0; i < code.length - 2; i++) {
            xorsum ^= parseInt(code[i], 16);
        }
        code = code.substr(0, code.length - 1) + toHex(xorsum, 1);
    };
    this.getCode = () => code;
    checksum();
}
export function flickerCanvas(width, height, bgColor, barColor) {
    let code, halfbyteid, clock, bitarray, canvas, ctx;
    this.reset = () => {
        halfbyteid = 0;
        clock = 1;
    };
    const setup = () => {
        let bits = new Object();
        bits['0'] = [0, 0, 0, 0, 0];
        bits['1'] = [0, 1, 0, 0, 0];
        bits['2'] = [0, 0, 1, 0, 0];
        bits['3'] = [0, 1, 1, 0, 0];
        bits['4'] = [0, 0, 0, 1, 0];
        bits['5'] = [0, 1, 0, 1, 0];
        bits['6'] = [0, 0, 1, 1, 0];
        bits['7'] = [0, 1, 1, 1, 0];
        bits['8'] = [0, 0, 0, 0, 1];
        bits['9'] = [0, 1, 0, 0, 1];
        bits['A'] = [0, 0, 1, 0, 1];
        bits['B'] = [0, 1, 1, 0, 1];
        bits['C'] = [0, 0, 0, 1, 1];
        bits['D'] = [0, 1, 0, 1, 1];
        bits['E'] = [0, 0, 1, 1, 1];
        bits['F'] = [0, 1, 1, 1, 1];
        code = '0FFF' + code;
        bitarray = new Array();
        for (let i = 0; i < code.length; i += 2) {
            bitarray[i] = bits[code[i + 1]];
            bitarray[i + 1] = bits[code[i]];
        }
    };
    const createCanvas = (width, height, bgColor, barColor) => {
        canvas = document.createElement('canvas');
        canvas.width = width;
        canvas.height = height;
        if (canvas.getContext) {
            ctx = canvas.getContext('2d');
        }
        ctx.fillStyle = bgColor;
        ctx.fillRect(0, 0, canvas.width, canvas.height);
    };
    this.step = () => {
        let margin = 7;
        let barwidth = canvas.width / 5;
        bitarray[halfbyteid][0] = clock;
        for (let i = 0; i < 5; i++) {
            if (bitarray[halfbyteid][i] == 1) {
                ctx.fillStyle = barColor;
            }
            else {
                ctx.fillStyle = bgColor;
            }
            ctx.fillRect(i * barwidth + margin, margin, barwidth - 2 * margin, canvas.height - 2 * margin);
        }
        clock--;
        if (clock < 0) {
            clock = 1;
            halfbyteid++;
            if (halfbyteid >= bitarray.length) {
                halfbyteid = 0;
            }
        }
        return 0;
    };
    this.getCanvas = () => canvas;
    this.setCode = newcode => {
        code = newcode.getCode();
        setup();
        this.reset();
    };
    createCanvas(width, height, bgColor, barColor);
}
//# sourceMappingURL=data:application/json;base64,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