// This service is based on the `ng2-cookies` package which sadly is not a service and does
// not use `DOCUMENT` injection and therefore doesn't work well with AoT production builds.
// Package: https://github.com/BCJTI/ng2-cookies
import { Inject, Injectable, PLATFORM_ID } from '@angular/core';
import { DOCUMENT, isPlatformBrowser } from '@angular/common';
import * as i0 from "@angular/core";
export class CookieService {
    constructor(document, 
    // Get the `PLATFORM_ID` so we can check if we're in a browser.
    platformId) {
        this.document = document;
        this.platformId = platformId;
        this.documentIsAccessible = isPlatformBrowser(this.platformId);
    }
    /**
     * Get cookie Regular Expression
     *
     * @param name Cookie name
     * @returns property RegExp
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    static getCookieRegExp(name) {
        const escapedName = name.replace(/([\[\]\{\}\(\)\|\=\;\+\?\,\.\*\^\$])/gi, '\\$1');
        return new RegExp('(?:^' + escapedName + '|;\\s*' + escapedName + ')=(.*?)(?:;|$)', 'g');
    }
    /**
     * Gets the unencoded version of an encoded component of a Uniform Resource Identifier (URI).
     *
     * @param encodedURIComponent A value representing an encoded URI component.
     *
     * @returns The unencoded version of an encoded component of a Uniform Resource Identifier (URI).
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    static safeDecodeURIComponent(encodedURIComponent) {
        try {
            return decodeURIComponent(encodedURIComponent);
        }
        catch {
            // probably it is not uri encoded. return as is
            return encodedURIComponent;
        }
    }
    /**
     * Return `true` if {@link Document} is accessible, otherwise return `false`
     *
     * @param name Cookie name
     * @returns boolean - whether cookie with specified name exists
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    check(name) {
        if (!this.documentIsAccessible) {
            return false;
        }
        name = encodeURIComponent(name);
        const regExp = CookieService.getCookieRegExp(name);
        return regExp.test(this.document.cookie);
    }
    /**
     * Get cookies by name
     *
     * @param name Cookie name
     * @returns property value
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    get(name) {
        if (this.documentIsAccessible && this.check(name)) {
            name = encodeURIComponent(name);
            const regExp = CookieService.getCookieRegExp(name);
            const result = regExp.exec(this.document.cookie);
            return result[1] ? CookieService.safeDecodeURIComponent(result[1]) : '';
        }
        else {
            return '';
        }
    }
    /**
     * Get all cookies in JSON format
     *
     * @returns all the cookies in json
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    getAll() {
        if (!this.documentIsAccessible) {
            return {};
        }
        const cookies = {};
        const document = this.document;
        if (document.cookie && document.cookie !== '') {
            document.cookie.split(';').forEach((currentCookie) => {
                const [cookieName, cookieValue] = currentCookie.split('=');
                cookies[CookieService.safeDecodeURIComponent(cookieName.replace(/^ /, ''))] = CookieService.safeDecodeURIComponent(cookieValue);
            });
        }
        return cookies;
    }
    set(name, value, expiresOrOptions, path, domain, secure, sameSite) {
        if (!this.documentIsAccessible) {
            return;
        }
        if (typeof expiresOrOptions === 'number' || expiresOrOptions instanceof Date || path || domain || secure || sameSite) {
            const optionsBody = {
                expires: expiresOrOptions,
                path,
                domain,
                secure,
                sameSite: sameSite ? sameSite : 'Lax',
            };
            this.set(name, value, optionsBody);
            return;
        }
        let cookieString = encodeURIComponent(name) + '=' + encodeURIComponent(value) + ';';
        const options = expiresOrOptions ? expiresOrOptions : {};
        if (options.expires) {
            if (typeof options.expires === 'number') {
                const dateExpires = new Date(new Date().getTime() + options.expires * 1000 * 60 * 60 * 24);
                cookieString += 'expires=' + dateExpires.toUTCString() + ';';
            }
            else {
                cookieString += 'expires=' + options.expires.toUTCString() + ';';
            }
        }
        if (options.path) {
            cookieString += 'path=' + options.path + ';';
        }
        if (options.domain) {
            cookieString += 'domain=' + options.domain + ';';
        }
        if (options.secure === false && options.sameSite === 'None') {
            options.secure = true;
            console.warn(`[ngx-cookie-service] Cookie ${name} was forced with secure flag because sameSite=None.` +
                `More details : https://github.com/stevermeister/ngx-cookie-service/issues/86#issuecomment-597720130`);
        }
        if (options.secure) {
            cookieString += 'secure;';
        }
        if (!options.sameSite) {
            options.sameSite = 'Lax';
        }
        cookieString += 'sameSite=' + options.sameSite + ';';
        this.document.cookie = cookieString;
    }
    /**
     * Delete cookie by name
     *
     * @param name   Cookie name
     * @param path   Cookie path
     * @param domain Cookie domain
     * @param secure Cookie secure flag
     * @param sameSite Cookie sameSite flag - https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie/SameSite
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    delete(name, path, domain, secure, sameSite = 'Lax') {
        if (!this.documentIsAccessible) {
            return;
        }
        const expiresDate = new Date('Thu, 01 Jan 1970 00:00:01 GMT');
        this.set(name, '', { expires: expiresDate, path, domain, secure, sameSite });
    }
    /**
     * Delete all cookies
     *
     * @param path   Cookie path
     * @param domain Cookie domain
     * @param secure Is the Cookie secure
     * @param sameSite Is the cookie same site
     *
     * @author: Stepan Suvorov
     * @since: 1.0.0
     */
    deleteAll(path, domain, secure, sameSite = 'Lax') {
        if (!this.documentIsAccessible) {
            return;
        }
        const cookies = this.getAll();
        for (const cookieName in cookies) {
            if (cookies.hasOwnProperty(cookieName)) {
                this.delete(cookieName, path, domain, secure, sameSite);
            }
        }
    }
}
CookieService.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "14.0.0-rc.1", ngImport: i0, type: CookieService, deps: [{ token: DOCUMENT }, { token: PLATFORM_ID }], target: i0.ɵɵFactoryTarget.Injectable });
CookieService.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "14.0.0-rc.1", ngImport: i0, type: CookieService, providedIn: 'root' });
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "14.0.0-rc.1", ngImport: i0, type: CookieService, decorators: [{
            type: Injectable,
            args: [{
                    providedIn: 'root',
                }]
        }], ctorParameters: function () { return [{ type: Document, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [PLATFORM_ID]
                }] }]; } });
//# sourceMappingURL=data:application/json;base64,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