import { Component, ElementRef, Input, Output, EventEmitter, ChangeDetectionStrategy, } from '@angular/core';
import { Timer } from './countdown.timer';
var CountdownComponent = /** @class */ (function () {
    function CountdownComponent(el, timer) {
        this.el = el;
        this.timer = timer;
        this.frequency = 1000;
        this._notify = {};
        this.hands = [];
        this.left = 0;
        this.paused = false;
        /** 两种情况会触发：时间终止或调用 `stop()` */
        this.stoped = false;
        this.start = new EventEmitter();
        this.finished = new EventEmitter();
        this.notify = new EventEmitter();
        this.event = new EventEmitter();
    }
    /** 开始，当 `demand: false` 时触发 */
    CountdownComponent.prototype.begin = function () {
        this.paused = false;
        this.start.emit();
        this.callEvent('start');
    };
    /** 重新开始 */
    CountdownComponent.prototype.restart = function () {
        if (!this.stoped)
            this.destroy();
        this.init();
        this.callEvent('restart');
    };
    /** 停止 */
    CountdownComponent.prototype.stop = function () {
        if (this.stoped)
            return;
        this.stoped = true;
        this.destroy();
        this.callEvent('stop');
    };
    /** 暂停（限未终止有效） */
    CountdownComponent.prototype.pause = function () {
        if (this.stoped || this.paused)
            return;
        this.paused = true;
        this.callEvent('pause');
    };
    /** 恢复 */
    CountdownComponent.prototype.resume = function () {
        if (this.stoped || !this.paused)
            return;
        this.paused = false;
        this.callEvent('resume');
    };
    CountdownComponent.prototype.callEvent = function (action) {
        this.event.emit({ action: action, left: this.left });
    };
    CountdownComponent.prototype.init = function () {
        var me = this;
        me.config = Object.assign({
            demand: false,
            leftTime: 0,
            template: '$!h!时$!m!分$!s!秒',
            effect: 'normal',
            varRegular: /\$\!([\-\w]+)\!/g,
            clock: ['d', 100, 2, 'h', 24, 2, 'm', 60, 2, 's', 60, 2, 'u', 10, 1],
        }, me.config);
        var el = me.el.nativeElement;
        me.paused = me.config.demand;
        me.stoped = false;
        // 分析markup
        var tmpl = el.innerHTML || me.config.template;
        me.config.varRegular.lastIndex = 0;
        el.innerHTML = tmpl.replace(me.config.varRegular, function (str, type) {
            // 时钟频率校正.
            if (type === 'u' || type === 's-ext')
                me.frequency = 100;
            // 生成hand的markup
            var content = '';
            if (type === 's-ext') {
                me.hands.push({ type: 's' });
                me.hands.push({ type: 'u' });
                content =
                    me.html('', 's', 'handlet') +
                        me.html('.', '', 'digital') +
                        me.html('', 'u', 'handlet');
            }
            else {
                me.hands.push({ type: type });
            }
            return me.html(content, type, 'hand');
        });
        var clock = me.config.clock;
        me.hands.forEach(function (hand) {
            var type = hand.type;
            var base = 100, i;
            hand.node = el.querySelector(".hand-" + type);
            // radix, bits 初始化
            for (i = clock.length - 3; i > -1; i -= 3) {
                if (type === clock[i]) {
                    break;
                }
                base *= clock[i + 1];
            }
            hand.base = base;
            hand.radix = clock[i + 1];
            hand.bits = clock[i + 2];
        });
        me.getLeft();
        me.reflow(0, true);
        // bind reflow to me
        var _reflow = me.reflow;
        me.reflow = function (count) {
            if (count === void 0) { count = 0; }
            return _reflow.apply(me, [count]);
        };
        // 构建 notify
        if (me.config.notify) {
            me.config.notify.forEach(function (time) {
                if (time < 1)
                    throw new Error("the notify config must be a positive integer.");
                time = time * 1000;
                time = time - (time % me.frequency);
                me._notify[time] = true;
            });
        }
        me.timer.add(me.reflow, me.frequency);
        // show
        el.style.display = 'inline';
        this.timer.start();
        return me;
    };
    CountdownComponent.prototype.destroy = function () {
        this.timer.remove(this.reflow);
        return this;
    };
    /**
     * 更新时钟
     */
    CountdownComponent.prototype.reflow = function (count, force) {
        if (count === void 0) { count = 0; }
        if (force === void 0) { force = false; }
        var me = this;
        if (!force && (me.paused || me.stoped))
            return;
        me.left = me.left - me.frequency * count;
        me.hands.forEach(function (hand) {
            hand.lastValue = hand.value;
            hand.value = Math.floor(me.left / hand.base) % hand.radix;
        });
        me.repaint();
        if (me._notify[me.left]) {
            me.notify.emit(me.left);
            me.callEvent('notify');
        }
        if (me.left < 1) {
            me.finished.emit(0);
            me.stoped = true;
            me.callEvent('finished');
            me.destroy();
        }
    };
    /**
     * 重绘时钟
     */
    CountdownComponent.prototype.repaint = function () {
        var me = this;
        if (me.config.repaint) {
            me.config.repaint.apply(me);
            return;
        }
        var content;
        me.hands.forEach(function (hand) {
            if (hand.lastValue !== hand.value) {
                content = '';
                me.toDigitals(hand.value, hand.bits).forEach(function (digital) {
                    content += me.html(digital.toString(), '', 'digital');
                });
                hand.node.innerHTML = content;
            }
        });
    };
    /**
     * 获取倒计时剩余帧数
     */
    CountdownComponent.prototype.getLeft = function () {
        var me = this;
        var left = me.config.leftTime * 1000;
        var end = me.config.stopTime;
        if (!left && end)
            left = end - new Date().getTime();
        me.left = left - (left % me.frequency);
    };
    /**
     * 生成需要的html代码，辅助工具
     */
    CountdownComponent.prototype.html = function (con, className, type) {
        switch (type) {
            case 'hand':
            case 'handlet':
                className = type + ' hand-' + className;
                break;
            case 'digital':
                if (con === '.') {
                    className = type + ' ' + type + '-point ' + className;
                }
                else {
                    className = type + ' ' + type + '-' + con + ' ' + className;
                }
                break;
        }
        return '<span class="' + className + '">' + con + '</span>';
    };
    /**
     * 把值转换为独立的数字形式
     */
    CountdownComponent.prototype.toDigitals = function (value, bits) {
        value = value < 0 ? 0 : value;
        var digitals = [];
        // 把时、分、秒等换算成数字.
        while (bits--) {
            digitals[bits] = value % 10;
            value = Math.floor(value / 10);
        }
        return digitals;
    };
    CountdownComponent.prototype.ngOnInit = function () {
        this.init();
        if (!this.config.demand)
            this.begin();
    };
    CountdownComponent.prototype.ngOnDestroy = function () {
        this.destroy();
    };
    CountdownComponent.prototype.ngOnChanges = function (changes) {
        if (!changes.config.firstChange) {
            this.restart();
        }
    };
    CountdownComponent.decorators = [
        { type: Component, args: [{
                    selector: 'countdown',
                    template: "<ng-content></ng-content>",
                    host: { '[class.count-down]': 'true' },
                    changeDetection: ChangeDetectionStrategy.OnPush,
                    styles: ["\n      :host {\n        display: none;\n      }\n    "]
                }] }
    ];
    /** @nocollapse */
    CountdownComponent.ctorParameters = function () { return [
        { type: ElementRef },
        { type: Timer }
    ]; };
    CountdownComponent.propDecorators = {
        config: [{ type: Input }],
        start: [{ type: Output }],
        finished: [{ type: Output }],
        notify: [{ type: Output }],
        event: [{ type: Output }]
    };
    return CountdownComponent;
}());
export { CountdownComponent };

//# sourceMappingURL=data:application/json;base64,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