import { Injectable, Component, ElementRef, Input, Output, EventEmitter, ChangeDetectionStrategy, NgModule } from '@angular/core';
import { CommonModule } from '@angular/common';

class Timer {
    constructor() {
        this.fns = [];
        this.commands = [];
        this.ing = false;
    }
    start() {
        if (this.ing === true)
            return;
        this.ing = true;
        this.nextTime = +new Date();
        this.process();
    }
    process() {
        while (this.commands.length) {
            this.commands.shift()();
        }
        let diff = +new Date() - this.nextTime;
        const count = 1 + Math.floor(diff / 100);
        diff = 100 - diff % 100;
        this.nextTime += 100 * count;
        let frequency, step, i, len;
        for (i = 0, len = this.fns.length; i < len; i += 2) {
            frequency = this.fns[i + 1];
            // 100/s
            if (0 === frequency) {
                this.fns[i](count);
                // 1000/s
            }
            else {
                // 先把末位至0，再每次加2
                frequency += 2 * count - 1;
                step = Math.floor(frequency / 20);
                if (step > 0) {
                    this.fns[i](step);
                }
                // 把末位还原成1
                this.fns[i + 1] = frequency % 20 + 1;
            }
        }
        if (this.ing) {
            setTimeout(() => this.process(), diff);
        }
    }
    add(fn, frequency) {
        this.commands.push(() => {
            this.fns.push(fn);
            this.fns.push(frequency === 1000 ? 1 : 0);
            this.ing = true;
        });
    }
    remove(fn) {
        this.commands.push(() => {
            const i = this.fns.indexOf(fn);
            if (i !== -1) {
                this.fns.splice(i, 2);
            }
            this.ing = this.fns.length > 0;
        });
    }
}
Timer.decorators = [
    { type: Injectable }
];

class CountdownComponent {
    constructor(el, timer) {
        this.el = el;
        this.timer = timer;
        this.frequency = 1000;
        this._notify = {};
        this.hands = [];
        this.left = 0;
        this.paused = false;
        /** 两种情况会触发：时间终止或调用 `stop()` */
        this.stoped = false;
        this.start = new EventEmitter();
        this.finished = new EventEmitter();
        this.notify = new EventEmitter();
        this.event = new EventEmitter();
    }
    /** 开始，当 `demand: false` 时触发 */
    begin() {
        this.paused = false;
        this.start.emit();
        this.callEvent('start');
    }
    /** 重新开始 */
    restart() {
        if (!this.stoped)
            this.destroy();
        this.init();
        this.callEvent('restart');
    }
    /** 停止 */
    stop() {
        if (this.stoped)
            return;
        this.stoped = true;
        this.destroy();
        this.callEvent('stop');
    }
    /** 暂停（限未终止有效） */
    pause() {
        if (this.stoped || this.paused)
            return;
        this.paused = true;
        this.callEvent('pause');
    }
    /** 恢复 */
    resume() {
        if (this.stoped || !this.paused)
            return;
        this.paused = false;
        this.callEvent('resume');
    }
    callEvent(action) {
        this.event.emit({ action, left: this.left });
    }
    init() {
        const me = this;
        me.config = Object.assign({
            demand: false,
            leftTime: 0,
            template: '$!h!时$!m!分$!s!秒',
            effect: 'normal',
            varRegular: /\$\!([\-\w]+)\!/g,
            clock: ['d', 100, 2, 'h', 24, 2, 'm', 60, 2, 's', 60, 2, 'u', 10, 1],
        }, me.config);
        const el = me.el.nativeElement;
        me.paused = me.config.demand;
        me.stoped = false;
        // 分析markup
        const tmpl = el.innerHTML || me.config.template;
        me.config.varRegular.lastIndex = 0;
        el.innerHTML = tmpl.replace(me.config.varRegular, (str, type) => {
            // 时钟频率校正.
            if (type === 'u' || type === 's-ext')
                me.frequency = 100;
            // 生成hand的markup
            let content = '';
            if (type === 's-ext') {
                me.hands.push({ type: 's' });
                me.hands.push({ type: 'u' });
                content =
                    me.html('', 's', 'handlet') +
                        me.html('.', '', 'digital') +
                        me.html('', 'u', 'handlet');
            }
            else {
                me.hands.push({ type: type });
            }
            return me.html(content, type, 'hand');
        });
        const clock = me.config.clock;
        me.hands.forEach((hand) => {
            const type = hand.type;
            let base = 100, i;
            hand.node = el.querySelector(`.hand-${type}`);
            // radix, bits 初始化
            for (i = clock.length - 3; i > -1; i -= 3) {
                if (type === clock[i]) {
                    break;
                }
                base *= clock[i + 1];
            }
            hand.base = base;
            hand.radix = clock[i + 1];
            hand.bits = clock[i + 2];
        });
        me.getLeft();
        me.reflow(0, true);
        // bind reflow to me
        const _reflow = me.reflow;
        me.reflow = (count = 0) => {
            return _reflow.apply(me, [count]);
        };
        // 构建 notify
        if (me.config.notify) {
            me.config.notify.forEach((time) => {
                if (time < 1)
                    throw new Error(`the notify config must be a positive integer.`);
                time = time * 1000;
                time = time - (time % me.frequency);
                me._notify[time] = true;
            });
        }
        me.timer.add(me.reflow, me.frequency);
        // show
        el.style.display = 'inline';
        this.timer.start();
        return me;
    }
    destroy() {
        this.timer.remove(this.reflow);
        return this;
    }
    /**
     * 更新时钟
     */
    reflow(count = 0, force = false) {
        const me = this;
        if (!force && (me.paused || me.stoped))
            return;
        me.left = me.left - me.frequency * count;
        me.hands.forEach((hand) => {
            hand.lastValue = hand.value;
            hand.value = Math.floor(me.left / hand.base) % hand.radix;
        });
        me.repaint();
        if (me._notify[me.left]) {
            me.notify.emit(me.left);
            me.callEvent('notify');
        }
        if (me.left < 1) {
            me.finished.emit(0);
            me.stoped = true;
            me.callEvent('finished');
            me.destroy();
        }
    }
    /**
     * 重绘时钟
     */
    repaint() {
        const me = this;
        if (me.config.repaint) {
            me.config.repaint.apply(me);
            return;
        }
        let content;
        me.hands.forEach((hand) => {
            if (hand.lastValue !== hand.value) {
                content = '';
                me.toDigitals(hand.value, hand.bits).forEach((digital) => {
                    content += me.html(digital.toString(), '', 'digital');
                });
                hand.node.innerHTML = content;
            }
        });
    }
    /**
     * 获取倒计时剩余帧数
     */
    getLeft() {
        const me = this;
        let left = me.config.leftTime * 1000;
        const end = me.config.stopTime;
        if (!left && end)
            left = end - new Date().getTime();
        me.left = left - (left % me.frequency);
    }
    /**
     * 生成需要的html代码，辅助工具
     */
    html(con, className, type) {
        switch (type) {
            case 'hand':
            case 'handlet':
                className = type + ' hand-' + className;
                break;
            case 'digital':
                if (con === '.') {
                    className = type + ' ' + type + '-point ' + className;
                }
                else {
                    className = type + ' ' + type + '-' + con + ' ' + className;
                }
                break;
        }
        return '<span class="' + className + '">' + con + '</span>';
    }
    /**
     * 把值转换为独立的数字形式
     */
    toDigitals(value, bits) {
        value = value < 0 ? 0 : value;
        const digitals = [];
        // 把时、分、秒等换算成数字.
        while (bits--) {
            digitals[bits] = value % 10;
            value = Math.floor(value / 10);
        }
        return digitals;
    }
    ngOnInit() {
        this.init();
        if (!this.config.demand)
            this.begin();
    }
    ngOnDestroy() {
        this.destroy();
    }
    ngOnChanges(changes) {
        if (!changes.config.firstChange) {
            this.restart();
        }
    }
}
CountdownComponent.decorators = [
    { type: Component, args: [{
                selector: 'countdown',
                template: `<ng-content></ng-content>`,
                host: { '[class.count-down]': 'true' },
                changeDetection: ChangeDetectionStrategy.OnPush,
                styles: [`
      :host {
        display: none;
      }
    `]
            }] }
];
/** @nocollapse */
CountdownComponent.ctorParameters = () => [
    { type: ElementRef },
    { type: Timer }
];
CountdownComponent.propDecorators = {
    config: [{ type: Input }],
    start: [{ type: Output }],
    finished: [{ type: Output }],
    notify: [{ type: Output }],
    event: [{ type: Output }]
};

class CountdownModule {
}
CountdownModule.decorators = [
    { type: NgModule, args: [{
                imports: [CommonModule],
                providers: [Timer],
                declarations: [CountdownComponent],
                exports: [CountdownComponent]
            },] }
];

/**
 * Generated bundle index. Do not edit.
 */

export { CountdownComponent, Timer, CountdownModule };

//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmd4LWNvdW50ZG93bi5qcy5tYXAiLCJzb3VyY2VzIjpbIm5nOi8vbmd4LWNvdW50ZG93bi9zcmMvY291bnRkb3duLnRpbWVyLnRzIiwibmc6Ly9uZ3gtY291bnRkb3duL3NyYy9jb3VudGRvd24uY29tcG9uZW50LnRzIiwibmc6Ly9uZ3gtY291bnRkb3duL3NyYy9jb3VudGRvd24ubW9kdWxlLnRzIiwibmc6Ly9uZ3gtY291bnRkb3duL25neC1jb3VudGRvd24udHMiXSwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSW5qZWN0YWJsZSB9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xyXG5cclxuQEluamVjdGFibGUoKVxyXG5leHBvcnQgY2xhc3MgVGltZXIge1xyXG4gIHByaXZhdGUgZm5zOiBhbnlbXSA9IFtdO1xyXG4gIHByaXZhdGUgY29tbWFuZHM6IEZ1bmN0aW9uW10gPSBbXTtcclxuICBwcml2YXRlIG5leHRUaW1lOiBudW1iZXI7XHJcbiAgcHJpdmF0ZSBpbmcgPSBmYWxzZTtcclxuXHJcbiAgc3RhcnQoKSB7XHJcbiAgICBpZiAodGhpcy5pbmcgPT09IHRydWUpIHJldHVybjtcclxuXHJcbiAgICB0aGlzLmluZyA9IHRydWU7XHJcbiAgICB0aGlzLm5leHRUaW1lID0gK25ldyBEYXRlKCk7XHJcbiAgICB0aGlzLnByb2Nlc3MoKTtcclxuICB9XHJcblxyXG4gIHByaXZhdGUgcHJvY2VzcygpIHtcclxuICAgIHdoaWxlICh0aGlzLmNvbW1hbmRzLmxlbmd0aCkge1xyXG4gICAgICB0aGlzLmNvbW1hbmRzLnNoaWZ0KCkoKTtcclxuICAgIH1cclxuICAgIGxldCBkaWZmID0gK25ldyBEYXRlKCkgLSB0aGlzLm5leHRUaW1lO1xyXG4gICAgY29uc3QgY291bnQgPSAxICsgTWF0aC5mbG9vcihkaWZmIC8gMTAwKTtcclxuXHJcbiAgICBkaWZmID0gMTAwIC0gZGlmZiAlIDEwMDtcclxuICAgIHRoaXMubmV4dFRpbWUgKz0gMTAwICogY291bnQ7XHJcblxyXG4gICAgbGV0IGZyZXF1ZW5jeTogbnVtYmVyLCBzdGVwOiBudW1iZXIsIGk6IG51bWJlciwgbGVuOiBudW1iZXI7XHJcbiAgICBmb3IgKGkgPSAwLCBsZW4gPSB0aGlzLmZucy5sZW5ndGg7IGkgPCBsZW47IGkgKz0gMikge1xyXG4gICAgICBmcmVxdWVuY3kgPSB0aGlzLmZuc1tpICsgMV07XHJcblxyXG4gICAgICAvLyAxMDAvc1xyXG4gICAgICBpZiAoMCA9PT0gZnJlcXVlbmN5KSB7XHJcbiAgICAgICAgdGhpcy5mbnNbaV0oY291bnQpO1xyXG4gICAgICAgIC8vIDEwMDAvc1xyXG4gICAgICB9IGVsc2Uge1xyXG4gICAgICAgIC8vIMOlwoXCiMOmworCisOmwpzCq8Okwr3CjcOowofCszDDr8K8wozDpcKGwo3DpsKvwo/DpsKswqHDpcKKwqAyXHJcbiAgICAgICAgZnJlcXVlbmN5ICs9IDIgKiBjb3VudCAtIDE7XHJcblxyXG4gICAgICAgIHN0ZXAgPSBNYXRoLmZsb29yKGZyZXF1ZW5jeSAvIDIwKTtcclxuICAgICAgICBpZiAoc3RlcCA+IDApIHtcclxuICAgICAgICAgIHRoaXMuZm5zW2ldKHN0ZXApO1xyXG4gICAgICAgIH1cclxuXHJcbiAgICAgICAgLy8gw6bCisKKw6bCnMKrw6TCvcKNw6jCv8KYw6XCjsKfw6bCiMKQMVxyXG4gICAgICAgIHRoaXMuZm5zW2kgKyAxXSA9IGZyZXF1ZW5jeSAlIDIwICsgMTtcclxuICAgICAgfVxyXG4gICAgfVxyXG5cclxuICAgIGlmICh0aGlzLmluZykge1xyXG4gICAgICBzZXRUaW1lb3V0KCgpID0+IHRoaXMucHJvY2VzcygpLCBkaWZmKTtcclxuICAgIH1cclxuICB9XHJcblxyXG4gIGFkZChmbjogRnVuY3Rpb24sIGZyZXF1ZW5jeTogbnVtYmVyKSB7XHJcbiAgICB0aGlzLmNvbW1hbmRzLnB1c2goKCkgPT4ge1xyXG4gICAgICB0aGlzLmZucy5wdXNoKGZuKTtcclxuICAgICAgdGhpcy5mbnMucHVzaChmcmVxdWVuY3kgPT09IDEwMDAgPyAxIDogMCk7XHJcbiAgICAgIHRoaXMuaW5nID0gdHJ1ZTtcclxuICAgIH0pO1xyXG4gIH1cclxuXHJcbiAgcmVtb3ZlKGZuOiBGdW5jdGlvbikge1xyXG4gICAgdGhpcy5jb21tYW5kcy5wdXNoKCgpID0+IHtcclxuICAgICAgY29uc3QgaSA9IHRoaXMuZm5zLmluZGV4T2YoZm4pO1xyXG4gICAgICBpZiAoaSAhPT0gLTEpIHtcclxuICAgICAgICB0aGlzLmZucy5zcGxpY2UoaSwgMik7XHJcbiAgICAgIH1cclxuICAgICAgdGhpcy5pbmcgPSB0aGlzLmZucy5sZW5ndGggPiAwO1xyXG4gICAgfSk7XHJcbiAgfVxyXG59XHJcbiIsImltcG9ydCB7XG4gIENvbXBvbmVudCxcbiAgRWxlbWVudFJlZixcbiAgSW5wdXQsXG4gIE9uQ2hhbmdlcyxcbiAgU2ltcGxlQ2hhbmdlcyxcbiAgT25EZXN0cm95LFxuICBPdXRwdXQsXG4gIEV2ZW50RW1pdHRlcixcbiAgT25Jbml0LFxuICBTaW1wbGVDaGFuZ2UsXG4gIENoYW5nZURldGVjdGlvblN0cmF0ZWd5LFxufSBmcm9tICdAYW5ndWxhci9jb3JlJztcblxuaW1wb3J0IHsgQ29uZmlnLCBIYW5kIH0gZnJvbSAnLi9pbnRlcmZhY2VzJztcbmltcG9ydCB7IFRpbWVyIH0gZnJvbSAnLi9jb3VudGRvd24udGltZXInO1xuXG5AQ29tcG9uZW50KHtcbiAgc2VsZWN0b3I6ICdjb3VudGRvd24nLFxuICB0ZW1wbGF0ZTogYDxuZy1jb250ZW50PjwvbmctY29udGVudD5gLFxuICBzdHlsZXM6IFtcbiAgICBgXG4gICAgICA6aG9zdCB7XG4gICAgICAgIGRpc3BsYXk6IG5vbmU7XG4gICAgICB9XG4gICAgYCxcbiAgXSxcbiAgaG9zdDogeyAnW2NsYXNzLmNvdW50LWRvd25dJzogJ3RydWUnIH0sXG4gIGNoYW5nZURldGVjdGlvbjogQ2hhbmdlRGV0ZWN0aW9uU3RyYXRlZ3kuT25QdXNoXG59KVxuZXhwb3J0IGNsYXNzIENvdW50ZG93bkNvbXBvbmVudCBpbXBsZW1lbnRzIE9uSW5pdCwgT25DaGFuZ2VzLCBPbkRlc3Ryb3kge1xuICBwcml2YXRlIGZyZXF1ZW5jeSA9IDEwMDA7XG4gIHByaXZhdGUgX25vdGlmeTogYW55ID0ge307XG4gIHByaXZhdGUgaGFuZHM6IEhhbmRbXSA9IFtdO1xuICBwcml2YXRlIGxlZnQgPSAwO1xuICBwcml2YXRlIHBhdXNlZCA9IGZhbHNlO1xuICAvKiogw6TCuMKkw6fCp8KNw6bCg8KFw6XChsK1w6TCvMKaw6jCp8Kmw6XCj8KRw6/CvMKaw6bCl8K2w6nCl8K0w6fCu8KIw6bCrcKiw6bCiMKWw6jCsMKDw6fClMKoIGBzdG9wKClgICovXG4gIHByaXZhdGUgc3RvcGVkID0gZmFsc2U7XG5cbiAgQElucHV0KClcbiAgY29uZmlnOiBDb25maWc7XG4gIEBPdXRwdXQoKVxuICBzdGFydCA9IG5ldyBFdmVudEVtaXR0ZXIoKTtcbiAgQE91dHB1dCgpXG4gIGZpbmlzaGVkID0gbmV3IEV2ZW50RW1pdHRlcigpO1xuICBAT3V0cHV0KClcbiAgbm90aWZ5ID0gbmV3IEV2ZW50RW1pdHRlcigpO1xuICBAT3V0cHV0KClcbiAgZXZlbnQgPSBuZXcgRXZlbnRFbWl0dGVyPHsgYWN0aW9uOiBzdHJpbmc7IGxlZnQ6IG51bWJlciB9PigpO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgZWw6IEVsZW1lbnRSZWYsIHByaXZhdGUgdGltZXI6IFRpbWVyKSB7fVxuXG4gIC8qKiDDpcK8woDDpcKnwovDr8K8wozDpcK9wpMgYGRlbWFuZDogZmFsc2VgIMOmwpfCtsOowqfCpsOlwo/CkSAqL1xuICBiZWdpbigpIHtcbiAgICB0aGlzLnBhdXNlZCA9IGZhbHNlO1xuICAgIHRoaXMuc3RhcnQuZW1pdCgpO1xuICAgIHRoaXMuY2FsbEV2ZW50KCdzdGFydCcpO1xuICB9XG5cbiAgLyoqIMOpwofCjcOmwpbCsMOlwrzCgMOlwqfCiyAqL1xuICByZXN0YXJ0KCk6IHZvaWQge1xuICAgIGlmICghdGhpcy5zdG9wZWQpIHRoaXMuZGVzdHJveSgpO1xuICAgIHRoaXMuaW5pdCgpO1xuICAgIHRoaXMuY2FsbEV2ZW50KCdyZXN0YXJ0Jyk7XG4gIH1cblxuICAvKiogw6XCgcKcw6bCrcKiICovXG4gIHN0b3AoKSB7XG4gICAgaWYgKHRoaXMuc3RvcGVkKSByZXR1cm47XG4gICAgdGhpcy5zdG9wZWQgPSB0cnVlO1xuICAgIHRoaXMuZGVzdHJveSgpO1xuICAgIHRoaXMuY2FsbEV2ZW50KCdzdG9wJyk7XG4gIH1cblxuICAvKiogw6bCmsKCw6XCgcKcw6/CvMKIw6nCmcKQw6bCnMKqw6fCu8KIw6bCrcKiw6bCnMKJw6bClcKIw6/CvMKJICovXG4gIHBhdXNlKCkge1xuICAgIGlmICh0aGlzLnN0b3BlZCB8fCB0aGlzLnBhdXNlZCkgcmV0dXJuO1xuICAgIHRoaXMucGF1c2VkID0gdHJ1ZTtcbiAgICB0aGlzLmNhbGxFdmVudCgncGF1c2UnKTtcbiAgfVxuXG4gIC8qKiDDpsKBwqLDpcKkwo0gKi9cbiAgcmVzdW1lKCkge1xuICAgIGlmICh0aGlzLnN0b3BlZCB8fCAhdGhpcy5wYXVzZWQpIHJldHVybjtcbiAgICB0aGlzLnBhdXNlZCA9IGZhbHNlO1xuICAgIHRoaXMuY2FsbEV2ZW50KCdyZXN1bWUnKTtcbiAgfVxuXG4gIHByaXZhdGUgY2FsbEV2ZW50KGFjdGlvbjogc3RyaW5nKSB7XG4gICAgdGhpcy5ldmVudC5lbWl0KHsgYWN0aW9uLCBsZWZ0OiB0aGlzLmxlZnQgfSk7XG4gIH1cblxuICBwcml2YXRlIGluaXQoKSB7XG4gICAgY29uc3QgbWUgPSB0aGlzO1xuICAgIG1lLmNvbmZpZyA9IE9iamVjdC5hc3NpZ24oXG4gICAgICA8Q29uZmlnPntcbiAgICAgICAgZGVtYW5kOiBmYWxzZSxcbiAgICAgICAgbGVmdFRpbWU6IDAsXG4gICAgICAgIHRlbXBsYXRlOiAnJCFoIcOmwpfCtiQhbSHDpcKIwoYkIXMhw6fCp8KSJyxcbiAgICAgICAgZWZmZWN0OiAnbm9ybWFsJyxcbiAgICAgICAgdmFyUmVndWxhcjogL1xcJFxcIShbXFwtXFx3XSspXFwhL2csXG4gICAgICAgIGNsb2NrOiBbJ2QnLCAxMDAsIDIsICdoJywgMjQsIDIsICdtJywgNjAsIDIsICdzJywgNjAsIDIsICd1JywgMTAsIDFdLFxuICAgICAgfSxcbiAgICAgIG1lLmNvbmZpZyxcbiAgICApO1xuICAgIGNvbnN0IGVsID0gbWUuZWwubmF0aXZlRWxlbWVudCBhcyBIVE1MRWxlbWVudDtcbiAgICBtZS5wYXVzZWQgPSBtZS5jb25maWcuZGVtYW5kO1xuICAgIG1lLnN0b3BlZCA9IGZhbHNlO1xuXG4gICAgLy8gw6XCiMKGw6bCnsKQbWFya3VwXG4gICAgY29uc3QgdG1wbCA9IGVsLmlubmVySFRNTCB8fCBtZS5jb25maWcudGVtcGxhdGU7XG4gICAgbWUuY29uZmlnLnZhclJlZ3VsYXIubGFzdEluZGV4ID0gMDtcbiAgICBlbC5pbm5lckhUTUwgPSB0bXBsLnJlcGxhY2UoXG4gICAgICBtZS5jb25maWcudmFyUmVndWxhcixcbiAgICAgIChzdHI6IHN0cmluZywgdHlwZTogc3RyaW5nKSA9PiB7XG4gICAgICAgIC8vIMOmwpfCtsOpwpLCn8OpwqLCkcOnwo7Ch8OmwqDCocOmwq3Coy5cbiAgICAgICAgaWYgKHR5cGUgPT09ICd1JyB8fCB0eXBlID09PSAncy1leHQnKSBtZS5mcmVxdWVuY3kgPSAxMDA7XG5cbiAgICAgICAgLy8gw6fClMKfw6bCiMKQaGFuZMOnwprChG1hcmt1cFxuICAgICAgICBsZXQgY29udGVudCA9ICcnO1xuICAgICAgICBpZiAodHlwZSA9PT0gJ3MtZXh0Jykge1xuICAgICAgICAgIG1lLmhhbmRzLnB1c2goeyB0eXBlOiAncycgfSk7XG4gICAgICAgICAgbWUuaGFuZHMucHVzaCh7IHR5cGU6ICd1JyB9KTtcbiAgICAgICAgICBjb250ZW50ID1cbiAgICAgICAgICAgIG1lLmh0bWwoJycsICdzJywgJ2hhbmRsZXQnKSArXG4gICAgICAgICAgICBtZS5odG1sKCcuJywgJycsICdkaWdpdGFsJykgK1xuICAgICAgICAgICAgbWUuaHRtbCgnJywgJ3UnLCAnaGFuZGxldCcpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIG1lLmhhbmRzLnB1c2goeyB0eXBlOiB0eXBlIH0pO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIG1lLmh0bWwoY29udGVudCwgdHlwZSwgJ2hhbmQnKTtcbiAgICAgIH0sXG4gICAgKTtcblxuICAgIGNvbnN0IGNsb2NrID0gbWUuY29uZmlnLmNsb2NrO1xuICAgIG1lLmhhbmRzLmZvckVhY2goKGhhbmQ6IEhhbmQpID0+IHtcbiAgICAgIGNvbnN0IHR5cGUgPSBoYW5kLnR5cGU7XG4gICAgICBsZXQgYmFzZSA9IDEwMCxcbiAgICAgICAgaTogbnVtYmVyO1xuXG4gICAgICBoYW5kLm5vZGUgPSBlbC5xdWVyeVNlbGVjdG9yKGAuaGFuZC0ke3R5cGV9YCk7XG4gICAgICAvLyByYWRpeCwgYml0cyDDpcKIwp3DpcKnwovDpcKMwpZcbiAgICAgIGZvciAoaSA9IGNsb2NrLmxlbmd0aCAtIDM7IGkgPiAtMTsgaSAtPSAzKSB7XG4gICAgICAgIGlmICh0eXBlID09PSBjbG9ja1tpXSkge1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG5cbiAgICAgICAgYmFzZSAqPSBjbG9ja1tpICsgMV07XG4gICAgICB9XG4gICAgICBoYW5kLmJhc2UgPSBiYXNlO1xuICAgICAgaGFuZC5yYWRpeCA9IGNsb2NrW2kgKyAxXTtcbiAgICAgIGhhbmQuYml0cyA9IGNsb2NrW2kgKyAyXTtcbiAgICB9KTtcblxuICAgIG1lLmdldExlZnQoKTtcbiAgICBtZS5yZWZsb3coMCwgdHJ1ZSk7XG5cbiAgICAvLyBiaW5kIHJlZmxvdyB0byBtZVxuICAgIGNvbnN0IF9yZWZsb3cgPSBtZS5yZWZsb3c7XG4gICAgbWUucmVmbG93ID0gKGNvdW50OiBudW1iZXIgPSAwKSA9PiB7XG4gICAgICByZXR1cm4gX3JlZmxvdy5hcHBseShtZSwgW2NvdW50XSk7XG4gICAgfTtcblxuICAgIC8vIMOmwp7ChMOlwrvCuiBub3RpZnlcbiAgICBpZiAobWUuY29uZmlnLm5vdGlmeSkge1xuICAgICAgbWUuY29uZmlnLm5vdGlmeS5mb3JFYWNoKCh0aW1lOiBudW1iZXIpID0+IHtcbiAgICAgICAgaWYgKHRpbWUgPCAxKVxuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgdGhlIG5vdGlmeSBjb25maWcgbXVzdCBiZSBhIHBvc2l0aXZlIGludGVnZXIuYCk7XG4gICAgICAgIHRpbWUgPSB0aW1lICogMTAwMDtcbiAgICAgICAgdGltZSA9IHRpbWUgLSAodGltZSAlIG1lLmZyZXF1ZW5jeSk7XG4gICAgICAgIG1lLl9ub3RpZnlbdGltZV0gPSB0cnVlO1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgbWUudGltZXIuYWRkKG1lLnJlZmxvdywgbWUuZnJlcXVlbmN5KTtcbiAgICAvLyBzaG93XG4gICAgZWwuc3R5bGUuZGlzcGxheSA9ICdpbmxpbmUnO1xuXG4gICAgdGhpcy50aW1lci5zdGFydCgpO1xuXG4gICAgcmV0dXJuIG1lO1xuICB9XG5cbiAgcHJpdmF0ZSBkZXN0cm95KCkge1xuICAgIHRoaXMudGltZXIucmVtb3ZlKHRoaXMucmVmbG93KTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiDDpsKbwrTDpsKWwrDDpsKXwrbDqcKSwp9cbiAgICovXG4gIHByaXZhdGUgcmVmbG93KGNvdW50OiBudW1iZXIgPSAwLCBmb3JjZTogYm9vbGVhbiA9IGZhbHNlKTogdm9pZCB7XG4gICAgY29uc3QgbWUgPSB0aGlzO1xuICAgIGlmICghZm9yY2UgJiYgKG1lLnBhdXNlZCB8fCBtZS5zdG9wZWQpKSByZXR1cm47XG4gICAgbWUubGVmdCA9IG1lLmxlZnQgLSBtZS5mcmVxdWVuY3kgKiBjb3VudDtcblxuICAgIG1lLmhhbmRzLmZvckVhY2goKGhhbmQ6IEhhbmQpID0+IHtcbiAgICAgIGhhbmQubGFzdFZhbHVlID0gaGFuZC52YWx1ZTtcbiAgICAgIGhhbmQudmFsdWUgPSBNYXRoLmZsb29yKG1lLmxlZnQgLyBoYW5kLmJhc2UpICUgaGFuZC5yYWRpeDtcbiAgICB9KTtcblxuICAgIG1lLnJlcGFpbnQoKTtcblxuICAgIGlmIChtZS5fbm90aWZ5W21lLmxlZnRdKSB7XG4gICAgICBtZS5ub3RpZnkuZW1pdChtZS5sZWZ0KTtcbiAgICAgIG1lLmNhbGxFdmVudCgnbm90aWZ5Jyk7XG4gICAgfVxuXG4gICAgaWYgKG1lLmxlZnQgPCAxKSB7XG4gICAgICBtZS5maW5pc2hlZC5lbWl0KDApO1xuICAgICAgbWUuc3RvcGVkID0gdHJ1ZTtcbiAgICAgIG1lLmNhbGxFdmVudCgnZmluaXNoZWQnKTtcbiAgICAgIG1lLmRlc3Ryb3koKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogw6nCh8KNw6fCu8KYw6bCl8K2w6nCksKfXG4gICAqL1xuICBwcml2YXRlIHJlcGFpbnQoKTogdm9pZCB7XG4gICAgY29uc3QgbWUgPSB0aGlzO1xuICAgIGlmIChtZS5jb25maWcucmVwYWludCkge1xuICAgICAgbWUuY29uZmlnLnJlcGFpbnQuYXBwbHkobWUpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGxldCBjb250ZW50OiBzdHJpbmc7XG5cbiAgICBtZS5oYW5kcy5mb3JFYWNoKChoYW5kOiBIYW5kKSA9PiB7XG4gICAgICBpZiAoaGFuZC5sYXN0VmFsdWUgIT09IGhhbmQudmFsdWUpIHtcbiAgICAgICAgY29udGVudCA9ICcnO1xuXG4gICAgICAgIG1lLnRvRGlnaXRhbHMoaGFuZC52YWx1ZSwgaGFuZC5iaXRzKS5mb3JFYWNoKChkaWdpdGFsOiBudW1iZXIpID0+IHtcbiAgICAgICAgICBjb250ZW50ICs9IG1lLmh0bWwoZGlnaXRhbC50b1N0cmluZygpLCAnJywgJ2RpZ2l0YWwnKTtcbiAgICAgICAgfSk7XG5cbiAgICAgICAgaGFuZC5ub2RlLmlubmVySFRNTCA9IGNvbnRlbnQ7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogw6jCjsK3w6XCj8KWw6XCgMKSw6jCrsKhw6bCl8K2w6XCicKpw6TCvcKZw6XCuMKnw6bClcKwXG4gICAqL1xuICBwcml2YXRlIGdldExlZnQoKTogdm9pZCB7XG4gICAgY29uc3QgbWUgPSB0aGlzO1xuICAgIGxldCBsZWZ0OiBudW1iZXIgPSBtZS5jb25maWcubGVmdFRpbWUgKiAxMDAwO1xuICAgIGNvbnN0IGVuZDogbnVtYmVyID0gbWUuY29uZmlnLnN0b3BUaW1lO1xuXG4gICAgaWYgKCFsZWZ0ICYmIGVuZCkgbGVmdCA9IGVuZCAtIG5ldyBEYXRlKCkuZ2V0VGltZSgpO1xuXG4gICAgbWUubGVmdCA9IGxlZnQgLSAobGVmdCAlIG1lLmZyZXF1ZW5jeSk7XG4gIH1cblxuICAvKipcbiAgICogw6fClMKfw6bCiMKQw6nCnMKAw6jCpsKBw6fCmsKEaHRtbMOkwrvCo8OnwqDCgcOvwrzCjMOowr7ChcOlworCqcOlwrfCpcOlwoXCt1xuICAgKi9cbiAgcHJpdmF0ZSBodG1sKGNvbjogc3RyaW5nLCBjbGFzc05hbWU6IHN0cmluZywgdHlwZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBzd2l0Y2ggKHR5cGUpIHtcbiAgICAgIGNhc2UgJ2hhbmQnOlxuICAgICAgY2FzZSAnaGFuZGxldCc6XG4gICAgICAgIGNsYXNzTmFtZSA9IHR5cGUgKyAnIGhhbmQtJyArIGNsYXNzTmFtZTtcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlICdkaWdpdGFsJzpcbiAgICAgICAgaWYgKGNvbiA9PT0gJy4nKSB7XG4gICAgICAgICAgY2xhc3NOYW1lID0gdHlwZSArICcgJyArIHR5cGUgKyAnLXBvaW50ICcgKyBjbGFzc05hbWU7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY2xhc3NOYW1lID0gdHlwZSArICcgJyArIHR5cGUgKyAnLScgKyBjb24gKyAnICcgKyBjbGFzc05hbWU7XG4gICAgICAgIH1cbiAgICAgICAgYnJlYWs7XG4gICAgfVxuICAgIHJldHVybiAnPHNwYW4gY2xhc3M9XCInICsgY2xhc3NOYW1lICsgJ1wiPicgKyBjb24gKyAnPC9zcGFuPic7XG4gIH1cblxuICAvKipcbiAgICogw6bCisKKw6XCgMK8w6jCvcKsw6bCjcKiw6TCuMK6w6fCi8Ksw6fCq8KLw6fCmsKEw6bClcKww6XCrcKXw6XCvcKiw6XCvMKPXG4gICAqL1xuICBwcml2YXRlIHRvRGlnaXRhbHModmFsdWU6IG51bWJlciwgYml0czogbnVtYmVyKTogbnVtYmVyW10ge1xuICAgIHZhbHVlID0gdmFsdWUgPCAwID8gMCA6IHZhbHVlO1xuICAgIGNvbnN0IGRpZ2l0YWxzID0gW107XG4gICAgLy8gw6bCisKKw6bCl8K2w6PCgMKBw6XCiMKGw6PCgMKBw6fCp8KSw6fCrcKJw6bCjcKiw6fCrsKXw6bCiMKQw6bClcKww6XCrcKXLlxuICAgIHdoaWxlIChiaXRzLS0pIHtcbiAgICAgIGRpZ2l0YWxzW2JpdHNdID0gdmFsdWUgJSAxMDtcbiAgICAgIHZhbHVlID0gTWF0aC5mbG9vcih2YWx1ZSAvIDEwKTtcbiAgICB9XG4gICAgcmV0dXJuIGRpZ2l0YWxzO1xuICB9XG5cbiAgbmdPbkluaXQoKSB7XG4gICAgdGhpcy5pbml0KCk7XG4gICAgaWYgKCF0aGlzLmNvbmZpZy5kZW1hbmQpIHRoaXMuYmVnaW4oKTtcbiAgfVxuXG4gIG5nT25EZXN0cm95KCk6IHZvaWQge1xuICAgIHRoaXMuZGVzdHJveSgpO1xuICB9XG5cbiAgbmdPbkNoYW5nZXMoXG4gICAgY2hhbmdlczogeyBbUCBpbiBrZXlvZiB0aGlzXT86IFNpbXBsZUNoYW5nZSB9ICYgU2ltcGxlQ2hhbmdlcyxcbiAgKTogdm9pZCB7XG4gICAgaWYgKCFjaGFuZ2VzLmNvbmZpZy5maXJzdENoYW5nZSkge1xuICAgICAgdGhpcy5yZXN0YXJ0KCk7XG4gICAgfVxuICB9XG59XG4iLCJpbXBvcnQgeyBDb21tb25Nb2R1bGUgfSBmcm9tICdAYW5ndWxhci9jb21tb24nO1xuaW1wb3J0IHsgTmdNb2R1bGUgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcblxuaW1wb3J0IHsgQ291bnRkb3duQ29tcG9uZW50IH0gZnJvbSAnLi9jb3VudGRvd24uY29tcG9uZW50JztcbmltcG9ydCB7IFRpbWVyIH0gZnJvbSAnLi9jb3VudGRvd24udGltZXInO1xuXG5ATmdNb2R1bGUoe1xuICAgIGltcG9ydHM6ICAgICAgICBbQ29tbW9uTW9kdWxlXSxcbiAgICBwcm92aWRlcnM6ICAgICAgW1RpbWVyXSxcbiAgICBkZWNsYXJhdGlvbnM6ICAgW0NvdW50ZG93bkNvbXBvbmVudF0sXG4gICAgZXhwb3J0czogICAgICAgIFtDb3VudGRvd25Db21wb25lbnRdXG59KVxuZXhwb3J0IGNsYXNzIENvdW50ZG93bk1vZHVsZSB7XG59XG4iLCIvKipcbiAqIEdlbmVyYXRlZCBidW5kbGUgaW5kZXguIERvIG5vdCBlZGl0LlxuICovXG5cbmV4cG9ydCAqIGZyb20gJy4vaW5kZXgnO1xuIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7SUFFQTtRQUVVLFFBQUcsR0FBVSxFQUFFLENBQUM7UUFDaEIsYUFBUSxHQUFlLEVBQUUsQ0FBQztRQUUxQixRQUFHLEdBQUcsS0FBSyxDQUFDO0tBZ0VyQjtJQTlEQyxLQUFLO1FBQ0gsSUFBSSxJQUFJLENBQUMsR0FBRyxLQUFLLElBQUk7WUFBRSxPQUFPO1FBRTlCLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDO1FBQ2hCLElBQUksQ0FBQyxRQUFRLEdBQUcsQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDO1FBQzVCLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztLQUNoQjtJQUVPLE9BQU87UUFDYixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFO1lBQzNCLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztTQUN6QjtRQUNELElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQ3ZDLE1BQU0sS0FBSyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxHQUFHLENBQUMsQ0FBQztRQUV6QyxJQUFJLEdBQUcsR0FBRyxHQUFHLElBQUksR0FBRyxHQUFHLENBQUM7UUFDeEIsSUFBSSxDQUFDLFFBQVEsSUFBSSxHQUFHLEdBQUcsS0FBSyxDQUFDO1FBRTdCLElBQUksU0FBaUIsRUFBRSxJQUFZLEVBQUUsQ0FBUyxFQUFFLEdBQVcsQ0FBQztRQUM1RCxLQUFLLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNsRCxTQUFTLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7O1lBRzVCLElBQUksQ0FBQyxLQUFLLFNBQVMsRUFBRTtnQkFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQzs7YUFFcEI7aUJBQU07O2dCQUVMLFNBQVMsSUFBSSxDQUFDLEdBQUcsS0FBSyxHQUFHLENBQUMsQ0FBQztnQkFFM0IsSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQyxDQUFDO2dCQUNsQyxJQUFJLElBQUksR0FBRyxDQUFDLEVBQUU7b0JBQ1osSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDbkI7O2dCQUdELElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLFNBQVMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO2FBQ3RDO1NBQ0Y7UUFFRCxJQUFJLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDWixVQUFVLENBQUMsTUFBTSxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDeEM7S0FDRjtJQUVELEdBQUcsQ0FBQyxFQUFZLEVBQUUsU0FBaUI7UUFDakMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7WUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxLQUFLLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDMUMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUM7U0FDakIsQ0FBQyxDQUFDO0tBQ0o7SUFFRCxNQUFNLENBQUMsRUFBWTtRQUNqQixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQztZQUNqQixNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUMvQixJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtnQkFDWixJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7YUFDdkI7WUFDRCxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztTQUNoQyxDQUFDLENBQUM7S0FDSjs7O1lBcEVGLFVBQVU7Ozs7SUNnRFQsWUFBb0IsRUFBYyxFQUFVLEtBQVk7UUFBcEMsT0FBRSxHQUFGLEVBQUUsQ0FBWTtRQUFVLFVBQUssR0FBTCxLQUFLLENBQU87UUFuQmhELGNBQVMsR0FBRyxJQUFJLENBQUM7UUFDakIsWUFBTyxHQUFRLEVBQUUsQ0FBQztRQUNsQixVQUFLLEdBQVcsRUFBRSxDQUFDO1FBQ25CLFNBQUksR0FBRyxDQUFDLENBQUM7UUFDVCxXQUFNLEdBQUcsS0FBSyxDQUFDOztRQUVmLFdBQU0sR0FBRyxLQUFLLENBQUM7UUFLdkIsVUFBSyxHQUFHLElBQUksWUFBWSxFQUFFLENBQUM7UUFFM0IsYUFBUSxHQUFHLElBQUksWUFBWSxFQUFFLENBQUM7UUFFOUIsV0FBTSxHQUFHLElBQUksWUFBWSxFQUFFLENBQUM7UUFFNUIsVUFBSyxHQUFHLElBQUksWUFBWSxFQUFvQyxDQUFDO0tBRUQ7O0lBRzVELEtBQUs7UUFDSCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztRQUNwQixJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2xCLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7S0FDekI7O0lBR0QsT0FBTztRQUNMLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTTtZQUFFLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNqQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDWixJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQzNCOztJQUdELElBQUk7UUFDRixJQUFJLElBQUksQ0FBQyxNQUFNO1lBQUUsT0FBTztRQUN4QixJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQztRQUNuQixJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDZixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQ3hCOztJQUdELEtBQUs7UUFDSCxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLE1BQU07WUFBRSxPQUFPO1FBQ3ZDLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDO1FBQ25CLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7S0FDekI7O0lBR0QsTUFBTTtRQUNKLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNO1lBQUUsT0FBTztRQUN4QyxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztRQUNwQixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0tBQzFCO0lBRU8sU0FBUyxDQUFDLE1BQWM7UUFDOUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO0tBQzlDO0lBRU8sSUFBSTtRQUNWLE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQztRQUNoQixFQUFFLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQ2Y7WUFDTixNQUFNLEVBQUUsS0FBSztZQUNiLFFBQVEsRUFBRSxDQUFDO1lBQ1gsUUFBUSxFQUFFLGlCQUFpQjtZQUMzQixNQUFNLEVBQUUsUUFBUTtZQUNoQixVQUFVLEVBQUUsa0JBQWtCO1lBQzlCLEtBQUssRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1NBQ3JFLEVBQ0QsRUFBRSxDQUFDLE1BQU0sQ0FDVixDQUFDO1FBQ0YsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxhQUE0QixDQUFDO1FBQzlDLEVBQUUsQ0FBQyxNQUFNLEdBQUcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUM7UUFDN0IsRUFBRSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUM7O1FBR2xCLE1BQU0sSUFBSSxHQUFHLEVBQUUsQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7UUFDaEQsRUFBRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQztRQUNuQyxFQUFFLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQ3pCLEVBQUUsQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUNwQixDQUFDLEdBQVcsRUFBRSxJQUFZOztZQUV4QixJQUFJLElBQUksS0FBSyxHQUFHLElBQUksSUFBSSxLQUFLLE9BQU87Z0JBQUUsRUFBRSxDQUFDLFNBQVMsR0FBRyxHQUFHLENBQUM7O1lBR3pELElBQUksT0FBTyxHQUFHLEVBQUUsQ0FBQztZQUNqQixJQUFJLElBQUksS0FBSyxPQUFPLEVBQUU7Z0JBQ3BCLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQzdCLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQzdCLE9BQU87b0JBQ0wsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLFNBQVMsQ0FBQzt3QkFDM0IsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsRUFBRSxFQUFFLFNBQVMsQ0FBQzt3QkFDM0IsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO2FBQy9CO2lCQUFNO2dCQUNMLEVBQUUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7YUFDL0I7WUFFRCxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztTQUN2QyxDQUNGLENBQUM7UUFFRixNQUFNLEtBQUssR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQztRQUM5QixFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQVU7WUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztZQUN2QixJQUFJLElBQUksR0FBRyxHQUFHLEVBQ1osQ0FBUyxDQUFDO1lBRVosSUFBSSxDQUFDLElBQUksR0FBRyxFQUFFLENBQUMsYUFBYSxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsQ0FBQzs7WUFFOUMsS0FBSyxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ3pDLElBQUksSUFBSSxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRTtvQkFDckIsTUFBTTtpQkFDUDtnQkFFRCxJQUFJLElBQUksS0FBSyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQzthQUN0QjtZQUNELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO1lBQ2pCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUMxQixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ2IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7O1FBR25CLE1BQU0sT0FBTyxHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUM7UUFDMUIsRUFBRSxDQUFDLE1BQU0sR0FBRyxDQUFDLFFBQWdCLENBQUM7WUFDNUIsT0FBTyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7U0FDbkMsQ0FBQzs7UUFHRixJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFO1lBQ3BCLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQVk7Z0JBQ3BDLElBQUksSUFBSSxHQUFHLENBQUM7b0JBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO2dCQUNuRSxJQUFJLEdBQUcsSUFBSSxHQUFHLElBQUksQ0FBQztnQkFDbkIsSUFBSSxHQUFHLElBQUksSUFBSSxJQUFJLEdBQUcsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUNwQyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQzthQUN6QixDQUFDLENBQUM7U0FDSjtRQUVELEVBQUUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDOztRQUV0QyxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sR0FBRyxRQUFRLENBQUM7UUFFNUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVuQixPQUFPLEVBQUUsQ0FBQztLQUNYO0lBRU8sT0FBTztRQUNiLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMvQixPQUFPLElBQUksQ0FBQztLQUNiOzs7O0lBS08sTUFBTSxDQUFDLFFBQWdCLENBQUMsRUFBRSxRQUFpQixLQUFLO1FBQ3RELE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQztRQUNoQixJQUFJLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQyxNQUFNLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQztZQUFFLE9BQU87UUFDL0MsRUFBRSxDQUFDLElBQUksR0FBRyxFQUFFLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDO1FBRXpDLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBVTtZQUMxQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7WUFDNUIsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7U0FDM0QsQ0FBQyxDQUFDO1FBRUgsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRWIsSUFBSSxFQUFFLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN2QixFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDeEIsRUFBRSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUN4QjtRQUVELElBQUksRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLEVBQUU7WUFDZixFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwQixFQUFFLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQztZQUNqQixFQUFFLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3pCLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUNkO0tBQ0Y7Ozs7SUFLTyxPQUFPO1FBQ2IsTUFBTSxFQUFFLEdBQUcsSUFBSSxDQUFDO1FBQ2hCLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUU7WUFDckIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzVCLE9BQU87U0FDUjtRQUVELElBQUksT0FBZSxDQUFDO1FBRXBCLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBVTtZQUMxQixJQUFJLElBQUksQ0FBQyxTQUFTLEtBQUssSUFBSSxDQUFDLEtBQUssRUFBRTtnQkFDakMsT0FBTyxHQUFHLEVBQUUsQ0FBQztnQkFFYixFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQWU7b0JBQzNELE9BQU8sSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsRUFBRSxFQUFFLEVBQUUsU0FBUyxDQUFDLENBQUM7aUJBQ3ZELENBQUMsQ0FBQztnQkFFSCxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUM7YUFDL0I7U0FDRixDQUFDLENBQUM7S0FDSjs7OztJQUtPLE9BQU87UUFDYixNQUFNLEVBQUUsR0FBRyxJQUFJLENBQUM7UUFDaEIsSUFBSSxJQUFJLEdBQVcsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDO1FBQzdDLE1BQU0sR0FBRyxHQUFXLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDO1FBRXZDLElBQUksQ0FBQyxJQUFJLElBQUksR0FBRztZQUFFLElBQUksR0FBRyxHQUFHLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUVwRCxFQUFFLENBQUMsSUFBSSxHQUFHLElBQUksSUFBSSxJQUFJLEdBQUcsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQ3hDOzs7O0lBS08sSUFBSSxDQUFDLEdBQVcsRUFBRSxTQUFpQixFQUFFLElBQVk7UUFDdkQsUUFBUSxJQUFJO1lBQ1YsS0FBSyxNQUFNLENBQUM7WUFDWixLQUFLLFNBQVM7Z0JBQ1osU0FBUyxHQUFHLElBQUksR0FBRyxRQUFRLEdBQUcsU0FBUyxDQUFDO2dCQUN4QyxNQUFNO1lBQ1IsS0FBSyxTQUFTO2dCQUNaLElBQUksR0FBRyxLQUFLLEdBQUcsRUFBRTtvQkFDZixTQUFTLEdBQUcsSUFBSSxHQUFHLEdBQUcsR0FBRyxJQUFJLEdBQUcsU0FBUyxHQUFHLFNBQVMsQ0FBQztpQkFDdkQ7cUJBQU07b0JBQ0wsU0FBUyxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsSUFBSSxHQUFHLEdBQUcsR0FBRyxHQUFHLEdBQUcsR0FBRyxHQUFHLFNBQVMsQ0FBQztpQkFDN0Q7Z0JBQ0QsTUFBTTtTQUNUO1FBQ0QsT0FBTyxlQUFlLEdBQUcsU0FBUyxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsU0FBUyxDQUFDO0tBQzdEOzs7O0lBS08sVUFBVSxDQUFDLEtBQWEsRUFBRSxJQUFZO1FBQzVDLEtBQUssR0FBRyxLQUFLLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7UUFDOUIsTUFBTSxRQUFRLEdBQUcsRUFBRSxDQUFDOztRQUVwQixPQUFPLElBQUksRUFBRSxFQUFFO1lBQ2IsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssR0FBRyxFQUFFLENBQUM7WUFDNUIsS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ2hDO1FBQ0QsT0FBTyxRQUFRLENBQUM7S0FDakI7SUFFRCxRQUFRO1FBQ04sSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ1osSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTTtZQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztLQUN2QztJQUVELFdBQVc7UUFDVCxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7S0FDaEI7SUFFRCxXQUFXLENBQ1QsT0FBNkQ7UUFFN0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFO1lBQy9CLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUNoQjtLQUNGOzs7WUEvUkYsU0FBUyxTQUFDO2dCQUNULFFBQVEsRUFBRSxXQUFXO2dCQUNyQixRQUFRLEVBQUUsMkJBQTJCO2dCQVFyQyxJQUFJLEVBQUUsRUFBRSxvQkFBb0IsRUFBRSxNQUFNLEVBQUU7Z0JBQ3RDLGVBQWUsRUFBRSx1QkFBdUIsQ0FBQyxNQUFNO3lCQVA3Qzs7OztLQUlDO2FBSUo7Ozs7WUEzQkMsVUFBVTtZQWFILEtBQUs7OztxQkF3QlgsS0FBSztvQkFFTCxNQUFNO3VCQUVOLE1BQU07cUJBRU4sTUFBTTtvQkFFTixNQUFNOzs7Ozs7WUN6Q1IsUUFBUSxTQUFDO2dCQUNOLE9BQU8sRUFBUyxDQUFDLFlBQVksQ0FBQztnQkFDOUIsU0FBUyxFQUFPLENBQUMsS0FBSyxDQUFDO2dCQUN2QixZQUFZLEVBQUksQ0FBQyxrQkFBa0IsQ0FBQztnQkFDcEMsT0FBTyxFQUFTLENBQUMsa0JBQWtCLENBQUM7YUFDdkM7OztBQ1hEOztHQUVHOzs7OyJ9