import { Component, EventEmitter, Input, NgModule, Output, forwardRef } from '@angular/core';
import { CommonModule } from '@angular/common';
import { FormsModule, NG_VALUE_ACCESSOR } from '@angular/forms';

class DurationPickerComponent {
    constructor() {
        this.valueChange = new EventEmitter();
        this._disabled = false;
        this.regex = /^[\+\-]?P(?!$)(\d+Y)?(\d+M)?(\d+W)?(\d+D)?(T(?=\d+[HMS])(\d+H)?(\d+M)?(\d+S)?)?$/;
        this._negative = false;
        this._years = 0;
        this._months = 0;
        this._weeks = 0;
        this._days = 0;
        this._hours = 0;
        this._minutes = 0;
        this._seconds = 0;
        this.config = {
            showNegative: false,
            showButtons: true,
            showPreview: true,
            showLetters: true,
            showYears: true,
            showMonths: true,
            showWeeks: true,
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            zeroValue: 'PT0S',
        };
        this.onChange = (_) => { };
        this.onTouched = () => { };
    }
    /**
     * @param {?} options
     * @return {?}
     */
    set options(options) {
        this.attachChanges(options);
    }
    /**
     * @return {?}
     */
    get value() {
        return this._value;
    }
    /**
     * @param {?} value
     * @return {?}
     */
    set value(value) {
        this._value = value;
        this.parse();
    }
    /**
     * @return {?}
     */
    get disabled() {
        return this._disabled;
    }
    /**
     * @param {?} disabled
     * @return {?}
     */
    set disabled(disabled) {
        this._disabled = disabled;
    }
    /**
     * @return {?}
     */
    get negative() { return this._negative; }
    /**
     * @param {?} value
     * @return {?}
     */
    set negative(value) {
        this._negative = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get years() { return this._years; }
    /**
     * @param {?} value
     * @return {?}
     */
    set years(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._years = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get months() { return this._months; }
    /**
     * @param {?} value
     * @return {?}
     */
    set months(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._months = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get weeks() { return this._weeks; }
    /**
     * @param {?} value
     * @return {?}
     */
    set weeks(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._weeks = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get days() { return this._days; }
    /**
     * @param {?} value
     * @return {?}
     */
    set days(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._days = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get hours() { return this._hours; }
    /**
     * @param {?} value
     * @return {?}
     */
    set hours(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._hours = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get minutes() { return this._minutes; }
    /**
     * @param {?} value
     * @return {?}
     */
    set minutes(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._minutes = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    get seconds() { return this._seconds; }
    /**
     * @param {?} value
     * @return {?}
     */
    set seconds(value) {
        value = this.parseNumber(value) > 0 ? value : 0;
        this._seconds = value;
        this.emitNewValue();
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.parse();
        this.value = this.generate();
    }
    /**
     * @param {?} fn
     * @return {?}
     */
    registerOnChange(fn) {
        this.onChange = fn;
    }
    /**
     * @param {?} fn
     * @return {?}
     */
    registerOnTouched(fn) {
        this.onTouched = fn;
    }
    /**
     * @param {?} value
     * @return {?}
     */
    writeValue(value) {
        if (value) {
            this.value = value;
        }
    }
    /**
     * @param {?} isDisabled
     * @return {?}
     */
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
    }
    /**
     * @return {?}
     */
    parse() {
        if (!this.value) {
            return;
        }
        const /** @type {?} */ match = this.regex.exec(this.value);
        if (!match) {
            console.error(`DurationPicker: invalid initial value: ${this.value}`);
            return;
        }
        this._negative = match[0].startsWith('-');
        this._years = this.parseNumber(match[1]);
        this._months = this.parseNumber(match[2]);
        this._weeks = this.parseNumber(match[3]);
        this._days = this.parseNumber(match[4]);
        this._hours = this.parseNumber(match[6]);
        this._minutes = this.parseNumber(match[7]);
        this._seconds = this.parseNumber(match[8]);
    }
    /**
     * @param {?} value
     * @return {?}
     */
    parseNumber(value) {
        return value ? parseInt(value, 10) : 0;
    }
    /**
     * @return {?}
     */
    generate() {
        let /** @type {?} */ output = 'P';
        if (this.config.showNegative && this.negative) {
            output = '-' + output;
        }
        if (this.config.showYears && this.years) {
            output += `${this.years}Y`;
        }
        if (this.config.showMonths && this.months) {
            output += `${this.months}M`;
        }
        if (this.config.showWeeks && this.weeks) {
            output += `${this.weeks}W`;
        }
        if (this.config.showDays && this.days) {
            output += `${this.days}D`;
        }
        if ((this.config.showHours && this.hours)
            || (this.config.showMinutes && this.minutes)
            || (this.config.showSeconds && this.seconds)) {
            output += 'T';
            if (this.config.showHours && this.hours) {
                output += `${this.hours}H`;
            }
            if (this.config.showMinutes && this.minutes) {
                output += `${this.minutes}M`;
            }
            if (this.config.showSeconds && this.seconds) {
                output += `${this.seconds}S`;
            }
        }
        // if all values are empty, just output null
        if (output === 'P' || output === '-P') {
            output = this.config.zeroValue;
        }
        return output;
    }
    /**
     * @return {?}
     */
    emitNewValue() {
        this.value = this.generate();
        this.valueChange.emit(this.value);
        this.onTouched();
        this.onChange(this.value);
    }
    /**
     * @param {?} options
     * @return {?}
     */
    attachChanges(options) {
        Object.keys(options).forEach(param => {
            if (this.config.hasOwnProperty(param)) {
                (this.config)[param] = options[param];
            }
        });
    }
}
DurationPickerComponent.decorators = [
    { type: Component, args: [{
                selector: 'app-duration-picker',
                template: `
    <table>
      <tbody>

      <tr *ngIf="config.showLetters">
        <td *ngIf="config.showNegative"></td>
        <td *ngIf="config.showYears">Y</td>
        <td *ngIf="config.showMonths">M</td>
        <td *ngIf="config.showWeeks">W</td>
        <td *ngIf="config.showDays">D</td>
        <td *ngIf="config.showHours">H</td>
        <td *ngIf="config.showMinutes">M</td>
        <td *ngIf="config.showSeconds">S</td>
      </tr>

      <tr *ngIf="config.showButtons">
        <td *ngIf="config.showNegative"></td>

        <td *ngIf="config.showYears">
          <a class="btn btn-link" id="dp-years-up" (click)="years = years + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showMonths">
          <a class="btn btn-link" id="dp-months-up" (click)="months = months + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showWeeks">
          <a class="btn btn-link" id="dp-weeks-up" (click)="weeks = weeks + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showDays">
          <a class="btn btn-link" id="dp-days-up" (click)="days = days + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showHours">
          <a class="btn btn-link" id="dp-hours-up" (click)="hours = hours + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showMinutes">
          <a class="btn btn-link" id="dp-minutes-up" (click)="minutes = minutes + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>

        <td *ngIf="config.showSeconds">
          <a class="btn btn-link" id="dp-seconds-up" (click)="seconds = seconds + 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-up"></span>
          </a>
        </td>
      </tr>

      <tr>
        <td *ngIf="config.showNegative">
          <a class="btn btn-link status-container" (click)="negative = !negative" [ngClass]="{ 'disabled': disabled }">
            <span *ngIf="!negative" class="glyphicon glyphicon-plus"></span>
            <span *ngIf="negative" class="glyphicon glyphicon-minus"></span>
          </a>
        </td>

        <td *ngIf="config.showYears">
          <input (blur)="onTouched()" type="number" placeholder="Y" id="dp-years" [(ngModel)]="years" [disabled]="disabled">
        </td>

        <td *ngIf="config.showMonths">
          <input (blur)="onTouched()" type="number" placeholder="M" id="dp-months" [(ngModel)]="months" [disabled]="disabled">
        </td>

        <td *ngIf="config.showWeeks">
          <input (blur)="onTouched()" type="number" placeholder="W" id="dp-weeks" [(ngModel)]="weeks" [disabled]="disabled">
        </td>

        <td *ngIf="config.showDays">
          <input (blur)="onTouched()" type="number" placeholder="D" id="dp-days" [(ngModel)]="days" [disabled]="disabled">
        </td>

        <td *ngIf="config.showHours">
          <input (blur)="onTouched()" type="number" placeholder="H" id="dp-hours" [(ngModel)]="hours" [disabled]="disabled">
        </td>

        <td *ngIf="config.showMinutes">
          <input (blur)="onTouched()" type="number" placeholder="M" id="dp-minutes" [(ngModel)]="minutes" [disabled]="disabled">
        </td>

        <td *ngIf="config.showSeconds">
          <input (blur)="onTouched()" type="number" placeholder="S" id="dp-seconds" [(ngModel)]="seconds" [disabled]="disabled">
        </td>

        <td *ngIf="config.showPreview">
          <span id="dp-duration-value">{{ value }}</span>
        </td>
      </tr>

      <tr *ngIf="config.showButtons">
        <td *ngIf="config.showNegative"></td>

        <td *ngIf="config.showYears">
          <a class="btn btn-link" id="dp-years-down" (click)="years = years - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showMonths">
          <a class="btn btn-link" id="dp-months-down" (click)="months = months - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showWeeks">
          <a class="btn btn-link" id="dp-weeks-down" (click)="weeks = weeks - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showDays">
          <a class="btn btn-link" id="dp-days-down" (click)="days = days - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showHours">
          <a class="btn btn-link" id="dp-hours-down" (click)="hours = hours - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showMinutes">
          <a class="btn btn-link" id="dp-minutes-down" (click)="minutes = minutes - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>

        <td *ngIf="config.showSeconds">
          <a class="btn btn-link" id="dp-seconds-down" (click)="seconds = seconds - 1" [ngClass]="{ 'disabled': disabled }">
            <span class="glyphicon glyphicon-chevron-down"></span>
          </a>
        </td>
      </tr>

      </tbody>
    </table>
  `,
                styles: [`
    tr {
      text-align: center;
    }

    input {
      width: 30px;
      height: 26px;
      margin-right: 2px;
      margin-left: 2px;
      padding: 0;

      font-size: 13px;
      color: #666;
      text-align: center;

      border: 1px solid #e7e7e7;
      border-radius: 4px;
      -webkit-box-shadow: inset 0 1px 1px rgba(0, 0, 0, 0.075);
              box-shadow: inset 0 1px 1px rgba(0, 0, 0, 0.075);
    }

    input[type="checkbox"] {
      display: none;
    }

    .status-container {
      margin-right: 5px;
    }

    a.btn {
      padding: 1px;
    }

    #dp-duration-value {
      margin-left: 10px;
    }

    input[type="number"]::-webkit-outer-spin-button,
    input[type="number"]::-webkit-inner-spin-button {
      -webkit-appearance: none;
      margin: 0;
    }
    input[type="number"] {
      -moz-appearance: textfield;
    }

    input[disabled] {
      cursor: not-allowed;
      background-color: #EEE;
      color: #9E9999;
    }

    a.disabled {
      cursor: not-allowed;
      pointer-events: none;
    }
  `],
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => DurationPickerComponent),
                        multi: true
                    }
                ],
            },] },
];
/**
 * @nocollapse
 */
DurationPickerComponent.ctorParameters = () => [];
DurationPickerComponent.propDecorators = {
    'options': [{ type: Input },],
    'value': [{ type: Input },],
    'valueChange': [{ type: Output },],
    'disabled': [{ type: Input },],
};

class DurationPickerModule {
}
DurationPickerModule.decorators = [
    { type: NgModule, args: [{
                imports: [
                    CommonModule,
                    FormsModule,
                ],
                declarations: [
                    DurationPickerComponent,
                ],
                exports: [
                    DurationPickerComponent,
                ],
            },] },
];
/**
 * @nocollapse
 */
DurationPickerModule.ctorParameters = () => [];

/**
 * Generated bundle index. Do not edit.
 */

export { DurationPickerModule, DurationPickerComponent as ɵa };
//# sourceMappingURL=ngx-duration-picker.js.map
