/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
import { Subject } from 'rxjs';
import { filter } from 'rxjs/operators';
import { createComponent } from '../framework/createComponent';
const /** @type {?} */ BROWSER_PREFIX = ['webkit', 'moz', 'MS', 'o', ''];
/**
 * @param {?} eventName
 * @param {?} element
 * @param {?} cb
 * @return {?}
 */
function register(eventName, element, cb) {
    BROWSER_PREFIX.forEach(p => {
        element.addEventListener(p ? p + eventName : eventName.toLowerCase(), cb, false);
    });
}
/**
 * A base class for supporting dynamic components.
 * There are 3 main support areas:
 * 1 - Easy wrapper for dynamic styling via CSS classes and inline styles.
 * 2 - Easy wrapper for interception of transition/animation end events.
 * 3 - Easy wrapper for component creation and injection.
 *
 * Dynamic css is done via direct element manipulation (via renderer), it does not use change detection
 * or binding. This is to allow better control over animation.
 *
 * Animation support is limited, only transition/keyframes END even are notified.
 * The animation support is needed since currently the angular animation module is limited as well and
 * does not support CSS animation that are not pre-parsed and are not in the styles metadata of a component.
 *
 * Capabilities: Add/Remove styls, Add/Remove classes, listen to animation/transition end event,
 * add components
 */
export class BaseDynamicComponent {
    /**
     * @param {?} el
     * @param {?} renderer
     */
    constructor(el, renderer) {
        this.el = el;
        this.renderer = renderer;
    }
    /**
     * @return {?}
     */
    activateAnimationListener() {
        if (this.animationEnd)
            return;
        this.animationEnd = new Subject();
        this.animationEnd$ = this.animationEnd.asObservable();
        register('TransitionEnd', this.el.nativeElement, (e) => this.onEnd(e));
        register('AnimationEnd', this.el.nativeElement, (e) => this.onEnd(e));
    }
    /**
     * Set a specific inline style on the overlay host element.
     * @param {?} prop The style key
     * @param {?} value The value, undefined to remove
     * @return {?}
     */
    setStyle(prop, value) {
        this.renderer.setStyle(this.el.nativeElement, prop, value);
        return this;
    }
    /**
     * @return {?}
     */
    forceReflow() {
        this.el.nativeElement.offsetWidth;
    }
    /**
     * @param {?} css
     * @param {?=} forceReflow
     * @return {?}
     */
    addClass(css, forceReflow = false) {
        css.split(' ')
            .forEach(c => this.renderer.addClass(this.el.nativeElement, c));
        if (forceReflow)
            this.forceReflow();
    }
    /**
     * @param {?} css
     * @param {?=} forceReflow
     * @return {?}
     */
    removeClass(css, forceReflow = false) {
        css.split(' ')
            .forEach(c => this.renderer.removeClass(this.el.nativeElement, c));
        if (forceReflow) {
            this.forceReflow();
        }
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        if (this.animationEnd && !this.animationEnd.closed) {
            this.animationEnd.complete();
        }
    }
    /**
     * @return {?}
     */
    myAnimationEnd$() {
        return this.animationEnd$.pipe(filter(e => e.target === this.el.nativeElement));
    }
    /**
     * Add a component, supply a view container ref.
     * Note: The components vcRef will result in a sibling.
     * @template T
     * @param {?} instructions
     * @return {?}
     */
    _addComponent(instructions) {
        const /** @type {?} */ cmpRef = createComponent(instructions);
        cmpRef.changeDetectorRef.detectChanges();
        return cmpRef;
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onEnd(event) {
        if (!this.animationEnd.closed) {
            this.animationEnd.next(event);
        }
    }
}
function BaseDynamicComponent_tsickle_Closure_declarations() {
    /** @type {?} */
    BaseDynamicComponent.prototype.animationEnd$;
    /** @type {?} */
    BaseDynamicComponent.prototype.animationEnd;
    /** @type {?} */
    BaseDynamicComponent.prototype.el;
    /** @type {?} */
    BaseDynamicComponent.prototype.renderer;
}
//# sourceMappingURL=base-dynamic-component.js.map
