/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
import { Subject } from 'rxjs';
import { PromiseCompleter } from '../framework/utils';
import { DialogBailOutError } from '../models/errors';
/**
 * API to an open modal window.
 * @template T
 */
export class DialogRef {
    /**
     * @param {?} overlay
     * @param {?=} context
     */
    constructor(overlay, context) {
        this.overlay = overlay;
        this.context = context;
        this._resultDeferred = new PromiseCompleter();
        this._onDestroy = new Subject();
        this.onDestroy = this._onDestroy.asObservable();
    }
    /**
     * A Promise that is resolved on a close event and rejected on a dismiss event.
     * @return {?}
     */
    get result() {
        return this._resultDeferred.promise;
    }
    /**
     * Set a close/dismiss guard
     * @param {?} guard
     * @return {?}
     */
    setCloseGuard(guard) {
        this.closeGuard = guard;
    }
    /**
     *  Close the modal with a return value, i.e: result.
     * @param {?=} result
     * @return {?}
     */
    close(result = null) {
        const /** @type {?} */ _close = () => {
            this.destroy();
            this._resultDeferred.resolve(result);
        };
        this._fireHook('beforeClose')
            .then(value => value !== true && _close())
            .catch(_close);
    }
    /**
     *  Close the modal without a return value, i.e: cancelled.
     *  This call is automatically invoked when a user either:
     *  - Presses an exit keyboard key (if configured).
     *  - Clicks outside of the modal window (if configured).
     *  Usually, dismiss represent a Cancel button or a X button.
     * @return {?}
     */
    dismiss() {
        const /** @type {?} */ _dismiss = () => {
            this.destroy();
            this._resultDeferred.promise.catch(() => { });
            this._resultDeferred.reject();
        };
        this._fireHook('beforeDismiss')
            .then(value => value !== true && _dismiss())
            .catch(_dismiss);
    }
    /**
     * Gracefully close the overlay/dialog with a rejected result.
     * Does not trigger canDestroy on the overlay.
     * @return {?}
     */
    bailOut() {
        if (this.destroyed !== true) {
            this.destroyed = true;
            this._onDestroy.next(null);
            this._onDestroy.complete();
            this._resultDeferred.reject(new DialogBailOutError());
        }
    }
    /**
     * @return {?}
     */
    destroy() {
        if (this.destroyed !== true) {
            this.destroyed = true;
            if (typeof this.overlayRef.instance.canDestroy === 'function') {
                this.overlayRef.instance.canDestroy()
                    .catch(() => { })
                    .then(() => this._destroy());
            }
            else {
                this._destroy();
            }
        }
    }
    /**
     * @return {?}
     */
    _destroy() {
        this._onDestroy.next(null);
        this._onDestroy.complete();
        this.overlayRef.destroy();
    }
    /**
     * @template T
     * @param {?} name
     * @return {?}
     */
    _fireHook(name) {
        const /** @type {?} */ gurad = this.closeGuard, /** @type {?} */
        fn = gurad && typeof gurad[name] === 'function' && gurad[name];
        return Promise.resolve(fn ? fn.call(gurad) : false);
    }
}
function DialogRef_tsickle_Closure_declarations() {
    /**
     * Reference to the overlay component ref.
     * @type {?}
     */
    DialogRef.prototype.overlayRef;
    /**
     * States if the modal is inside a specific element.
     * @type {?}
     */
    DialogRef.prototype.inElement;
    /** @type {?} */
    DialogRef.prototype.destroyed;
    /**
     * Fired before dialog is destroyed.
     * No need to unsubscribe, done automatically.
     * Note: Always called.
     * When called, overlayRef might or might not be destroyed.
     * @type {?}
     */
    DialogRef.prototype.onDestroy;
    /** @type {?} */
    DialogRef.prototype._resultDeferred;
    /** @type {?} */
    DialogRef.prototype._onDestroy;
    /** @type {?} */
    DialogRef.prototype.closeGuard;
    /** @type {?} */
    DialogRef.prototype.overlay;
    /** @type {?} */
    DialogRef.prototype.context;
}
//# sourceMappingURL=dialog-ref.js.map
