'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var astring = _interopDefault(require('astring'));
var jsep = _interopDefault(require('jsep'));

/* eslint-disable sort-keys */

// since our usage is fairly narrow, we don't really need to install extra deps such ast-types or @babel/types.
// the set of builders I've prepared here should be sufficient for our needs

function literal(value) {
  return {
    type: 'Literal',
    value,
  };
}

function identifier(name) {
  return {
    type: 'Identifier',
    name,
  };
}

function logicalExpression(operator, left, right) {
  return {
    type: 'LogicalExpression',
    operator,
    left,
    right,
  };
}

function binaryExpression(operator, left, right) {
  return {
    type: 'BinaryExpression',
    operator,
    left,
    right,
  };
}

function unaryExpression(operator, argument, prefix = true) {
  return {
    type: 'UnaryExpression',
    operator,
    argument,
    prefix,
  };
}

function memberExpression(object, property, computed = false) {
  return {
    type: 'MemberExpression',
    object,
    property,
    computed,
  };
}

function assignmentExpression(operator, left, right) {
  return {
    type: 'AssignmentExpression',
    operator,
    left,
    right,
  };
}

function callExpression(callee, _arguments) {
  return {
    type: 'CallExpression',
    callee,
    arguments: _arguments,
  };
}

function returnStatement(argument) {
  return {
    type: 'ReturnStatement',
    argument,
  };
}

function sequenceExpression(expressions) {
  return {
    type: 'SequenceExpression',
    expressions,
  };
}

const SCOPE_ID = 'scope';

const SCOPE_NODE = identifier(SCOPE_ID);
const SCOPE = {
  DESTROY: memberExpression(SCOPE_NODE, identifier('destroy')),
  EVALUATE: memberExpression(SCOPE_NODE, identifier('evaluate')),
  LAST_INDEX: memberExpression(SCOPE_NODE, identifier('lastIndex')),
  PATH: memberExpression(SCOPE_NODE, identifier('path')),
  PROPERTY: memberExpression(SCOPE_NODE, identifier('property')),
};

const SANDBOX_NODE = memberExpression(
  SCOPE_NODE,
  identifier('sandbox'),
);
const SANDBOX = {
  VALUE: memberExpression(SANDBOX_NODE, identifier('value')),
};

jsep.addUnaryOp('@');
jsep.addUnaryOp('@.');

function parseFilterExpression(expr) {
  return process(jsep(expr));
}

function parseAndStringifyFilterExpression(expr) {
  return literal(astring.generate(returnStatement(process(jsep(expr)))));
}

function process(node) {
  switch (node.type) {
    case 'LogicalExpression':
    case 'BinaryExpression':
      node.left = process(node.left);
      node.right = process(node.right);
      break;
    case 'UnaryExpression':
      if (node.operator === '@') {
        return processJsonPathPlusAddition(node);
      }

      if (node.operator === '@.') {
        return memberExpression(SANDBOX.VALUE, node.argument);
      }

      break;
    case 'MemberExpression':
      node.object = process(node.object);
      node.property = process(node.property);
      break;
    case 'Identifier':
      throw new ReferenceError(`'${node.name}' is not defined`);
  }

  return node;
}

function processJsonPathPlusAddition(node) {
  if (node.argument.type === 'CallExpression') {
    switch (node.argument.callee.type) {
      case 'Literal':
        return binaryExpression(
          '===',
          SANDBOX.VALUE,
          identifier(node.argument.callee.raw),
        );
      case 'Identifier':
        switch (node.argument.callee.name) {
          case 'string':
          case 'number':
          case 'boolean':
            return binaryExpression(
              '===',
              unaryExpression('typeof', SANDBOX.VALUE),
              literal(node.argument.callee.name),
            );
          case 'array':
            return callExpression(
              memberExpression(
                identifier('Array'),
                identifier('isArray'),
              ),
              [SANDBOX.VALUE],
            );
          case 'object':
            return logicalExpression(
              '&&',
              binaryExpression('!==', SANDBOX.VALUE, identifier('null')),
              binaryExpression(
                '===',
                unaryExpression('typeof', SANDBOX.VALUE),
                literal('object'),
              ),
            );
          case 'integer':
            return callExpression(
              memberExpression(
                identifier('Number'),
                identifier('isInteger'),
              ),
              [SANDBOX.VALUE],
            );
          default:
            throw new SyntaxError(
              `Unsupported shorthand '@${node.argument.callee.name}()'`,
            );
        }
    }
  }

  return memberExpression(SANDBOX_NODE, node.argument);
}

class Feedback {
  constructor() {
    this.ticks = 0;
    this._tick = -1;
    this.descendant = false;
  }

  get tick() {
    return this._tick;
  }

  set tick(value) {
    this._tick = value;
    this.ticks++;
  }
}

function finalize(node) {
  switch (node.type) {
    case 'SequenceExpression':
      node.type = 'BinaryExpression';
      node.operator = '===';
      node.left = SCOPE.PROPERTY;
      node.right = node.expressions[0].right.arguments[0];
      delete node.expressions;

      break;
    case 'LogicalExpression':
      finalize(node.right);

      break;
  }
}

class ESTree {
  constructor() {
    this.root = null;
  }

  addDep(node) {
    if (this.root === null) {
      this.root = node;
      return;
    }

    this.root = logicalExpression('&&', this.root, node);
  }

  bail(expression) {
    throw new SyntaxError(`Unsupported syntax: ${expression.type}`);
  }

  finalize() {
    finalize(this.root);
  }
}

function baseline(ast) {
  const feedback = new Feedback();
  const tree = new ESTree();

  for (let i = 0; i < ast.length; i++) {
    const node = ast[i];
    const { expression, operation, scope } = node;
    if (expression.type === 'root') continue;

    if (scope === 'descendant') {
      feedback.tick = -1;
      feedback.descendant = true;
    }

    feedback.tick++;

    switch (operation) {
      case 'member':
        switch (expression.type) {
          case 'identifier':
            tree.addDep(generatePlainMember(expression.value, scope, feedback));
            break;
          case 'wildcard':
            if (tree.root !== null) {
              tree.addDep(assertNoOOB(SCOPE.PATH, generateTick(feedback)));
            }

            break;
          default:
            throw new Error('Unsupported syntax');
        }

        break;
      case 'subscript':
        switch (expression.type) {
          case 'string_literal':
            tree.addDep(generatePlainMember(expression.value, scope, feedback));
            break;
          case 'filter_expression':
            if (
              feedback.descendant &&
              ast[i - 1].expression.type === 'wildcard'
            ) {
              // urgh.
              feedback.tick = -1;
            }

            if (i + 1 === ast.length) {
              tree.addDep(parseFilterExpression(expression.value.slice(1)));
            } else if (feedback.descendant) {
              tree.bail(expression);
              // body = generateLogicalAndExpression(
              //   body,
              //   generateLogicalAndExpression(
              //     assertNoOOB(PATH_NODE, b.binaryExpression('+', SCOPE_LAST_INDEX_NODE, b.literal(1))),
              //     assignToLastIndex(
              //       b.callExpression(
              //         b.memberExpression(
              //           SCOPE_NODE,
              //           b.identifier('evaluateDeep'),
              //         ),
              //         [
              //           parseAndStringifyFilterExpression(
              //             expression.value.slice(1),
              //           ),
              //           generateTick(tick, descendant),
              //           b.literal(i),
              //         ],
              //       ),
              //     ),
              //   ),
              // );
            } else {
              tree.addDep(
                callExpression(SCOPE.EVALUATE, [
                  parseAndStringifyFilterExpression(expression.value.slice(1)),
                  generateTick(feedback),
                  literal(i),
                ]),
              );
            }

            break;
          case 'wildcard':
            if (tree.root !== null) {
              tree.addDep(assertNoOOB(SCOPE.PATH, generateTick(feedback)));
            }

            break;
          default:
            tree.bail(expression);
        }

        break;
      default:
        tree.bail(expression);
    }
  }

  tree.finalize();

  if (feedback.tick >= feedback.descendant ? 1 : 0) {
    feedback.tick++;
    tree.addDep(
      binaryExpression(
        '===',
        memberExpression(SCOPE.PATH, identifier('length')),
        generateTick(feedback),
      ),
    );
  }

  if (!feedback.descendant) {
    tree.addDep(
      binaryExpression(
        '===',
        unaryExpression('void', callExpression(SCOPE.DESTROY, [])),
        unaryExpression('void', literal(0)),
      ),
    );
  }

  return tree.root;
}

function assertNoOOB(arrNode, length) {
  return binaryExpression(
    '>',
    memberExpression(arrNode, identifier('length')),
    length,
  );
}

function generateTick({ tick: value, descendant }) {
  const tick = literal(value);

  if (descendant === false) {
    return tick;
  }

  if (value === 0) {
    return SCOPE.LAST_INDEX;
  }

  return binaryExpression('+', SCOPE.LAST_INDEX, tick);
}

function generateIdentifier(value, tick) {
  return logicalExpression(
    '&&',
    assertNoOOB(SCOPE.PATH, tick),
    binaryExpression(
      '===',
      memberExpression(SCOPE.PATH, tick, true),
      literal(value),
    ),
  );
}

function assignToLastIndex(expr) {
  return sequenceExpression([
    assignmentExpression('=', SCOPE.LAST_INDEX, expr),
    assertNonNegativeIndex(),
  ]);
}

function generateIndexOf(value) {
  return assignToLastIndex(
    callExpression(memberExpression(SCOPE.PATH, identifier('indexOf')), [
      literal(value),
      SCOPE.LAST_INDEX,
    ]),
  );
}

function generatePlainMember(value, scope, feedback) {
  if (scope === 'descendant') {
    return generateIndexOf(value);
  } else {
    return generateIdentifier(value, generateTick(feedback));
  }
}

function assertNonNegativeIndex() {
  return binaryExpression(
    '!==',
    SCOPE.LAST_INDEX,
    unaryExpression('-', literal(1)),
  );
}

/* parser generated by jison 0.4.18 */
/*
  Returns a Parser object of the following structure:

  Parser: {
    yy: {}
  }

  Parser.prototype: {
    yy: {},
    trace: function(),
    symbols_: {associative list: name ==> number},
    terminals_: {associative list: number ==> name},
    productions_: [...],
    performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate, $$, _$),
    table: [...],
    defaultActions: {...},
    parseError: function(str, hash),
    parse: function(input),

    lexer: {
        EOF: 1,
        parseError: function(str, hash),
        setInput: function(input),
        input: function(),
        unput: function(str),
        more: function(),
        less: function(n),
        pastInput: function(),
        upcomingInput: function(),
        showPosition: function(),
        test_match: function(regex_match_array, rule_index),
        next: function(),
        lex: function(),
        begin: function(condition),
        popState: function(),
        _currentRules: function(),
        topState: function(),
        pushState: function(condition),

        options: {
            ranges: boolean           (optional: true ==> token location info will include a .range[] member)
            flex: boolean             (optional: true ==> flex-like lexing behaviour where the rules are tested exhaustively to find the longest match)
            backtrack_lexer: boolean  (optional: true ==> lexer regexes are tested in order and for each matching regex the action code is invoked; the lexer terminates the scan when a token is returned by the action code)
        },

        performAction: function(yy, yy_, $avoiding_name_collisions, YY_START),
        rules: [...],
        conditions: {associative list: name ==> set},
    }
  }


  token location info (@$, _$, etc.): {
    first_line: n,
    last_line: n,
    first_column: n,
    last_column: n,
    range: [start_number, end_number]       (where the numbers are indexes into the input string, regular zero-based)
  }


  the parseError function receives a 'hash' object with these members for lexer and parser errors: {
    text:        (matched text)
    token:       (the produced terminal token, if any)
    line:        (yylineno)
  }
  while parser (grammar) errors will also provide these members, i.e. parser errors deliver a superset of attributes: {
    loc:         (yylloc)
    expected:    (string describing the set of expected tokens)
    recoverable: (boolean: TRUE when the parser has a error recovery rule available for this particular error)
  }
*/
var parser = (function(){
var o=function(k,v,o,l){for(o=o||{},l=k.length;l--;o[k[l]]=v);return o},$V0=[1,5],$V1=[1,6],$V2=[1,7],$V3=[1,8],$V4=[1,9],$V5=[1,18],$V6=[1,19],$V7=[1,20],$V8=[1,12,14,22],$V9=[1,29],$Va=[1,30],$Vb=[1,33],$Vc=[1,35],$Vd=[1,31],$Ve=[1,36],$Vf=[1,37],$Vg=[24,28];
var parser = {trace: function trace () { },
yy: {},
symbols_: {"error":2,"JSON_PATH":3,"DOLLAR":4,"PATH_COMPONENTS":5,"LEADING_CHILD_MEMBER_EXPRESSION":6,"PATH_COMPONENT":7,"MEMBER_COMPONENT":8,"SUBSCRIPT_COMPONENT":9,"CHILD_MEMBER_COMPONENT":10,"DESCENDANT_MEMBER_COMPONENT":11,"DOT":12,"MEMBER_EXPRESSION":13,"DOT_DOT":14,"STAR":15,"IDENTIFIER":16,"SCRIPT_EXPRESSION":17,"INTEGER":18,"END":19,"CHILD_SUBSCRIPT_COMPONENT":20,"DESCENDANT_SUBSCRIPT_COMPONENT":21,"[":22,"SUBSCRIPT":23,"]":24,"SUBSCRIPT_EXPRESSION":25,"SUBSCRIPT_EXPRESSION_LIST":26,"SUBSCRIPT_EXPRESSION_LISTABLE":27,",":28,"STRING_LITERAL":29,"ARRAY_SLICE":30,"FILTER_EXPRESSION":31,"QQ_STRING":32,"Q_STRING":33,"$accept":0,"$end":1},
terminals_: {2:"error",4:"DOLLAR",12:"DOT",14:"DOT_DOT",15:"STAR",16:"IDENTIFIER",17:"SCRIPT_EXPRESSION",18:"INTEGER",19:"END",22:"[",24:"]",28:",",30:"ARRAY_SLICE",31:"FILTER_EXPRESSION",32:"QQ_STRING",33:"Q_STRING"},
productions_: [0,[3,1],[3,2],[3,1],[3,2],[5,1],[5,2],[7,1],[7,1],[8,1],[8,1],[10,2],[6,1],[11,2],[13,1],[13,1],[13,1],[13,1],[13,1],[9,1],[9,1],[20,3],[21,4],[23,1],[23,1],[26,1],[26,3],[27,1],[27,1],[27,1],[25,1],[25,1],[25,1],[29,1],[29,1]],
performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate /* action[1] */, $$ /* vstack */, _$ /* lstack */) {
/* this == yyval */
// Courtesy of jsonpath, https://github.com/dchester/jsonpath/blob/master/include/action.js

if (!yy.ast) {
  yy.ast = _ast;
  _ast.initialize();
}

var $0 = $$.length - 1;
switch (yystate) {
case 1:
yy.ast.set({ expression: { type: "root", value: $$[$0] } }); yy.ast.unshift(); return yy.ast.yield()
case 2:
yy.ast.set({ expression: { type: "root", value: $$[$0-1] } }); yy.ast.unshift(); return yy.ast.yield()
case 3:
yy.ast.unshift(); return yy.ast.yield()
case 4:
yy.ast.set({ operation: "member", scope: "child", expression: { type: "identifier", value: $$[$0-1] }}); yy.ast.unshift(); return yy.ast.yield()
case 5: case 6: case 11: case 13: case 18: case 21: case 22: case 23:

break;
case 7:
yy.ast.set({ operation: "member" }); yy.ast.push();
break;
case 8:
yy.ast.set({ operation: "subscript" }); yy.ast.push(); 
break;
case 9: case 19:
yy.ast.set({ scope: "child" });
break;
case 10: case 20:
yy.ast.set({ scope: "descendant" });
break;
case 12:
yy.ast.set({ scope: "child", operation: "member" });
break;
case 14:
yy.ast.set({ expression: { type: "wildcard", value: $$[$0] } });
break;
case 15:
yy.ast.set({ expression: { type: "identifier", value: $$[$0] } });
break;
case 16:
yy.ast.set({ expression: { type: "script_expression", value: $$[$0] } });
break;
case 17:
yy.ast.set({ expression: { type: "numeric_literal", value: parseInt($$[$0]) } });
break;
case 24:
$$[$0].length > 1? yy.ast.set({ expression: { type: "union", value: $$[$0] } }) : this.$ = $$[$0];
break;
case 25:
this.$ = [$$[$0]];
break;
case 26:
this.$ = $$[$0-2].concat($$[$0]);
break;
case 27:
this.$ = { expression: { type: "numeric_literal", value: parseInt($$[$0]) } }; yy.ast.set(this.$);
break;
case 28:
this.$ = { expression: { type: "string_literal", value: $$[$0] } }; yy.ast.set(this.$);
break;
case 29:
this.$ = { expression: { type: "slice", value: $$[$0] } }; yy.ast.set(this.$);
break;
case 30:
this.$ = { expression: { type: "wildcard", value: $$[$0] } }; yy.ast.set(this.$);
break;
case 31:
this.$ = { expression: { type: "script_expression", value: $$[$0] } }; yy.ast.set(this.$);
break;
case 32:
this.$ = { expression: { type: "filter_expression", value: $$[$0] } }; yy.ast.set(this.$);
break;
case 33: case 34:
this.$ = $$[$0];
break;
}
},
table: [{3:1,4:[1,2],6:3,13:4,15:$V0,16:$V1,17:$V2,18:$V3,19:$V4},{1:[3]},{1:[2,1],5:10,7:11,8:12,9:13,10:14,11:15,12:$V5,14:$V6,20:16,21:17,22:$V7},{1:[2,3],5:21,7:11,8:12,9:13,10:14,11:15,12:$V5,14:$V6,20:16,21:17,22:$V7},o($V8,[2,12]),o($V8,[2,14]),o($V8,[2,15]),o($V8,[2,16]),o($V8,[2,17]),o($V8,[2,18]),{1:[2,2],7:22,8:12,9:13,10:14,11:15,12:$V5,14:$V6,20:16,21:17,22:$V7},o($V8,[2,5]),o($V8,[2,7]),o($V8,[2,8]),o($V8,[2,9]),o($V8,[2,10]),o($V8,[2,19]),o($V8,[2,20]),{13:23,15:$V0,16:$V1,17:$V2,18:$V3,19:$V4},{13:24,15:$V0,16:$V1,17:$V2,18:$V3,19:$V4,22:[1,25]},{15:$V9,17:$Va,18:$Vb,23:26,25:27,26:28,27:32,29:34,30:$Vc,31:$Vd,32:$Ve,33:$Vf},{1:[2,4],7:22,8:12,9:13,10:14,11:15,12:$V5,14:$V6,20:16,21:17,22:$V7},o($V8,[2,6]),o($V8,[2,11]),o($V8,[2,13]),{15:$V9,17:$Va,18:$Vb,23:38,25:27,26:28,27:32,29:34,30:$Vc,31:$Vd,32:$Ve,33:$Vf},{24:[1,39]},{24:[2,23]},{24:[2,24],28:[1,40]},{24:[2,30]},{24:[2,31]},{24:[2,32]},o($Vg,[2,25]),o($Vg,[2,27]),o($Vg,[2,28]),o($Vg,[2,29]),o($Vg,[2,33]),o($Vg,[2,34]),{24:[1,41]},o($V8,[2,21]),{18:$Vb,27:42,29:34,30:$Vc,32:$Ve,33:$Vf},o($V8,[2,22]),o($Vg,[2,26])],
defaultActions: {27:[2,23],29:[2,30],30:[2,31],31:[2,32]},
parseError: function parseError (str, hash) {
    if (hash.recoverable) {
        this.trace(str);
    } else {
        var error = new Error(str);
        error.hash = hash;
        throw error;
    }
},
parse: function parse(input) {
    var self = this, stack = [0], vstack = [null], lstack = [], table = this.table, yytext = '', yylineno = 0, yyleng = 0, TERROR = 2, EOF = 1;
    var args = lstack.slice.call(arguments, 1);
    var lexer = Object.create(this.lexer);
    var sharedState = { yy: {} };
    for (var k in this.yy) {
        if (Object.prototype.hasOwnProperty.call(this.yy, k)) {
            sharedState.yy[k] = this.yy[k];
        }
    }
    lexer.setInput(input, sharedState.yy);
    sharedState.yy.lexer = lexer;
    sharedState.yy.parser = this;
    if (typeof lexer.yylloc == 'undefined') {
        lexer.yylloc = {};
    }
    var yyloc = lexer.yylloc;
    lstack.push(yyloc);
    var ranges = lexer.options && lexer.options.ranges;
    if (typeof sharedState.yy.parseError === 'function') {
        this.parseError = sharedState.yy.parseError;
    } else {
        this.parseError = Object.getPrototypeOf(this).parseError;
    }
    
        var lex = function () {
            var token;
            token = lexer.lex() || EOF;
            if (typeof token !== 'number') {
                token = self.symbols_[token] || token;
            }
            return token;
        };
    var symbol, state, action, r, yyval = {}, p, len, newState, expected;
    while (true) {
        state = stack[stack.length - 1];
        if (this.defaultActions[state]) {
            action = this.defaultActions[state];
        } else {
            if (symbol === null || typeof symbol == 'undefined') {
                symbol = lex();
            }
            action = table[state] && table[state][symbol];
        }
                    if (typeof action === 'undefined' || !action.length || !action[0]) {
                var errStr = '';
                expected = [];
                for (p in table[state]) {
                    if (this.terminals_[p] && p > TERROR) {
                        expected.push('\'' + this.terminals_[p] + '\'');
                    }
                }
                if (lexer.showPosition) {
                    errStr = 'Parse error on line ' + (yylineno + 1) + ':\n' + lexer.showPosition() + '\nExpecting ' + expected.join(', ') + ', got \'' + (this.terminals_[symbol] || symbol) + '\'';
                } else {
                    errStr = 'Parse error on line ' + (yylineno + 1) + ': Unexpected ' + (symbol == EOF ? 'end of input' : '\'' + (this.terminals_[symbol] || symbol) + '\'');
                }
                this.parseError(errStr, {
                    text: lexer.match,
                    token: this.terminals_[symbol] || symbol,
                    line: lexer.yylineno,
                    loc: yyloc,
                    expected: expected
                });
            }
        if (action[0] instanceof Array && action.length > 1) {
            throw new Error('Parse Error: multiple actions possible at state: ' + state + ', token: ' + symbol);
        }
        switch (action[0]) {
        case 1:
            stack.push(symbol);
            vstack.push(lexer.yytext);
            lstack.push(lexer.yylloc);
            stack.push(action[1]);
            symbol = null;
            {
                yyleng = lexer.yyleng;
                yytext = lexer.yytext;
                yylineno = lexer.yylineno;
                yyloc = lexer.yylloc;
            }
            break;
        case 2:
            len = this.productions_[action[1]][1];
            yyval.$ = vstack[vstack.length - len];
            yyval._$ = {
                first_line: lstack[lstack.length - (len || 1)].first_line,
                last_line: lstack[lstack.length - 1].last_line,
                first_column: lstack[lstack.length - (len || 1)].first_column,
                last_column: lstack[lstack.length - 1].last_column
            };
            if (ranges) {
                yyval._$.range = [
                    lstack[lstack.length - (len || 1)].range[0],
                    lstack[lstack.length - 1].range[1]
                ];
            }
            r = this.performAction.apply(yyval, [
                yytext,
                yyleng,
                yylineno,
                sharedState.yy,
                action[1],
                vstack,
                lstack
            ].concat(args));
            if (typeof r !== 'undefined') {
                return r;
            }
            if (len) {
                stack = stack.slice(0, -1 * len * 2);
                vstack = vstack.slice(0, -1 * len);
                lstack = lstack.slice(0, -1 * len);
            }
            stack.push(this.productions_[action[1]][0]);
            vstack.push(yyval.$);
            lstack.push(yyval._$);
            newState = table[stack[stack.length - 2]][stack[stack.length - 1]];
            stack.push(newState);
            break;
        case 3:
            return true;
        }
    }
    return true;
}};
// Courtesy of jsonpath, https://github.com/dchester/jsonpath/blob/master/include/module.js

var _ast = {

  initialize: function() {
    this._nodes = [];
    this._node = {};
    this._stash = [];
  },

  set: function(props) {
    for (var k in props) this._node[k] = props[k];
    return this._node;
  },

  node: function(obj) {
    if (arguments.length) this._node = obj;
    return this._node;
  },

  push: function() {
    this._nodes.push(this._node);
    this._node = {};
  },

  unshift: function() {
    this._nodes.unshift(this._node);
    this._node = {};
  },

  yield: function() {
    var _nodes = this._nodes;
    this.initialize();
    return _nodes;
  }
};
/* generated by jison-lex 0.3.4 */
var lexer = (function(){
var lexer = ({

EOF:1,

parseError:function parseError(str, hash) {
        if (this.yy.parser) {
            this.yy.parser.parseError(str, hash);
        } else {
            throw new Error(str);
        }
    },

// resets the lexer, sets new input
setInput:function (input, yy) {
        this.yy = yy || this.yy || {};
        this._input = input;
        this._more = this._backtrack = this.done = false;
        this.yylineno = this.yyleng = 0;
        this.yytext = this.matched = this.match = '';
        this.conditionStack = ['INITIAL'];
        this.yylloc = {
            first_line: 1,
            first_column: 0,
            last_line: 1,
            last_column: 0
        };
        if (this.options.ranges) {
            this.yylloc.range = [0,0];
        }
        this.offset = 0;
        return this;
    },

// consumes and returns one char from the input
input:function () {
        var ch = this._input[0];
        this.yytext += ch;
        this.yyleng++;
        this.offset++;
        this.match += ch;
        this.matched += ch;
        var lines = ch.match(/(?:\r\n?|\n).*/g);
        if (lines) {
            this.yylineno++;
            this.yylloc.last_line++;
        } else {
            this.yylloc.last_column++;
        }
        if (this.options.ranges) {
            this.yylloc.range[1]++;
        }

        this._input = this._input.slice(1);
        return ch;
    },

// unshifts one char (or a string) into the input
unput:function (ch) {
        var len = ch.length;
        var lines = ch.split(/(?:\r\n?|\n)/g);

        this._input = ch + this._input;
        this.yytext = this.yytext.substr(0, this.yytext.length - len);
        //this.yyleng -= len;
        this.offset -= len;
        var oldLines = this.match.split(/(?:\r\n?|\n)/g);
        this.match = this.match.substr(0, this.match.length - 1);
        this.matched = this.matched.substr(0, this.matched.length - 1);

        if (lines.length - 1) {
            this.yylineno -= lines.length - 1;
        }
        var r = this.yylloc.range;

        this.yylloc = {
            first_line: this.yylloc.first_line,
            last_line: this.yylineno + 1,
            first_column: this.yylloc.first_column,
            last_column: lines ?
                (lines.length === oldLines.length ? this.yylloc.first_column : 0)
                 + oldLines[oldLines.length - lines.length].length - lines[0].length :
              this.yylloc.first_column - len
        };

        if (this.options.ranges) {
            this.yylloc.range = [r[0], r[0] + this.yyleng - len];
        }
        this.yyleng = this.yytext.length;
        return this;
    },

// When called from action, caches matched text and appends it on next action
more:function () {
        this._more = true;
        return this;
    },

// When called from action, signals the lexer that this rule fails to match the input, so the next matching rule (regex) should be tested instead.
reject:function () {
        if (this.options.backtrack_lexer) {
            this._backtrack = true;
        } else {
            return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n' + this.showPosition(), {
                text: "",
                token: null,
                line: this.yylineno
            });

        }
        return this;
    },

// retain first n characters of the match
less:function (n) {
        this.unput(this.match.slice(n));
    },

// displays already matched input, i.e. for error messages
pastInput:function () {
        var past = this.matched.substr(0, this.matched.length - this.match.length);
        return (past.length > 20 ? '...':'') + past.substr(-20).replace(/\n/g, "");
    },

// displays upcoming input, i.e. for error messages
upcomingInput:function () {
        var next = this.match;
        if (next.length < 20) {
            next += this._input.substr(0, 20-next.length);
        }
        return (next.substr(0,20) + (next.length > 20 ? '...' : '')).replace(/\n/g, "");
    },

// displays the character position where the lexing error occurred, i.e. for error messages
showPosition:function () {
        var pre = this.pastInput();
        var c = new Array(pre.length + 1).join("-");
        return pre + this.upcomingInput() + "\n" + c + "^";
    },

// test the lexed token: return FALSE when not a match, otherwise return token
test_match:function(match, indexed_rule) {
        var token,
            lines,
            backup;

        if (this.options.backtrack_lexer) {
            // save context
            backup = {
                yylineno: this.yylineno,
                yylloc: {
                    first_line: this.yylloc.first_line,
                    last_line: this.last_line,
                    first_column: this.yylloc.first_column,
                    last_column: this.yylloc.last_column
                },
                yytext: this.yytext,
                match: this.match,
                matches: this.matches,
                matched: this.matched,
                yyleng: this.yyleng,
                offset: this.offset,
                _more: this._more,
                _input: this._input,
                yy: this.yy,
                conditionStack: this.conditionStack.slice(0),
                done: this.done
            };
            if (this.options.ranges) {
                backup.yylloc.range = this.yylloc.range.slice(0);
            }
        }

        lines = match[0].match(/(?:\r\n?|\n).*/g);
        if (lines) {
            this.yylineno += lines.length;
        }
        this.yylloc = {
            first_line: this.yylloc.last_line,
            last_line: this.yylineno + 1,
            first_column: this.yylloc.last_column,
            last_column: lines ?
                         lines[lines.length - 1].length - lines[lines.length - 1].match(/\r?\n?/)[0].length :
                         this.yylloc.last_column + match[0].length
        };
        this.yytext += match[0];
        this.match += match[0];
        this.matches = match;
        this.yyleng = this.yytext.length;
        if (this.options.ranges) {
            this.yylloc.range = [this.offset, this.offset += this.yyleng];
        }
        this._more = false;
        this._backtrack = false;
        this._input = this._input.slice(match[0].length);
        this.matched += match[0];
        token = this.performAction.call(this, this.yy, this, indexed_rule, this.conditionStack[this.conditionStack.length - 1]);
        if (this.done && this._input) {
            this.done = false;
        }
        if (token) {
            return token;
        } else if (this._backtrack) {
            // recover context
            for (var k in backup) {
                this[k] = backup[k];
            }
            return false; // rule action called reject() implying the next rule should be tested instead.
        }
        return false;
    },

// return next match in input
next:function () {
        if (this.done) {
            return this.EOF;
        }
        if (!this._input) {
            this.done = true;
        }

        var token,
            match,
            tempMatch,
            index;
        if (!this._more) {
            this.yytext = '';
            this.match = '';
        }
        var rules = this._currentRules();
        for (var i = 0; i < rules.length; i++) {
            tempMatch = this._input.match(this.rules[rules[i]]);
            if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
                match = tempMatch;
                index = i;
                if (this.options.backtrack_lexer) {
                    token = this.test_match(tempMatch, rules[i]);
                    if (token !== false) {
                        return token;
                    } else if (this._backtrack) {
                        match = false;
                        continue; // rule action called reject() implying a rule MISmatch.
                    } else {
                        // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
                        return false;
                    }
                } else if (!this.options.flex) {
                    break;
                }
            }
        }
        if (match) {
            token = this.test_match(match, rules[index]);
            if (token !== false) {
                return token;
            }
            // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
            return false;
        }
        if (this._input === "") {
            return this.EOF;
        } else {
            return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. Unrecognized text.\n' + this.showPosition(), {
                text: "",
                token: null,
                line: this.yylineno
            });
        }
    },

// return next match that has a token
lex:function lex () {
        var r = this.next();
        if (r) {
            return r;
        } else {
            return this.lex();
        }
    },

// activates a new lexer condition state (pushes the new lexer condition state onto the condition stack)
begin:function begin (condition) {
        this.conditionStack.push(condition);
    },

// pop the previously active lexer condition state off the condition stack
popState:function popState () {
        var n = this.conditionStack.length - 1;
        if (n > 0) {
            return this.conditionStack.pop();
        } else {
            return this.conditionStack[0];
        }
    },

// produce the lexer rule set which is active for the currently active lexer condition state
_currentRules:function _currentRules () {
        if (this.conditionStack.length && this.conditionStack[this.conditionStack.length - 1]) {
            return this.conditions[this.conditionStack[this.conditionStack.length - 1]].rules;
        } else {
            return this.conditions["INITIAL"].rules;
        }
    },

// return the currently active lexer condition state; when an index argument is provided it produces the N-th previous condition state, if available
topState:function topState (n) {
        n = this.conditionStack.length - 1 - Math.abs(n || 0);
        if (n >= 0) {
            return this.conditionStack[n];
        } else {
            return "INITIAL";
        }
    },

// alias for begin(condition)
pushState:function pushState (condition) {
        this.begin(condition);
    },

// return the number of states currently on the stack
stateStackSize:function stateStackSize() {
        return this.conditionStack.length;
    },
options: {},
performAction: function anonymous(yy,yy_,$avoiding_name_collisions,YY_START) {
switch($avoiding_name_collisions) {
case 0:return 4
case 1:return 14
case 2:return 12
case 3:return 15
case 4:return 16
case 5:return 22
case 6:return 24
case 7:return 28
case 8:return 30
case 9:return 18
case 10:yy_.yytext = yy_.yytext.substr(1,yy_.yyleng-2); return 32;
case 11:yy_.yytext = yy_.yytext.substr(1,yy_.yyleng-2); return 33;
case 12:return 17
case 13:return 31
}
},
rules: [/^(?:\$)/,/^(?:\.\.)/,/^(?:\.)/,/^(?:\*)/,/^(?:[a-zA-Z_]+[a-zA-Z0-9_]*)/,/^(?:\[)/,/^(?:\])/,/^(?:,)/,/^(?:((-?(?:0|[1-9][0-9]*)))?\:((-?(?:0|[1-9][0-9]*)))?(\:((-?(?:0|[1-9][0-9]*)))?)?)/,/^(?:(-?(?:0|[1-9][0-9]*)))/,/^(?:"(?:\\["bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^"\\])*")/,/^(?:'(?:\\['bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^'\\])*')/,/^(?:\(.+?\)(?=\]))/,/^(?:\?\(.+?\)(?=\]))/],
conditions: {"INITIAL":{"rules":[0,1,2,3,4,5,6,7,8,9,10,11,12,13],"inclusive":true}}
});
return lexer;
})();
parser.lexer = lexer;
function Parser () {
  this.yy = {};
}
Parser.prototype = parser;parser.Parser = Parser;
return new Parser;
})();
var Parser = parser.Parser;

const cache = new Map([
  ['$', 'return scope.path.length === 0'],
  ['$..*', 'return true'],
]);

function generate(path) {
  const cachedValue = cache.get(path);
  if (cachedValue !== void 0) {
    return cachedValue === null ? cachedValue : constructFn(cachedValue);
  }

  try {
    const parser = new Parser();
    const code = baseline(parser.parse(path));
    const body = astring.generate(generateBody(code));
    cache.set(path, body);
    return constructFn(body);
  } catch (ex) {
    cache.set(path, null);
    return null;
  }
}

function generateBody(body) {
  return returnStatement(body);
}

function constructFn(body) {
  return Function(SCOPE_ID, body);
}

class JSONPathExpression {
  constructor(path, onMatch, onError) {
    this.path = path;
    this.matches = generate(path);
    this.onMatch = onMatch;
    this.onError = onError;
  }
}

class Path extends Array {
  static get [Symbol.species]() {
    return Array;
  }

  indexOf(item, fromIndex) {
    return super.indexOf(item, fromIndex);
  }
}

const isObject = maybeObj =>
  typeof maybeObj === 'object' && maybeObj !== null;

class Sandbox {
  constructor(path, root, history) {
    this._history = history === null ? [this] : history;

    this.path = path;
    this.value = root;
  }

  get pos() {
    return this.path.length - 1;
  }

  push() {
    const sandbox = new Sandbox(
      this.path,
      this.property !== null && isObject(this.value)
        ? this.value[this.property]
        : null,
      this._history,
    );
    this._history.push(sandbox);
    return sandbox;
  }

  pop() {
    this._history.length = this.path.length + 1;
    return this._history[this._history.length - 1];
  }

  get property() {
    return unwrapOrNull(this.path, this.pos);
  }

  get parentProperty() {
    const { parent } = this;
    return parent === null ? null : parent.property;
  }

  get parent() {
    return unwrapOrNull(this._history, this.pos);
  }

  get root() {
    return this._history[0].value;
  }
}

function unwrapOrNull(collection, pos) {
  return pos >= 0 && collection.length > pos ? collection[pos] : null;
}

class Scope {
  constructor(root, exprs) {
    this._lookupCache = new Set();
    this._lastIndex = -1;
    this.pos = -1;

    this.ticks = 0;

    this.exprs = exprs;
    this.markedForCollection = [];

    this.state = new Map();
    this.path = new Path();
    this.sandbox = new Sandbox(this.path, root, null);
  }

  enter(key) {
    this._lookupCache.clear();
    this.pos = 0;
    this.ticks += 1;
    this.path.push(key);
    this.sandbox = this.sandbox.push();

    return this.path.length;
  }

  get property() {
    return this.sandbox.property;
  }

  exit(pos) {
    this._lookupCache.clear();
    this._lastIndex = -1;
    this.pos = -1;
    this.path.length = pos - 1;
    this.sandbox = this.sandbox.pop();

    return this.path.length;
  }

  set lastIndex(value) {
    const exprState = this._getCacheForExpr(this.exprs[this.pos]);
    exprState.lastIndex = value;
    this._lastIndex = value;
  }

  get lastIndex() {
    if (this._lastIndex !== -1) {
      return this._lastIndex;
    }

    return this._getCacheForExpr(this.exprs[this.pos]).lastIndex;
  }

  next(path) {
    this._lastIndex = -1;
    this.pos += 1;

    return this._lookupCache.has(path);
  }

  _getCacheForExpr(expr) {
    const cache = this.state.get(expr);
    if (cache !== void 0) {
      return cache;
    }

    const newCache = {
      evalResult: {},
      lastIndex: 0,
    };

    this.state.set(expr, newCache);
    return newCache;
  }

  evaluate(code, index, id) {
    const { evalResult } = this._getCacheForExpr(this.exprs[this.pos]);

    if (this.path.length === index + 1) {
      try {
        if (constructFn(code)(this)) {
          evalResult[id] = true;
          return true;
        }
      } catch {
        evalResult[id] = false;
        // happens
      }
    }

    return evalResult[id] === true;
  }

  // store(value, id) {
  //   const { evalResult } = this._getCacheForExpr(this.exprs[this.pos]);
  //
  //   if (value === true) {
  //     evalResult[id] = this.path.length;
  //     return evalResult[id];
  //   }
  //
  //   const lastKnownIndex = evalResult[id];
  //   if (lastKnownIndex === void 0 || lastKnownIndex < fromIndex) {
  //     return -1;
  //   }
  //
  //   return lastKnownIndex;
  // }

  // todo: use this concept instead of path.indexOf
  evaluateDeep(code, fromIndex, id) {
    const { evalResult } = this._getCacheForExpr(this.exprs[this.pos]);

    try {
      if (constructFn(code)(this)) {
        evalResult[id] = this.path.length - 1;
        return evalResult[id];
      }
    } catch (ex) {
      // happens
    }

    const lastKnownIndex = evalResult[id];
    if (lastKnownIndex === void 0 || lastKnownIndex < fromIndex) {
      return -1;
    }

    return lastKnownIndex;
  }

  hit(path) {
    this._lookupCache.add(path);
  }

  miss() {
    // stub
  }

  destroy() {
    // this.markedForCollection.push(this.exprs[this.pos]);
  }

  collect() {
    while (this.markedForCollection.length > 0) {
      const expr = this.markedForCollection.pop();
      this.exprs.splice(this.exprs.indexOf(expr, 1));
      this.state.delete(expr);
    }
  }
}

function _traverse(curObj, scope) {
  for (const key of Object.keys(curObj)) {
    const value = curObj[key];
    const pos = scope.enter(key);

    for (; scope.pos < scope.exprs.length; ) {
      const expr = scope.exprs[scope.pos];
      const hasBeenMatched = scope.next(expr.path);

      try {
        hasBeenMatched;
        if (expr.matches(scope)) {
          expr.onMatch(value, scope.path);
          scope.hit(expr.path);
        }
      } catch (ex) {
        expr.onError(ex);
      }
    }

    if (value !== null && typeof value === 'object') {
      _traverse(value, scope);
    }

    scope.exit(pos);
  }

  scope.collect();
}

function traverse(root, exprs) {
  const scope = new Scope(root, exprs);

  for (const expr of scope.exprs) {
    scope.next();

    try {
      if (expr.matches(scope)) {
        expr.onMatch(root, scope.path);
      }
    } catch (ex) {
      expr.onError(ex);
    }
  }

  _traverse(root, scope);
}

exports.JSONPathExpression = JSONPathExpression;
exports.Parser = Parser;
exports.Scope = Scope;
exports.constructFn = constructFn;
exports.generate = generate;
exports.traverse = traverse;
