"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = fetch;
Object.defineProperty(exports, "Response", {
  enumerable: true,
  get: function () {
    return _response.default;
  }
});
Object.defineProperty(exports, "Headers", {
  enumerable: true,
  get: function () {
    return _headers.default;
  }
});
Object.defineProperty(exports, "Request", {
  enumerable: true,
  get: function () {
    return _request.default;
  }
});
Object.defineProperty(exports, "FetchError", {
  enumerable: true,
  get: function () {
    return _fetchError.default;
  }
});

var _http = _interopRequireDefault(require("http"));

var _https = _interopRequireDefault(require("https"));

var _zlib = _interopRequireDefault(require("zlib"));

var _stream = _interopRequireWildcard(require("stream"));

var _dataUriToBuffer = _interopRequireDefault(require("data-uri-to-buffer"));

var _body = _interopRequireWildcard(require("./body.js"));

var _response = _interopRequireDefault(require("./response.js"));

var _headers = _interopRequireWildcard(require("./headers.js"));

var _request = _interopRequireWildcard(require("./request.js"));

var _fetchError = _interopRequireDefault(require("./errors/fetch-error.js"));

var _abortError = _interopRequireDefault(require("./errors/abort-error.js"));

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Index.js
 *
 * a request API compatible with window.fetch
 *
 * All spec algorithm step numbers are based on https://fetch.spec.whatwg.org/commit-snapshots/ae716822cb3a61843226cd090eefc6589446c1d2/.
 */

/**
 * Fetch function
 *
 * @param   Mixed    url   Absolute url or Request instance
 * @param   Object   opts  Fetch options
 * @return  Promise
 */
function fetch(url, options_) {
  // Allow custom promise
  if (!fetch.Promise) {
    throw new Error('native promise missing, set fetch.Promise to your favorite alternative');
  } // Regex for data uri


  const dataUriRegex = /^\s*data:([a-z]+\/[a-z]+(;[a-z-]+=[a-z-]+)?)?(;base64)?,[\w!$&',()*+;=\-.~:@/?%\s]*\s*$/i; // If valid data uri

  if (dataUriRegex.test(url)) {
    const data = (0, _dataUriToBuffer.default)(url);
    const res = new _response.default(data, {
      headers: {
        'Content-Type': data.type
      }
    });
    return fetch.Promise.resolve(res);
  } // If invalid data uri


  if (url.toString().startsWith('data:')) {
    const request = new _request.default(url, options_);
    return fetch.Promise.reject(new _fetchError.default(`[${request.method}] ${request.url} invalid URL`, 'system'));
  }

  _body.default.Promise = fetch.Promise; // Wrap http.request into fetch

  return new fetch.Promise((resolve, reject) => {
    // Build request object
    const request = new _request.default(url, options_);
    const options = (0, _request.getNodeRequestOptions)(request);
    const send = (options.protocol === 'https:' ? _https.default : _http.default).request;
    const {
      signal
    } = request;
    let response = null;

    const abort = () => {
      const error = new _abortError.default('The operation was aborted.');
      reject(error);

      if (request.body && request.body instanceof _stream.default.Readable) {
        request.body.destroy(error);
      }

      if (!response || !response.body) {
        return;
      }

      response.body.emit('error', error);
    };

    if (signal && signal.aborted) {
      abort();
      return;
    }

    const abortAndFinalize = () => {
      abort();
      finalize();
    }; // Send request


    const request_ = send(options);

    if (signal) {
      signal.addEventListener('abort', abortAndFinalize);
    }

    function finalize() {
      request_.abort();

      if (signal) {
        signal.removeEventListener('abort', abortAndFinalize);
      }
    }

    if (request.timeout) {
      request_.setTimeout(request.timeout, () => {
        finalize();
        reject(new _fetchError.default(`network timeout at: ${request.url}`, 'request-timeout'));
      });
    }

    request_.on('error', err => {
      reject(new _fetchError.default(`request to ${request.url} failed, reason: ${err.message}`, 'system', err));
      finalize();
    });
    request_.on('response', res => {
      const headers = (0, _headers.createHeadersLenient)(res.headers); // HTTP fetch step 5

      if (fetch.isRedirect(res.statusCode)) {
        // HTTP fetch step 5.2
        const location = headers.get('Location'); // HTTP fetch step 5.3

        const locationURL = location === null ? null : new URL(location, request.url); // HTTP fetch step 5.5

        switch (request.redirect) {
          case 'error':
            reject(new _fetchError.default(`uri requested responds with a redirect, redirect mode is set to error: ${request.url}`, 'no-redirect'));
            finalize();
            return;

          case 'manual':
            // Node-fetch-specific step: make manual redirect a bit easier to use by setting the Location header value to the resolved URL.
            if (locationURL !== null) {
              // Handle corrupted header
              try {
                headers.set('Location', locationURL);
              } catch (error) {
                // istanbul ignore next: nodejs server prevent invalid response headers, we can't test this through normal request
                reject(error);
              }
            }

            break;

          case 'follow':
            {
              // HTTP-redirect fetch step 2
              if (locationURL === null) {
                break;
              } // HTTP-redirect fetch step 5


              if (request.counter >= request.follow) {
                reject(new _fetchError.default(`maximum redirect reached at: ${request.url}`, 'max-redirect'));
                finalize();
                return;
              } // HTTP-redirect fetch step 6 (counter increment)
              // Create a new Request object.


              const requestOptions = {
                headers: new _headers.default(request.headers),
                follow: request.follow,
                counter: request.counter + 1,
                agent: request.agent,
                compress: request.compress,
                method: request.method,
                body: request.body,
                signal: request.signal,
                timeout: request.timeout
              }; // HTTP-redirect fetch step 9

              if (res.statusCode !== 303 && request.body && (0, _body.getTotalBytes)(request) === null) {
                reject(new _fetchError.default('Cannot follow redirect with body being a readable stream', 'unsupported-redirect'));
                finalize();
                return;
              } // HTTP-redirect fetch step 11


              if (res.statusCode === 303 || (res.statusCode === 301 || res.statusCode === 302) && request.method === 'POST') {
                requestOptions.method = 'GET';
                requestOptions.body = undefined;
                requestOptions.headers.delete('content-length');
              } // HTTP-redirect fetch step 15


              resolve(fetch(new _request.default(locationURL, requestOptions)));
              finalize();
              return;
            }

          default: // Do nothing

        }
      } // Prepare response


      res.once('end', () => {
        if (signal) {
          signal.removeEventListener('abort', abortAndFinalize);
        }
      });
      let body = (0, _stream.pipeline)(res, new _stream.PassThrough(), error => {
        reject(error);
      });
      const responseOptions = {
        url: request.url,
        status: res.statusCode,
        statusText: res.statusMessage,
        headers,
        size: request.size,
        timeout: request.timeout,
        counter: request.counter,
        highWaterMark: request.highWaterMark
      }; // HTTP-network fetch step 12.1.1.3

      const codings = headers.get('Content-Encoding'); // HTTP-network fetch step 12.1.1.4: handle content codings
      // in following scenarios we ignore compression support
      // 1. compression support is disabled
      // 2. HEAD request
      // 3. no Content-Encoding header
      // 4. no content response (204)
      // 5. content not modified response (304)

      if (!request.compress || request.method === 'HEAD' || codings === null || res.statusCode === 204 || res.statusCode === 304) {
        response = new _response.default(body, responseOptions);
        resolve(response);
        return;
      } // For Node v6+
      // Be less strict when decoding compressed responses, since sometimes
      // servers send slightly invalid responses that are still accepted
      // by common browsers.
      // Always using Z_SYNC_FLUSH is what cURL does.


      const zlibOptions = {
        flush: _zlib.default.Z_SYNC_FLUSH,
        finishFlush: _zlib.default.Z_SYNC_FLUSH
      }; // For gzip

      if (codings === 'gzip' || codings === 'x-gzip') {
        body = (0, _stream.pipeline)(body, _zlib.default.createGunzip(zlibOptions), error => {
          reject(error);
        });
        response = new _response.default(body, responseOptions);
        resolve(response);
        return;
      } // For deflate


      if (codings === 'deflate' || codings === 'x-deflate') {
        // Handle the infamous raw deflate response from old servers
        // a hack for old IIS and Apache servers
        const raw = (0, _stream.pipeline)(res, new _stream.PassThrough(), error => {
          reject(error);
        });
        raw.once('data', chunk => {
          // See http://stackoverflow.com/questions/37519828
          if ((chunk[0] & 0x0F) === 0x08) {
            body = (0, _stream.pipeline)(body, _zlib.default.createInflate(), error => {
              reject(error);
            });
          } else {
            body = (0, _stream.pipeline)(body, _zlib.default.createInflateRaw(), error => {
              reject(error);
            });
          }

          response = new _response.default(body, responseOptions);
          resolve(response);
        });
        return;
      } // For br


      if (codings === 'br' && typeof _zlib.default.createBrotliDecompress === 'function') {
        body = (0, _stream.pipeline)(body, _zlib.default.createBrotliDecompress(), error => {
          reject(error);
        });
        response = new _response.default(body, responseOptions);
        resolve(response);
        return;
      } // Otherwise, use response as-is


      response = new _response.default(body, responseOptions);
      resolve(response);
    });
    (0, _body.writeToStream)(request_, request);
  });
}
/**
 * Redirect code matching
 *
 * @param   Number   code  Status code
 * @return  Boolean
 */


fetch.isRedirect = code => [301, 302, 303, 307, 308].includes(code); // Expose Promise


fetch.Promise = global.Promise;