var ABGRColor, COMPONENT_BLUE, COMPONENT_GREEN, COMPONENT_RED, Color, ColorCutQuantizer, QUANTIZE_WORD_MASK, QUANTIZE_WORD_WIDTH, RGBAColor, Swatch, Vbox, approximateToRgb888, isLittleEndian, modifySignificantOctet, modifyWordWidth, quantizeFromRgb888, quantizedBlue, quantizedGreen, quantizedRed, sort;

Swatch = require('../../swatch');

sort = function(arr, lower, upper) {
  var partition, pivot, swap;
  swap = function(a, b) {
    var t;
    t = arr[a];
    arr[a] = arr[b];
    return arr[b] = t;
  };
  partition = function(pivot, left, right) {
    var index, j, ref, ref1, v, value;
    index = left;
    value = arr[pivot];
    swap(pivot, right);
    for (v = j = ref = left, ref1 = right - 1; ref <= ref1 ? j <= ref1 : j >= ref1; v = ref <= ref1 ? ++j : --j) {
      if (arr[v] > value) {
        swap(v, index);
        index++;
      }
    }
    swap(right, index);
    return index;
  };
  if (lower < upper) {
    pivot = lower + Math.ceil((upper - lower) / 2);
    pivot = partition(pivot, lower, upper);
    sort(arr, lower, pivot - 1);
    return sort(arr, pivot + 1, upper);
  }
};

COMPONENT_RED = -3;

COMPONENT_GREEN = -2;

COMPONENT_BLUE = -1;

QUANTIZE_WORD_WIDTH = 5;

QUANTIZE_WORD_MASK = (1 << QUANTIZE_WORD_WIDTH) - 1;

RGBAColor = {
  red: function(c) {
    return c >> 24;
  },
  green: function(c) {
    return c << 8 >> 24;
  },
  blue: function(c) {
    return c << 16 >> 24;
  },
  alpha: function(c) {
    return c << 24 >> 24;
  }
};

ABGRColor = {
  red: function(c) {
    return c << 24 >> 24;
  },
  green: function(c) {
    return c << 16 >> 24;
  },
  blue: function(c) {
    return c << 8 >> 24;
  },
  alpha: function(c) {
    return c >> 24;
  }
};

isLittleEndian = function() {
  var a, b, c;
  a = new ArrayBuffer(4);
  b = new Uint8Array(a);
  c = new Uint32Array(a);
  b[0] = 0xa1;
  b[1] = 0xb2;
  b[2] = 0xc3;
  b[3] = 0xd4;
  if (c[0] === 0xd4c3b2a1) {
    return true;
  }
  if (c[0] === 0xa1b2c3d4) {
    return false;
  }
  throw new Error("Failed to determin endianness");
};

Color = isLittleEndian() ? ABGRColor : RGBAColor;

modifyWordWidth = function(value, current, target) {
  var newValue;
  newValue = 0;
  if (target > current) {
    newValue = value << (target - current);
  } else {
    newValue = value >> (current - target);
  }
  return newValue & ((1 << target) - 1);
};

modifySignificantOctet = function(a, dimension, lower, upper) {
  var color, i, j, k, ref, ref1, ref2, ref3;
  switch (dimension) {
    case COMPONENT_RED:
      break;
    case COMPONENT_GREEN:
      for (i = j = ref = lower, ref1 = upper; ref <= ref1 ? j <= ref1 : j >= ref1; i = ref <= ref1 ? ++j : --j) {
        color = a[i];
        a[i] = quantizedGreen(color) << (QUANTIZE_WORD_WIDTH + QUANTIZE_WORD_WIDTH) | quantizedRed(color) << QUANTIZE_WORD_WIDTH | quantizedBlue(color);
      }
      break;
    case COMPONENT_BLUE:
      for (i = k = ref2 = lower, ref3 = upper; ref2 <= ref3 ? k <= ref3 : k >= ref3; i = ref2 <= ref3 ? ++k : --k) {
        color = a[i];
        a[i] = quantizedBlue(color) << (QUANTIZE_WORD_WIDTH + QUANTIZE_WORD_WIDTH) | quantizedGreen(color) << QUANTIZE_WORD_WIDTH | quantizedRed(color);
      }
      break;
  }
};

quantizeFromRgb888 = function(color) {
  var b, g, r;
  r = modifyWordWidth(Color.red(color), 8, QUANTIZE_WORD_WIDTH);
  g = modifyWordWidth(Color.green(color), 8, QUANTIZE_WORD_WIDTH);
  b = modifyWordWidth(Color.blue(color), 8, QUANTIZE_WORD_WIDTH);
  return r << (QUANTIZE_WORD_WIDTH + QUANTIZE_WORD_WIDTH) | g << QUANTIZE_WORD_WIDTH | b;
};

approximateToRgb888 = function(r, g, b) {
  var color;
  if (!((g != null) && (b != null))) {
    color = r;
    r = quantizedRed(color);
    g = quantizedGreen(color);
    b = quantizedBlue(color);
  }
  return [modifyWordWidth(r, QUANTIZE_WORD_WIDTH, 8), modifyWordWidth(g, QUANTIZE_WORD_WIDTH, 8), modifyWordWidth(b, QUANTIZE_WORD_WIDTH, 8)];
};

quantizedRed = function(color) {
  return color >> (QUANTIZE_WORD_WIDTH + QUANTIZE_WORD_WIDTH) & QUANTIZE_WORD_MASK;
};

quantizedGreen = function(color) {
  return color >> QUANTIZE_WORD_WIDTH & QUANTIZE_WORD_MASK;
};

quantizedBlue = function(color) {
  return color & QUANTIZE_WORD_MASK;
};

module.exports = ColorCutQuantizer = (function() {
  function ColorCutQuantizer(data, opts) {
    var c, color, distinctColorCount, distinctColorIndex, i, j, k, l, m, quantizedColor, ref, ref1, ref2, ref3;
    this.opts = opts;
    this.hist = new Uint32Array(1 << (QUANTIZE_WORD_WIDTH * 3));
    this.pixels = new Uint32Array(data.length);
    for (i = j = 0, ref = data.length - 1; 0 <= ref ? j <= ref : j >= ref; i = 0 <= ref ? ++j : --j) {
      this.pixels[i] = quantizedColor = quantizeFromRgb888(data[i]);
      this.hist[quantizedColor]++;
    }
    distinctColorCount = 0;
    for (color = k = 0, ref1 = this.hist.length - 1; 0 <= ref1 ? k <= ref1 : k >= ref1; color = 0 <= ref1 ? ++k : --k) {
      if (this.hist[color] > 0) {
        distinctColorCount++;
      }
    }
    this.colors = new Uint32Array(distinctColorCount);
    distinctColorIndex = 0;
    for (color = l = 0, ref2 = this.hist.length - 1; 0 <= ref2 ? l <= ref2 : l >= ref2; color = 0 <= ref2 ? ++l : --l) {
      if (this.hist[color] > 0) {
        this.colors[distinctColorIndex++] = color;
      }
    }
    if (distinctColorCount <= this.opts.colorCount) {
      this.quantizedColors = [];
      for (i = m = 0, ref3 = this.colors.length - 1; 0 <= ref3 ? m <= ref3 : m >= ref3; i = 0 <= ref3 ? ++m : --m) {
        c = this.colors[i];
        this.quantizedColors.push(new Swatch(approximateToRgb888(c), this.hist[c]));
      }
    } else {
      this.quantizedColors = this.quantizePixels(this.opts.colorCount);
    }
  }

  ColorCutQuantizer.prototype.getQuantizedColors = function() {
    return this.quantizedColors;
  };

  ColorCutQuantizer.prototype.quantizePixels = function(maxColors) {
    var pq;
    pq = new PriorityQueue({
      comparator: Vbox.comparator
    });
    pq.queue(new Vbox(this.colors, this.hist, 0, this.colors.length - 1));
    this.splitBoxes(pq, maxColors);
    return this.generateAverageColors(pq);
  };

  ColorCutQuantizer.prototype.splitBoxes = function(queue, maxSize) {
    var vbox;
    while (queue.length < maxSize) {
      vbox = queue.dequeue();
      if (vbox != null ? vbox.canSplit() : void 0) {
        queue.queue(vbox.splitBox());
        queue.queue(vbox);
      } else {
        return;
      }
    }
  };

  ColorCutQuantizer.prototype.generateAverageColors = function(vboxes) {
    var colors;
    colors = [];
    while (vboxes.length > 0) {
      colors.push(vboxes.dequeue().getAverageColor());
    }
    return colors;
  };

  return ColorCutQuantizer;

})();

Vbox = (function() {
  Vbox.comparator = function(lhs, rhs) {
    return lhs.getVolume() - rhs.getVolume();
  };

  function Vbox(colors1, hist, lowerIndex, upperIndex) {
    this.colors = colors1;
    this.hist = hist;
    this.lowerIndex = lowerIndex;
    this.upperIndex = upperIndex;
    this.fitBox();
  }

  Vbox.prototype.getVolume = function() {
    return (this.maxRed - this.minRed + 1) * (this.maxGreen - this.minGreen + 1) * (this.maxBlue - this.minBlue + 1);
  };

  Vbox.prototype.canSplit = function() {
    return this.getColorCount() > 1;
  };

  Vbox.prototype.getColorCount = function() {
    return 1 + this.upperIndex - this.lowerIndex;
  };

  Vbox.prototype.fitBox = function() {
    var b, color, count, g, i, j, r, ref, ref1;
    this.minRed = this.minGreen = this.minBlue = Number.MAX_VALUE;
    this.maxRed = this.maxGreen = this.maxBlue = Number.MIN_VALUE;
    this.population = 0;
    count = 0;
    for (i = j = ref = this.lowerIndex, ref1 = this.upperIndex; ref <= ref1 ? j <= ref1 : j >= ref1; i = ref <= ref1 ? ++j : --j) {
      color = this.colors[i];
      count += this.hist[color];
      r = quantizedRed(color);
      g = quantizedGreen(color);
      b = quantizedBlue(color);
      if (r > this.maxRed) {
        this.maxRed = r;
      }
      if (r < this.minRed) {
        this.minRed = r;
      }
      if (g > this.maxGreen) {
        this.maxGreen = g;
      }
      if (g < this.minGreen) {
        this.minGreen = g;
      }
      if (b > this.maxBlue) {
        this.maxRed = b;
      }
      if (b < this.minBlue) {
        this.minRed = b;
      }
    }
    return this.population = count;
  };

  Vbox.prototype.splitBox = function() {
    var newBox, splitPoint;
    if (!this.canSplit()) {
      throw new Error("Cannot split a box with only 1 color");
    }
    splitPoint = this.findSplitPoint();
    newBox = new Vbox(this.colors, this.hist, splitPoint + 1, this.upperIndex);
    this.upperIndex = splitPoint;
    this.fitBox();
    return newBox;
  };

  Vbox.prototype.getLongestColorDimension = function() {
    var blueLength, greenLength, redLength;
    redLength = this.maxRed - this.minRed;
    greenLength = this.maxGreen - this.minGreen;
    blueLength = this.maxBlue - this.minBlue;
    if (redLength >= greenLength && redLength >= blueLength) {
      return COMPONENT_RED;
    }
    if (greenLength >= redLength && greenLength >= blueLength) {
      return COMPONENT_GREEN;
    }
    return COMPONENT_BLUE;
  };

  Vbox.prototype.findSplitPoint = function() {
    var count, i, j, longestDimension, midPoint, ref, ref1;
    longestDimension = this.getLongestColorDimension();
    modifySignificantOctet(this.colors, longestDimension, this.lowerIndex, this.upperIndex);
    sort(this.colors, this.lowerIndex, this.upperIndex + 1);
    modifySignificantOctet(this.colors, longestDimension, this.lowerIndex, this.upperIndex);
    midPoint = this.population / 2;
    count = 0;
    for (i = j = ref = this.lowerIndex, ref1 = this.upperIndex; ref <= ref1 ? j <= ref1 : j >= ref1; i = ref <= ref1 ? ++j : --j) {
      count += this.hist[this.colors[i]];
      if (count >= midPoint) {
        return i;
      }
    }
    return this.lowerIndex;
  };

  Vbox.prototype.getAverageColor = function() {
    var blueMean, blueSum, color, colorPopulation, greenMean, greenSum, i, j, redMean, redSum, ref, ref1, totalPopulation;
    redSum = greenSum = blueSum = 0;
    totalPopulation = 0;
    for (i = j = ref = this.lowerIndex, ref1 = this.upperIndex; ref <= ref1 ? j <= ref1 : j >= ref1; i = ref <= ref1 ? ++j : --j) {
      color = this.colors[i];
      colorPopulation = this.hist[color];
      totalPopulation += colorPopulation;
      redSum += colorPopulation * quantizedRed(color);
      greenSum += colorPopulation * quantizedGreen(color);
      blueSum += colorPopulation * quantizedBlue(color);
    }
    redMean = Math.round(redSum / totalPopulation);
    greenMean = Math.round(greenSum / totalPopulation);
    blueMean = Math.round(blueSum / totalPopulation);
    return new Swatch(approximateToRgb888(redMean, greenMean, blueMean), totalPopulation);
  };

  return Vbox;

})();
