/*
 * quantize.js Copyright 2008 Nick Rabinowitz
 * Ported to node.js by Olivier Lesnicki
 * Licensed under the MIT license: http://www.opensource.org/licenses/mit-license.php
 */

// fill out a couple protovis dependencies
/*
 * Block below copied from Protovis: http://mbostock.github.com/protovis/
 * Copyright 2010 Stanford Visualization Group
 * Licensed under the BSD License: http://www.opensource.org/licenses/bsd-license.php
 */
if (!pv) {
    var pv = {
        map: function(array, f) {
            var o = {};
            return f ? array.map(function(d, i) {
                o.index = i;
                return f.call(o, d);
            }) : array.slice();
        },
        naturalOrder: function(a, b) {
            return a - b;
        },
        sum: function(array, f) {
            var o = {};
            return array.reduce(f ? function(p, d, i) {
                o.index = i;
                return p + f.call(o, d);
            } : function(p, d) {
                return p + d;
            }, 0);
        },
        max: function(array, f) {
            return Math.max.apply(null, f ? pv.map(array, f) : array);
        }
    }
}

/**
 * Basic Javascript port of the MMCQ (modified median cut quantization)
 * algorithm from the Leptonica library (http://www.leptonica.com/).
 * Returns a color map you can use to map original pixels to the reduced
 * palette. Still a work in progress.
 *
 * @author Nick Rabinowitz
 * @example

// array of pixels as [R,G,B] arrays
var myPixels = [[190,197,190], [202,204,200], [207,214,210], [211,214,211], [205,207,207]
                // etc
                ];
var maxColors = 4;

var cmap = MMCQ.quantize(myPixels, maxColors);
var newPalette = cmap.palette();
var newPixels = myPixels.map(function(p) {
    return cmap.map(p);
});

 */
var MMCQ = (function() {
    // private constants
    var sigbits = 5,
        rshift = 8 - sigbits,
        maxIterations = 1000,
        fractByPopulations = 0.75;

    // get reduced-space color index for a pixel

    function getColorIndex(r, g, b) {
        return (r << (2 * sigbits)) + (g << sigbits) + b;
    }

    // Simple priority queue

    function PQueue(comparator) {
        var contents = [],
            sorted = false;

        function sort() {
            contents.sort(comparator);
            sorted = true;
        }

        return {
            push: function(o) {
                contents.push(o);
                sorted = false;
            },
            peek: function(index) {
                if (!sorted) sort();
                if (index === undefined) index = contents.length - 1;
                return contents[index];
            },
            pop: function() {
                if (!sorted) sort();
                return contents.pop();
            },
            size: function() {
                return contents.length;
            },
            map: function(f) {
                return contents.map(f);
            },
            debug: function() {
                if (!sorted) sort();
                return contents;
            }
        };
    }

    // 3d color space box

    function VBox(r1, r2, g1, g2, b1, b2, histo) {
        var vbox = this;
        vbox.r1 = r1;
        vbox.r2 = r2;
        vbox.g1 = g1;
        vbox.g2 = g2;
        vbox.b1 = b1;
        vbox.b2 = b2;
        vbox.histo = histo;
    }
    VBox.prototype = {
        volume: function(force) {
            var vbox = this;
            if (!vbox._volume || force) {
                vbox._volume = ((vbox.r2 - vbox.r1 + 1) * (vbox.g2 - vbox.g1 + 1) * (vbox.b2 - vbox.b1 + 1));
            }
            return vbox._volume;
        },
        count: function(force) {
            var vbox = this,
                histo = vbox.histo;
            if (!vbox._count_set || force) {
                var npix = 0,
                    i, j, k;
                for (i = vbox.r1; i <= vbox.r2; i++) {
                    for (j = vbox.g1; j <= vbox.g2; j++) {
                        for (k = vbox.b1; k <= vbox.b2; k++) {
                            index = getColorIndex(i, j, k);
                            npix += histo[index];
                        }
                    }
                }
                vbox._count = npix;
                vbox._count_set = true;
            }
            return vbox._count;
        },
        copy: function() {
            var vbox = this;
            return new VBox(vbox.r1, vbox.r2, vbox.g1, vbox.g2, vbox.b1, vbox.b2, vbox.histo);
        },
        avg: function(force) {
            var vbox = this,
                histo = vbox.histo;
            if (!vbox._avg || force) {
                var ntot = 0,
                    mult = 1 << (8 - sigbits),
                    // mult = (8 - sigbits),
                    rsum = 0,
                    gsum = 0,
                    bsum = 0,
                    hval,
                    i, j, k, histoindex;
                for (i = vbox.r1; i <= vbox.r2; i++) {
                    for (j = vbox.g1; j <= vbox.g2; j++) {
                        for (k = vbox.b1; k <= vbox.b2; k++) {
                            histoindex = getColorIndex(i, j, k);
                            hval = histo[histoindex];
                            ntot += hval;
                            rsum += (hval * (i + 0.5) * mult);
                            gsum += (hval * (j + 0.5) * mult);
                            bsum += (hval * (k + 0.5) * mult);
                        }
                    }
                }
                if (ntot) {
                    vbox._avg = [~~(rsum / ntot), ~~ (gsum / ntot), ~~ (bsum / ntot)];
                } else {
                    //console.log('empty box');
                    vbox._avg = [~~(mult * (vbox.r1 + vbox.r2 + 1) / 2), ~~ (mult * (vbox.g1 + vbox.g2 + 1) / 2), ~~ (mult * (vbox.b1 + vbox.b2 + 1) / 2)];
                }
            }
            return vbox._avg;
        },
        contains: function(pixel) {
            var vbox = this,
                rval = pixel[0] >> rshift;
            gval = pixel[1] >> rshift;
            bval = pixel[2] >> rshift;
            return (rval >= vbox.r1 && rval <= vbox.r2 &&
                gval >= vbox.g1 && gval <= vbox.g2 &&
                bval >= vbox.b1 && bval <= vbox.b2);
        }
    };

    // Color map

    function CMap() {
        this.vboxes = new PQueue(function(a, b) {
            return pv.naturalOrder(
                a.vbox.count() * a.vbox.volume(),
                b.vbox.count() * b.vbox.volume()
            )
        });;
    }
    CMap.prototype = {
        push: function(vbox) {
            this.vboxes.push({
                vbox: vbox,
                color: vbox.avg()
            });
        },
        palette: function() {
            return this.vboxes.map(function(vb) {
                return vb.color
            });
        },
        size: function() {
            return this.vboxes.size();
        },
        map: function(color) {
            var vboxes = this.vboxes;
            for (var i = 0; i < vboxes.size(); i++) {
                if (vboxes.peek(i).vbox.contains(color)) {
                    return vboxes.peek(i).color;
                }
            }
            return this.nearest(color);
        },
        nearest: function(color) {
            var vboxes = this.vboxes,
                d1, d2, pColor;
            for (var i = 0; i < vboxes.size(); i++) {
                d2 = Math.sqrt(
                    Math.pow(color[0] - vboxes.peek(i).color[0], 2) +
                    Math.pow(color[1] - vboxes.peek(i).color[1], 2) +
                    Math.pow(color[2] - vboxes.peek(i).color[2], 2)
                );
                if (d2 < d1 || d1 === undefined) {
                    d1 = d2;
                    pColor = vboxes.peek(i).color;
                }
            }
            return pColor;
        },
        forcebw: function() {
            // XXX: won't  work yet
            var vboxes = this.vboxes;
            vboxes.sort(function(a, b) {
                return pv.naturalOrder(pv.sum(a.color), pv.sum(b.color))
            });

            // force darkest color to black if everything < 5
            var lowest = vboxes[0].color;
            if (lowest[0] < 5 && lowest[1] < 5 && lowest[2] < 5)
                vboxes[0].color = [0, 0, 0];

            // force lightest color to white if everything > 251
            var idx = vboxes.length - 1,
                highest = vboxes[idx].color;
            if (highest[0] > 251 && highest[1] > 251 && highest[2] > 251)
                vboxes[idx].color = [255, 255, 255];
        }
    };


    function getAll(pixels, shouldIgnore) {
        var histosize = 1 << (3 * sigbits),
            histo = new Uint32Array(histosize),
            index, rval, gval, bval;
        var rmin = 1000000,
            rmax = 0,
            gmin = 1000000,
            gmax = 0,
            bmin = 1000000,
            bmax = 0;

        var pixelCount = pixels.length / 4,
            i = 0;

        // Yes, it matters
        if (typeof shouldIgnore === 'function') {
          while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              if (shouldIgnore(r, g, b, a)) continue;
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              index = getColorIndex(rval, gval, bval);
              histo[index]++;
              if (rval < rmin) rmin = rval;
              else if (rval > rmax) rmax = rval;
              if (gval < gmin) gmin = gval;
              else if (gval > gmax) gmax = gval;
              if (bval < bmin) bmin = bval;
              else if (bval > bmax) bmax = bval;
          }
        } else {
          while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              index = getColorIndex(rval, gval, bval);
              histo[index]++;
              if (rval < rmin) rmin = rval;
              else if (rval > rmax) rmax = rval;
              if (gval < gmin) gmin = gval;
              else if (gval > gmax) gmax = gval;
              if (bval < bmin) bmin = bval;
              else if (bval > bmax) bmax = bval;
          }
        }

        return {
          histo: histo,
          vbox: new VBox(rmin, rmax, gmin, gmax, bmin, bmax, histo)
        };
    }

    // histo (1-d array, giving the number of pixels in
    // each quantized region of color space), or null on error

    function getHisto(pixels, shouldIgnore) {
        var histosize = 1 << (3 * sigbits),
            histo = new Uint32Array(histosize),
            index, rval, gval, bval;

        var pixelCount = pixels.length / 4,
            i = 0;

        // Yes, it matters
        if (typeof shouldIgnore === 'function') {
          while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              if (shouldIgnore(r, g, b, a)) continue;
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              index = getColorIndex(rval, gval, bval);
              histo[index]++;
          }
        } else {
          while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              index = getColorIndex(rval, gval, bval);
              histo[index]++;
          }
        }

        return histo;
    }

    function vboxFromPixels(pixels, histo, shouldIgnore) {
        var rmin = 1000000,
            rmax = 0,
            gmin = 1000000,
            gmax = 0,
            bmin = 1000000,
            bmax = 0,
            rval, gval, bval;
        // find min/max
        var pixelCount = pixels.length / 4,
            i = 0;

        // Yes, it matters
        if (typeof shouldIgnore === 'function') {
          while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              if (shouldIgnore(r, g, b, a)) continue;
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              if (rval < rmin) rmin = rval;
              else if (rval > rmax) rmax = rval;
              if (gval < gmin) gmin = gval;
              else if (gval > gmax) gmax = gval;
              if (bval < bmin) bmin = bval;
              else if (bval > bmax) bmax = bval;
          }
        } else {
            while (i < pixelCount) {
              offset = i * 4;
              i++;
              r = pixels[offset + 0];
              g = pixels[offset + 1];
              b = pixels[offset + 2];
              a = pixels[offset + 3];
              rval = r >> rshift;
              gval = g >> rshift;
              bval = b >> rshift;
              if (rval < rmin) rmin = rval;
              else if (rval > rmax) rmax = rval;
              if (gval < gmin) gmin = gval;
              else if (gval > gmax) gmax = gval;
              if (bval < bmin) bmin = bval;
              else if (bval > bmax) bmax = bval;
          }
        }
        return new VBox(rmin, rmax, gmin, gmax, bmin, bmax, histo);
    }

    function medianCutApply(histo, vbox) {
        if (!vbox.count()) return;

        var rw = vbox.r2 - vbox.r1 + 1,
            gw = vbox.g2 - vbox.g1 + 1,
            bw = vbox.b2 - vbox.b1 + 1,
            maxw = pv.max([rw, gw, bw]);
        // only one pixel, no split
        if (vbox.count() == 1) {
            return [vbox.copy()]
        }
        /* Find the partial sum arrays along the selected axis. */
        var total = 0,
            partialsum,
            lookaheadsum,
            i, j, k, sum, index;
        // var D = ['r', 'g', 'b'],
        //   indexer = getColorIndex;
        // if (maxw == gw) {
        //   D = ['g', 'r', 'b'];
        //   indexer = function(g, r, b) { return getColorIndex(r, g, b); };
        // } else if (maxw == bw) {
        //   indexer = function(b, r, g) { return getColorIndex(r, g, b); };
        //   D = ['b', 'r', 'g'];
        // }
        // partialsum = new Uint32Array(vbox[D[0] + "2"] + 1);
        // console.log(vbox[D[0] + "2"])
        // for (i = vbox[D[0] + "1"]; i <= vbox[D[0] + "2"]; i++) {
        //     sum = 0;
        //     for (j = vbox[D[1] + "1"]; j <= vbox[D[1] + "2"]; j++) {
        //         for (k = vbox[D[2] + "1"]; k <= vbox[D[2] + "2"]; k++) {
        //             index = indexer(i, j, k);
        //             sum += histo[index];
        //         }
        //     }
        //     total += sum;
        //     console.log(i + "->" + total)
        //     partialsum[i] = total;
        // }
        var maxd = 'b';
        if (maxw == rw) {
            maxd = 'r';
            partialsum = new Uint32Array(vbox.r2 + 1);
            for (i = vbox.r1; i <= vbox.r2; i++) {
                sum = 0;
                for (j = vbox.g1; j <= vbox.g2; j++) {
                    for (k = vbox.b1; k <= vbox.b2; k++) {
                        index = getColorIndex(i, j, k);
                        sum += histo[index];
                    }
                }
                total += sum;
                partialsum[i] = total;
            }
        } else if (maxw == gw) {
            maxd = 'g';
            partialsum = new Uint32Array(vbox.g2 + 1);
            for (i = vbox.g1; i <= vbox.g2; i++) {
                sum = 0;
                for (j = vbox.r1; j <= vbox.r2; j++) {
                    for (k = vbox.b1; k <= vbox.b2; k++) {
                        index = getColorIndex(j, i, k);
                        sum += histo[index];
                    }
                }
                total += sum;
                partialsum[i] = total;
            }
        } else { /* maxw == bw */
            // maxd = 'b';
            partialsum = new Uint32Array(vbox.b2 + 1);
            for (i = vbox.b1; i <= vbox.b2; i++) {
                sum = 0;
                for (j = vbox.r1; j <= vbox.r2; j++) {
                    for (k = vbox.g1; k <= vbox.g2; k++) {
                        index = getColorIndex(j, k, i);
                        sum += histo[index];
                    }
                }
                total += sum;
                partialsum[i] = total;
            }
        }
        var splitPoint = -1;
        lookaheadsum = new Uint32Array(partialsum.length);
        for (i = 0; i < partialsum.length; i++) {
          var d = partialsum[i];
          if (splitPoint < 0 && d > (total / 2)) splitPoint = i;
          lookaheadsum[i] = total - d
        }
        // partialsum.forEach(function(d, i) {
        //   if (splitPoint < 0 && d > (total / 2)) splitPoint = i
        //     lookaheadsum[i] = total - d
        // });

        // console.log('cut')
        function doCut(color) {
            var dim1 = color + '1',
                dim2 = color + '2',
                left, right, vbox1, vbox2, d2, count2 = 0,
                i = splitPoint;
            vbox1 = vbox.copy();
            vbox2 = vbox.copy();
            left = i - vbox[dim1];
            right = vbox[dim2] - i;
            if (left <= right) {
                d2 = Math.min(vbox[dim2] - 1, ~~ (i + right / 2));
                d2 = Math.max(0, d2);
            } else {
                d2 = Math.max(vbox[dim1], ~~ (i - 1 - left / 2));
                d2 = Math.min(vbox[dim2], d2);
            }
            // console.log(partialsum[d2])
            // avoid 0-count boxes
            while (!partialsum[d2]) d2++;
            count2 = lookaheadsum[d2];
            // console.log('-_-')
            while (!count2 && partialsum[d2 - 1]) count2 = lookaheadsum[--d2];
            // set dimensions
            vbox1[dim2] = d2;
            vbox2[dim1] = vbox1[dim2] + 1;
            // console.log('vbox counts:', vbox.count(), vbox1.count(), vbox2.count());
            return [vbox1, vbox2];

        }
        // determine the cut planes
        return doCut(maxd);
        // return maxw == rw ? doCut('r') :
        //     maxw == gw ? doCut('g') :
        //     doCut('b');
    }

    function quantize(pixels, opts) {
        var maxcolors = opts.colorCount;
        // short-circuit
        if (!pixels.length || maxcolors < 2 || maxcolors > 256) {
            // console.log('wrong number of maxcolors');
            return false;
        }

        var hasFilters = Array.isArray(opts.filters) && opts.filters.length > 0;
        function shouldIgnore(r, g, b, a) {
          for (var i = 0; i < opts.filters.length; i++) {
            var f = opts.filters[i];
            if (!f(r, g, b, a)) {
              return true;
            }
          }
          return false;
        }

        var r = getAll(pixels, hasFilters ? houldIgnore : null);
        // XXX: check color content and convert to grayscale if insufficient

        // var histo = getHisto(pixels, hasFilters ? shouldIgnore : null),
        var histo = r.histo,
            histosize = 1 << (3 * sigbits);

        // check that we aren't below maxcolors already
        var nColors = Object.keys(histo).length;
        if (nColors <= maxcolors) {
            // XXX: generate the new colors from the histo and return
        }

        // get the beginning vbox from the colors
        // var vbox = vboxFromPixels(pixels, histo, hasFilters ? shouldIgnore : null),
        var vbox = r.vbox,
            pq = new PQueue(function(a, b) {
                return pv.naturalOrder(a.count(), b.count())
            });
        pq.push(vbox);

        // inner function to do the iteration

        function iter(lh, target) {
            var ncolors = 1,
                niters = 0,
                vbox;
            while (niters < maxIterations) {
                vbox = lh.pop();
                if (!vbox.count()) { /* just put it back */
                    // lh.push(vbox); // Maybe not
                    niters++;
                    continue;
                }
                // do the cut
                var vboxes = medianCutApply(histo, vbox),
                    vbox1 = vboxes[0],
                    vbox2 = vboxes[1];

                if (!vbox1) {
                    // console.log("vbox1 not defined; shouldn't happen!");
                    return;
                }
                lh.push(vbox1);
                if (vbox2) { /* vbox2 can be null */
                    lh.push(vbox2);
                    ncolors++;
                }
                if (ncolors >= target) return;
                if (niters++ > maxIterations) {
                    return;
                }
            }
        }

        // first set of colors, sorted by population
        iter(pq, fractByPopulations * maxcolors);
        // console.log(pq.size(), pq.debug().length, pq.debug().slice());

        // Re-sort by the product of pixel occupancy times the size in color space.
        var pq2 = new PQueue(function(a, b) {
            return pv.naturalOrder(a.count() * a.volume(), b.count() * b.volume())
        });
        while (pq.size()) {
            pq2.push(pq.pop());
        }

        // next set - generate the median cuts using the (npix * vol) sorting.
        iter(pq2, maxcolors - pq2.size());

        // calculate the actual colors
        var cmap = new CMap();
        while (pq2.size()) {
            var v = pq2.pop(),
              c = vbox.avg();
            if (!hasFilters || !shouldIgnore(c[0], c[1], c[2], 255)) {
              cmap.push(v);
            }
        }

        return cmap;
    }

    return {
        quantize: quantize,
        getAll: getAll,
        medianCutApply: medianCutApply
    }
})();

module.exports = MMCQ.quantize
module.exports.getAll = MMCQ.getAll
module.exports.splitBox = MMCQ.medianCutApply
