import nock from 'nock';
import fetch from 'node-fetch';
import { Buffer } from 'safe-buffer';
import { fetchAsArrayBuffer, fetchAsBlob, fetchAsJson, fetchAsText } from '../../fetch-as/index.js';
import { errorData, successData, url } from './CONSTANTS.js';
function errorNock(uri, data) {
    return nock(uri)
        .persist(true)
        .get(n => /^\/error/i.test(n))
        .reply(404, () => {
        return { ...data };
    });
}
function successNock(uri, data) {
    return nock(uri)
        .persist(true)
        .get(n => /^\/ok/i.test(n))
        .reply(200, () => {
        return { ...data };
    });
}
let nocks;
beforeAll(() => {
    Object.assign(global, { fetch });
    nocks = [
        errorNock(url, errorData),
        successNock(url, successData),
    ];
});
describe('fetch-as::success', () => {
    it(`returns response with 'fetchAsArrayBuffer'`, async () => {
        const d = await fetchAsArrayBuffer(`${url}/ok`);
        expect(d.status).toStrictEqual(200);
        expect(d.data && Buffer.from(d.data)).toStrictEqual(Buffer.from(JSON.stringify({ ...successData })));
    });
    it(`returns response with 'fetchAsBlob'`, async () => {
        const d = await fetchAsBlob(`${url}/ok`);
        expect(d.status).toStrictEqual(200);
        expect(d.data?.size).toStrictEqual(16);
        expect(d.data?.type).toStrictEqual('application/json');
    });
    it(`returns response with 'fetchAsJson'`, async () => {
        const d = await fetchAsJson(`${url}/ok`);
        expect(d.status).toStrictEqual(200);
        expect(d.data).toStrictEqual({ ...successData });
    });
    it(`returns response with 'fetchAsText'`, async () => {
        const d = await fetchAsText(`${url}/ok`);
        expect(d.status).toStrictEqual(200);
        expect(d.data).toStrictEqual(JSON.stringify({ ...successData }));
    });
});
describe('fetch-as::failure', () => {
    it(`returns failed response with 'fetchAsArrayBuffer'`, async () => {
        const d = await fetchAsArrayBuffer(`${url}/error`);
        expect(d.status).toBeGreaterThan(399);
        expect(d.error && Buffer.from(d.error)).toStrictEqual(Buffer.from(JSON.stringify({ ...errorData })));
    });
    it(`returns failed response with 'fetchAsBlob'`, async () => {
        const d = await fetchAsBlob(`${url}/error`);
        expect(d.status).toBeGreaterThan(399);
        expect(d.error?.size).toStrictEqual(42);
        expect(d.error?.type).toStrictEqual('application/json');
    });
    it(`returns failed response with 'fetchAsJson'`, async () => {
        const d = await fetchAsJson(`${url}/error`);
        expect(d.status).toBeGreaterThan(399);
        expect(d.error).toStrictEqual({ ...errorData });
    });
    it(`returns failed response with 'fetchAsText'`, async () => {
        const d = await fetchAsText(`${url}/error`);
        expect(d.status).toBeGreaterThan(399);
        expect(d.error).toStrictEqual(JSON.stringify({ ...errorData }));
    });
    afterAll(() => {
        nocks.forEach(n => n.persist(false));
        nock.cleanAll();
    });
});
//# sourceMappingURL=fetch-as.spec.js.map