import './setup.js';
import { PollingObserver } from '../../polling-observer/index.js';
it(`finishes polling with condition fulfills`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(d => Boolean(d && d.items.length > 0));
    obs.observe(async () => {
        return new Promise((resolve) => {
            setTimeout(() => resolve(data), 2e3);
        });
    }, { interval: 1e3, timeout: 5e3 });
    obs.onfinish = (d) => {
        const { status, value } = d;
        expect(status).toStrictEqual('finish');
        expect(value).toStrictEqual({ ...data });
        done();
    };
}, 10e3);
it(`timeouts a polling`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 7e3));
    }, { interval: 1e3, timeout: 5e3 });
    obs.onfinish = (d) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        done();
    };
}, 10e3);
it(`timeouts a polling with > 1 repeat`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 1e3, timeout: 5e3 });
    obs.onfinish = (d) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        done();
    };
}, 10e3);
it(`reads records when polling finishes`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 1e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(records[0].toJSON()).toMatchObject({
            duration: expect.any(Number),
            entryType: expect.stringMatching('polling-measure'),
            name: expect.stringMatching(/^polling:\d+/gi),
            startTime: expect.any(Number),
        });
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        expect(obs.takeRecords()[0].toJSON()).toMatchObject({
            duration: expect.any(Number),
            entryType: expect.stringMatching('polling-measure'),
            name: expect.stringMatching(/^polling:\d+/gi),
            startTime: expect.any(Number),
        });
        done();
    };
}, 10e3);
it(`clears records when observer disconnects`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 1e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        obs.disconnect();
        expect(obs.takeRecords().length).toBeLessThan(1);
        done();
    };
}, 10e3);
it(`forces polling to stop by disconnecting the observer`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => {
        obs.disconnect();
        return false;
    });
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 2e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('finish');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toStrictEqual(1);
        expect(obs.takeRecords().length).toStrictEqual(0);
        done();
    };
}, 10e3);
it(`disconnects observer before first polling initiates`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 2e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('finish');
        expect(value).toStrictEqual(undefined);
        expect(records).toStrictEqual([]);
        expect(obs.takeRecords()).toStrictEqual([]);
        done();
    };
    obs.disconnect();
}, 10e3);
it(`re-observes after disconnected`, async (done) => {
    const getMockData = () => ({ items: [Math.floor(Math.random() * Math.PI)] });
    const pollingFn = (d) => async () => {
        return new Promise(resolve => setTimeout(() => resolve(d), 1));
    };
    const obs = new PollingObserver(() => false);
    const pollingOpts = { interval: 1e3, timeout: 3e3 };
    await new Promise((resolve) => {
        const mockData = getMockData();
        obs.onfinish = (d, records) => {
            const { status, value } = d;
            expect(status).toStrictEqual('timeout');
            expect(value).toStrictEqual({ ...mockData });
            expect(records.length).toBeGreaterThan(1);
            expect(obs.takeRecords().length).toBeGreaterThan(1);
            resolve();
        };
        obs.observe(pollingFn(mockData), pollingOpts);
    });
    obs.disconnect();
    expect(obs.takeRecords().length).toBeLessThan(1);
    await new Promise((resolve) => {
        const mockData2 = getMockData();
        obs.onfinish = (d, records) => {
            const { status, value } = d;
            expect(status).toStrictEqual('timeout');
            expect(value).toStrictEqual({ ...mockData2 });
            expect(records.length).toBeGreaterThan(1);
            expect(obs.takeRecords().length).toBeGreaterThan(1);
            resolve();
        };
        obs.observe(pollingFn(mockData2), pollingOpts);
    });
    done();
}, 10e3);
it(`polls with optional 'interval'`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(() => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { timeout: 5e3 });
    obs.onfinish = (d) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    };
}, 10e3);
it(`polls with optional 'timeout'`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const startsAt = +new Date();
    const obs = new PollingObserver(() => {
        const endsAt = +new Date();
        return Math.floor(endsAt - startsAt) > 5e3;
    });
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 2e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('finish');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    };
}, 10e3);
it(`polls with optional 'options'`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const startsAt = +new Date();
    const obs = new PollingObserver(() => {
        const endsAt = +new Date();
        return Math.floor(endsAt - startsAt) > 5e3;
    });
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('finish');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    };
}, 10e3);
it(`polls with async 'conditionCallback'`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(async () => false);
    obs.observe(async () => {
        return new Promise(resolve => setTimeout(() => resolve(data), 1));
    }, { interval: 2e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    };
}, 10e3);
it(`polls with non-async 'callback'`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(async () => false);
    obs.observe(() => data, { interval: 2e3, timeout: 5e3 });
    obs.onfinish = (d, records) => {
        const { status, value } = d;
        expect(status).toStrictEqual('timeout');
        expect(value).toStrictEqual({ ...data });
        expect(records.length).toBeGreaterThan(1);
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    };
}, 8e3);
it(`polls without 'onfinish' callback`, (done) => {
    const data = { items: [Math.floor(Math.random() * Math.PI)] };
    const obs = new PollingObserver(async () => false);
    setTimeout(() => {
        expect(obs.takeRecords().length).toBeGreaterThan(1);
        done();
    }, 8e3);
    obs.observe(() => data, { interval: 2e3, timeout: 5e3 });
    expect(obs.takeRecords().length).toBeLessThan(1);
}, 10e3);
//# sourceMappingURL=polling-observer.spec.js.map