/**
 * @typedef NetworkOwnOptions
 * @property {string} [baseDir] - Project base directory for component loading
 * @property {ComponentLoader} [componentLoader] - Component loader instance to use, if any
 * @property {Object} [flowtrace] - Flowtrace instance to use for tracing this network run
 */
/**
 * @typedef { NetworkOwnOptions & import("./ComponentLoader").ComponentLoaderOptions} NetworkOptions
 */
export class BaseNetwork extends EventEmitter {
    /**
     * All NoFlo networks are instantiated with a graph. Upon instantiation
     * they will load all the needed components, instantiate them, and
     * set up the defined connections and IIPs.
     *
     * @param {import("fbp-graph").Graph} graph - Graph definition to build a Network for
     * @param {NetworkOptions} options - Network options
     */
    constructor(graph: import("fbp-graph").Graph, options?: NetworkOptions);
    options: NetworkOptions;
    /** @type {Object<string, NetworkProcess>} */
    processes: {
        [x: string]: NetworkProcess;
    };
    /** @type {Array<internalSocket.InternalSocket>} */
    connections: Array<internalSocket.InternalSocket>;
    /** @type {Array<NetworkIIP>} */
    initials: Array<NetworkIIP>;
    /** @type {Array<NetworkIIP>} */
    nextInitials: Array<NetworkIIP>;
    /** @type {Array<import("./InternalSocket").InternalSocket>} */
    defaults: Array<import("./InternalSocket").InternalSocket>;
    graph: import("fbp-graph/lib/Graph").Graph;
    started: boolean;
    stopped: boolean;
    debug: boolean;
    /** @type {Array<NetworkEvent>} */
    eventBuffer: Array<NetworkEvent>;
    baseDir: any;
    /** @type {Date | null} */
    startupDate: Date | null;
    /** @type {ComponentLoader} */
    loader: ComponentLoader;
    flowtraceName: string;
    /**
     * @returns {number}
     */
    uptime(): number;
    /**
     * @returns {string[]}
     */
    getActiveProcesses(): string[];
    /**
     * @param {string} event
     * @param {any} payload
     * @private
     */
    private traceEvent;
    /**
     * @param {string} event
     * @param {any} payload
     * @protected
     */
    protected bufferedEmit(event: string, payload: any): void;
    /**
     * @callback ComponentLoadCallback
     * @param {Error|null} err
     * @param {import("./Component").Component} [component]
     * @returns {void}
     */
    /**
     * @param {string} component
     * @param {import("fbp-graph/lib/Types").GraphNodeMetadata} metadata
     * @param {ComponentLoadCallback} [callback]
     * @returns {Promise<import("./Component").Component>}
     */
    load(component: string, metadata: import("fbp-graph/lib/Types").GraphNodeMetadata, callback?: (err: Error | null, component?: import("./Component").Component) => void): Promise<import("./Component").Component>;
    /**
     * @callback AddNodeCallback
     * @param {Error|null} error
     * @param {NetworkProcess} [process]
     * @returns {void}
     */
    /**
     * @param {import("fbp-graph/lib/Types").GraphNode} node
     * @param {Object} options
     * @param {AddNodeCallback} [callback]
     * @returns {Promise<NetworkProcess>}
     */
    addNode(node: import("fbp-graph/lib/Types").GraphNode, options: any, callback?: (error: Error | null, process?: NetworkProcess) => void): Promise<NetworkProcess>;
    /**
     * @param {import("fbp-graph/lib/Types").GraphNode} node
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<void>}
     */
    removeNode(node: import("fbp-graph/lib/Types").GraphNode, callback?: (err?: Error | null) => void): Promise<void>;
    /**
     * @param {string} oldId
     * @param {string} newId
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<void>}
     */
    renameNode(oldId: string, newId: string, callback?: (err?: Error | null) => void): Promise<void>;
    /**
     * @param {string} id compone
     * @returns {NetworkProcess|void}
     */
    getNode(id: string): NetworkProcess | void;
    /**
     * @callback ErrorableCallback
     * @param {Error|null} [err]
     * @returns {void}
     */
    /**
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<this>}
     */
    connect(callback?: (err?: Error | null) => void): Promise<this>;
    /**
     * @private
     * @param {NetworkProcess} node
     */
    private subscribeSubgraph;
    /**
     * @param {internalSocket.InternalSocket} socket
     * @param {NetworkProcess} [source]
     */
    subscribeSocket(socket: internalSocket.InternalSocket, source?: NetworkProcess): void;
    /**
     * @param {NetworkProcess} node
     */
    subscribeNode(node: NetworkProcess): void;
    abortDebounce: boolean;
    /**
     * @protected
     * @param {string} node
     * @param {string} direction
     * @returns Promise<NetworkProcess>
     */
    protected ensureNode(node: string, direction: string): Promise<any>;
    /**
     * @callback AddEdgeCallback
     * @param {Error|null} error
     * @param {internalSocket.InternalSocket} [socket]
     * @returns {void}
     */
    /**
     * @param {import("fbp-graph/lib/Types").GraphEdge} edge
     * @param {Object} options
     * @param {AddEdgeCallback} [callback]
     * @returns {Promise<internalSocket.InternalSocket>}
     */
    addEdge(edge: import("fbp-graph/lib/Types").GraphEdge, options: any, callback?: (error: Error | null, socket?: internalSocket.InternalSocket) => void): Promise<internalSocket.InternalSocket>;
    /**
     * @param {import("fbp-graph/lib/Types").GraphEdge} edge
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<void>}
     */
    removeEdge(edge: import("fbp-graph/lib/Types").GraphEdge, callback?: (err?: Error | null) => void): Promise<void>;
    /**
     * @protected
     * @param {import("fbp-graph/lib/Types").GraphNode} node
     * @returns {Promise<void>}
     */
    protected addDefaults(node: import("fbp-graph/lib/Types").GraphNode): Promise<void>;
    /**
     * @param {import("fbp-graph/lib/Types").GraphIIP} initializer
     * @param {Object} options
     * @param {AddEdgeCallback} [callback]
     * @returns {Promise<internalSocket.InternalSocket>}
     */
    addInitial(initializer: import("fbp-graph/lib/Types").GraphIIP, options: any, callback?: (error: Error | null, socket?: internalSocket.InternalSocket) => void): Promise<internalSocket.InternalSocket>;
    /**
     * @param {import("fbp-graph/lib/Types").GraphIIP} initializer
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<void>}
     */
    removeInitial(initializer: import("fbp-graph/lib/Types").GraphIIP, callback?: (err?: Error | null) => void): Promise<void>;
    /**
     * @returns Promise<void>
     */
    sendInitials(): Promise<void>;
    isStarted(): boolean;
    isStopped(): boolean;
    isRunning(): boolean;
    /**
     * @protected
     * @returns {Promise<void>}
     */
    protected startComponents(): Promise<void>;
    /**
     * @returns Promise<void>
     */
    sendDefaults(): Promise<void>;
    /**
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<this>}
     */
    start(callback?: (err?: Error | null) => void): Promise<this>;
    /**
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<this>}
     */
    stop(callback?: (err?: Error | null) => void): Promise<this>;
    /**
     * @param {boolean} started
     */
    setStarted(started: boolean): void;
    checkIfFinished(): void;
    debouncedEnd: Function;
    getDebug(): boolean;
    /**
     * @param {boolean} active
     */
    setDebug(active: boolean): void;
    /**
     * @param {Object|null} flowtrace
     * @param {string|null} [name]
     * @param {boolean} [main]
     */
    setFlowtrace(flowtrace: any | null, name?: string | null, main?: boolean): void;
    flowtrace: any;
}
export type NetworkProcess = {
    id: string;
    componentName?: string;
    component?: import("./Component").Component;
};
export type NetworkIIP = {
    socket: internalSocket.InternalSocket;
    data: any;
};
export type NetworkEvent = {
    type: string;
    payload: any;
};
export type NetworkOwnOptions = {
    /**
     * - Project base directory for component loading
     */
    baseDir?: string;
    /**
     * - Component loader instance to use, if any
     */
    componentLoader?: ComponentLoader;
    /**
     * - Flowtrace instance to use for tracing this network run
     */
    flowtrace?: any;
};
export type NetworkOptions = NetworkOwnOptions & import("./ComponentLoader").ComponentLoaderOptions;
import { EventEmitter } from "events";
import * as internalSocket from "./InternalSocket";
import { ComponentLoader } from "./ComponentLoader";
