/**
 * @callback ProcessingFunction
 * @param {ProcessInput} input
 * @param {ProcessOutput} output
 * @param {ProcessContext} context
 * @returns {Promise<any> | void}
 */
/**
 * @typedef ComponentOptions
 * @property {import("./Ports").InPortsOptions | InPorts} [inPorts] - Inports for the component
 * @property {import("./Ports").OutPortsOptions | OutPorts} [outPorts] - Outports for the component
 * @property {string} [icon]
 * @property {string} [description]
 * @property {ProcessingFunction} [options.process] - Component processsing function
 * @property {boolean} [ordered] - Whether component should send
 * packets in same order it received them
 * @property {boolean} [autoOrdering]
 * @property {boolean} [activateOnInput] - Whether component should
 * activate when it receives packets
 * @property {Object<string, Array<string>>} [forwardBrackets] - Mappings of forwarding ports
 */
/**
 * @typedef BracketContext
 * @property {Object<string,Object>} in
 * @property {Object<string,Object>} out
 */
/** @typedef {{ __resolved?: boolean, __bracketClosingAfter?: BracketContext[], [key: string]: any }} ProcessResult */
export class Component extends EventEmitter {
    /**
     * @param {ComponentOptions} [options]
     */
    constructor(options?: ComponentOptions);
    inPorts: InPorts;
    outPorts: OutPorts;
    icon: string;
    description: string;
    /** @type {string|null} */
    componentName: string | null;
    /** @type {string|null} */
    baseDir: string | null;
    started: boolean;
    load: number;
    ordered: boolean;
    autoOrdering: boolean;
    /** @type {ProcessResult[]} */
    outputQ: ProcessResult[];
    /** @type {BracketContext} */
    bracketContext: BracketContext;
    activateOnInput: boolean;
    forwardBrackets: {
        [x: string]: string[];
    };
    /** @type string | null */
    nodeId: string | null;
    __openConnections: number;
    getDescription(): string;
    isReady(): boolean;
    isSubgraph(): boolean;
    /**
     * @param {string} icon - Updated icon for the component
     */
    setIcon(icon: string): void;
    getIcon(): string;
    /**
     * @param {Error} e
     * @param {Array<string>} [groups]
     * @param {string} [errorPort]
     * @param {string | null} [scope]
     */
    error(e: Error, groups?: Array<string>, errorPort?: string, scope?: string | null): void;
    /**
     * @callback ErrorableCallback
     * @param {Error | null} error
     */
    /**
     * @param {ErrorableCallback} callback - Callback for when teardown is ready
     * @returns {Promise<void> | void}
     */
    setUp(callback: (error: Error | null) => any): Promise<void> | void;
    /**
     * @param {ErrorableCallback} callback - Callback for when teardown is ready
     * @returns {Promise<void> | void}
     */
    tearDown(callback: (error: Error | null) => any): Promise<void> | void;
    /**
     * @param {ErrorableCallback} [callback] - Callback for when shutdown is ready
     * @returns {Promise<void>}
     */
    start(callback?: (error: Error | null) => any): Promise<void>;
    /**
     * @param {ErrorableCallback} [callback] - Callback for when shutdown is ready
     * @returns {Promise<void>}
     */
    shutdown(callback?: (error: Error | null) => any): Promise<void>;
    isStarted(): boolean;
    prepareForwarding(): void;
    isLegacy(): boolean;
    /**
     * @param {ProcessingFunction} handle - Processing function
     * @returns {this}
     */
    process(handle: ProcessingFunction): this;
    handle: ProcessingFunction;
    /**
     * @param {InPort|string} port
     * @returns {boolean}
     */
    isForwardingInport(port: InPort | string): boolean;
    /**
     * @param {InPort|string} inport
     * @param {OutPort|string} outport
     * @returns {boolean}
     */
    isForwardingOutport(inport: InPort | string, outport: OutPort | string): boolean;
    isOrdered(): boolean;
    /**
     * @param {IP} ip
     * @param {InPort} port
     * @returns {void}
     */
    handleIP(ip: IP, port: InPort): void;
    /**
     * @param {string} type
     * @param {string} port
     * @param {string|null} scope
     * @param {number|null} [idx]
     */
    getBracketContext(type: string, port: string, scope: string | null, idx?: number | null): any;
    /**
     * @param {ProcessResult} result
     * @param {Object} port
     * @param {IP} packet
     * @param {boolean} [before]
     */
    addToResult(result: ProcessResult, port: any, packet: IP, before?: boolean): void;
    /** @private */
    private getForwardableContexts;
    /** @private */
    private addBracketForwards;
    /** @private */
    private processOutputQueue;
    /**
     * @param {Object} context
     * @param {boolean} context.activated
     * @param {boolean} context.deactivated
     * @param {Object} context.result
     */
    activate(context: {
        activated: boolean;
        deactivated: boolean;
        result: any;
    }): void;
    /**
     * @param {Object} context
     * @param {boolean} context.activated
     * @param {boolean} context.deactivated
     */
    deactivate(context: {
        activated: boolean;
        deactivated: boolean;
    }): void;
}
export namespace Component {
    const description: string;
    const icon: any;
}
export type ProcessingFunction = (input: ProcessInput, output: ProcessOutput, context: ProcessContext) => Promise<any> | void;
export type ComponentOptions = {
    /**
     * - Inports for the component
     */
    inPorts?: import("./Ports").InPortsOptions | InPorts;
    /**
     * - Outports for the component
     */
    outPorts?: import("./Ports").OutPortsOptions | OutPorts;
    icon?: string;
    description?: string;
    /**
     * - Component processsing function
     */
    process?: ProcessingFunction;
    /**
     * - Whether component should send
     * packets in same order it received them
     */
    ordered?: boolean;
    autoOrdering?: boolean;
    /**
     * - Whether component should
     * activate when it receives packets
     */
    activateOnInput?: boolean;
    /**
     * - Mappings of forwarding ports
     */
    forwardBrackets?: {
        [x: string]: Array<string>;
    };
};
export type BracketContext = {
    in: {
        [x: string]: any;
    };
    out: {
        [x: string]: any;
    };
};
export type ProcessResult = {
    [key: string]: any;
    __resolved?: boolean;
    __bracketClosingAfter?: BracketContext[];
};
import { EventEmitter } from "events";
import { InPorts } from "./Ports";
import { OutPorts } from "./Ports";
import InPort from "./InPort";
import OutPort from "./OutPort";
import IP from "./IP";
import ProcessInput from "./ProcessInput";
import ProcessOutput from "./ProcessOutput";
import ProcessContext from "./ProcessContext";
