/**
 * @callback ComponentFactory
 * @param {import("fbp-graph/lib/Types").GraphNodeMetadata} [metadata]
 * @returns {import("./Component").Component}
 */
/**
 * @typedef {Object} ModuleComponent
 * @property {ComponentFactory} getComponent
 */
/** @typedef {string | ModuleComponent | ComponentFactory | import("fbp-graph").Graph } ComponentDefinition */
/** @typedef {string | ModuleComponent | ComponentFactory } ComponentDefinitionWithoutGraph */
/**
 * @typedef {Object<string, ComponentDefinition>} ComponentList
 */
/**
 * @typedef {Object} ComponentSources
 * @property {string} name
 * @property {string} library
 * @property {string} code
 * @property {string} language
 * @property {string} [tests]
 */
/**
 * @typedef ComponentLoaderOptions
 * @property {boolean} [cache]
 * @property {boolean} [discover]
 * @property {boolean} [recursive]
 * @property {string[]} [runtimes]
 * @property {string} [manifest]
 */
export class ComponentLoader {
    /**
     * @param {string} baseDir
     * @param {ComponentLoaderOptions} [options]
     */
    constructor(baseDir: string, options?: ComponentLoaderOptions);
    baseDir: string;
    options: ComponentLoaderOptions;
    /** @type {ComponentList|null} */
    components: ComponentList | null;
    /** @type {Object<string, string>} */
    libraryIcons: {
        [x: string]: string;
    };
    /** @type {Object<string, Object>} */
    sourcesForComponents: {
        [x: string]: any;
    };
    /** @type {Object<string, string>} */
    specsForComponents: {
        [x: string]: string;
    };
    /** @type {Promise<ComponentList> | null}; */
    processing: Promise<ComponentList> | null;
    ready: boolean;
    /**
     * @param {string} name
     * @returns {string}
     */
    getModulePrefix(name: string): string;
    /**
     * @param {any} [callback] - Legacy callback
     * @returning {Promise<ComponentList>} Promise resolving to list of loaded components
     */
    listComponents(callback?: any): Promise<{
        [x: string]: ComponentDefinition;
    }>;
    /**
     * @param {string} name - Component name
     * @param {import("fbp-graph/lib/Types").GraphNodeMetadata} meta - Node metadata
     * @param {any} [cb] - Legacy callback
     * @returns {Promise<import("./Component").Component>}
     */
    load(name: string, meta: import("fbp-graph/lib/Types").GraphNodeMetadata, cb?: any): Promise<import("./Component").Component>;
    /**
     * @protected
     * @param {string} name
     * @param {ComponentDefinitionWithoutGraph} component
     * @param {import("fbp-graph/lib/Types").GraphNodeMetadata} metadata
     * @returns {Promise<import("./Component").Component>}
     */
    protected createComponent(name: string, component: ComponentDefinitionWithoutGraph, metadata: import("fbp-graph/lib/Types").GraphNodeMetadata): Promise<import("./Component").Component>;
    /**
     * @param {import("fbp-graph").Graph|object|string} cPath
     * @returns {boolean}
     */
    isGraph(cPath: import("fbp-graph").Graph | object | string): boolean;
    /**
     * @protected
     * @param {string} name
     * @param {import("fbp-graph").Graph} component
     * @param {import("fbp-graph/lib/Types").GraphNodeMetadata} metadata
     * @returns {Promise<import("../components/Graph").Graph>}
     */
    protected loadGraph(name: string, component: import("fbp-graph").Graph, metadata: import("fbp-graph/lib/Types").GraphNodeMetadata): Promise<import("../components/Graph").Graph>;
    /**
     * @param {string} name - Icon to set
     * @param {import("./Component").Component} instance
     */
    setIcon(name: string, instance: import("./Component").Component): void;
    /**
     * @param {string} prefix
     * @returns {string|null}
     */
    getLibraryIcon(prefix: string): string | null;
    /**
     * @param {string} prefix
     * @param {string} icon
     */
    setLibraryIcon(prefix: string, icon: string): void;
    /**
     * @param {string} packageId
     * @param {string} name
     * @returns {string}
     */
    normalizeName(packageId: string, name: string): string;
    /**
     * @callback ErrorableCallback
     * @param {Error|null} error
     * @returns {void}
     */
    /**
     * @param {string} packageId
     * @param {string} name
     * @param {ComponentDefinition} cPath
     * @param {ErrorableCallback} [callback]
     */
    registerComponent(packageId: string, name: string, cPath: ComponentDefinition, callback?: (error: Error | null) => void): void;
    /**
     * @param {string} packageId
     * @param {string} name
     * @param {import("fbp-graph").Graph} gPath
     * @param {ErrorableCallback} [callback]
     */
    registerGraph(packageId: string, name: string, gPath: import("fbp-graph").Graph, callback?: (error: Error | null) => void): void;
    /**
     * @callback CustomLoader
     * @param {ComponentLoader} loader
     * @param {ErrorableCallback} callback
     * @returns {void}
     */
    /**
     * @param {CustomLoader} loader
     * @param {ErrorableCallback} callback
     */
    registerLoader(loader: (loader: ComponentLoader, callback: (error: Error | null) => void) => void, callback: (error: Error | null) => void): void;
    /**
     * @param {string} packageId
     * @param {string} name
     * @param {string} source
     * @param {string} language
     * @param {ErrorableCallback} [callback]
     * @returns {Promise<void>}
     */
    setSource(packageId: string, name: string, source: string, language: string, callback?: (error: Error | null) => void): Promise<void>;
    /**
     * @callback SourceCallback
     * @param {Error|null} error
     * @param {ComponentSources} [source]
     */
    /**
     * @param {string} name
     * @param {SourceCallback} [callback]
     * @returns {Promise<ComponentSources>}
     */
    getSource(name: string, callback?: (error: Error | null, source?: ComponentSources) => any): Promise<ComponentSources>;
    getLanguages(): string[];
    clear(): void;
}
export type ComponentFactory = (metadata?: import("fbp-graph/lib/Types").GraphNodeMetadata) => import("./Component").Component;
export type ModuleComponent = {
    getComponent: ComponentFactory;
};
export type ComponentDefinition = string | Graph | ModuleComponent | ComponentFactory;
export type ComponentDefinitionWithoutGraph = string | ModuleComponent | ComponentFactory;
export type ComponentList = {
    [x: string]: ComponentDefinition;
};
export type ComponentSources = {
    name: string;
    library: string;
    code: string;
    language: string;
    tests?: string;
};
export type ComponentLoaderOptions = {
    cache?: boolean;
    discover?: boolean;
    recursive?: boolean;
    runtimes?: string[];
    manifest?: string;
};
import { Graph } from "fbp-graph";
