import { defineUntypedSchema } from 'untyped';
import defu from 'defu';
import { join, resolve } from 'pathe';
import { isDevelopment, isDebug } from 'std-env';
import { findWorkspaceDir } from 'pkg-types';
import createResolver from 'postcss-import-resolver';
import { withoutLeadingSlash } from 'ufo';

const adhoc = defineUntypedSchema({
  components: {
    $resolve: (val) => {
      if (Array.isArray(val)) {
        return { dirs: val };
      }
      if (val === void 0 || val === true) {
        return { dirs: [{ path: "~/components/global", global: true }, "~/components"] };
      }
      return val;
    }
  },
  imports: {
    global: false,
    dirs: []
  },
  pages: void 0,
  telemetry: void 0
});

const app = defineUntypedSchema({
  vue: {
    compilerOptions: {}
  },
  app: {
    baseURL: {
      $resolve: async (val) => val || process.env.NUXT_APP_BASE_URL || "/"
    },
    buildAssetsDir: {
      $resolve: async (val) => val || process.env.NUXT_APP_BUILD_ASSETS_DIR || "/_nuxt/"
    },
    cdnURL: {
      $resolve: async (val, get) => await get("dev") ? "" : (process.env.NUXT_APP_CDN_URL ?? val) || ""
    },
    head: {
      $resolve: async (val, get) => {
        const resolved = defu(val, await get("meta"), {
          meta: [],
          link: [],
          style: [],
          script: [],
          noscript: []
        });
        if (!resolved.meta.find((m) => m.charset)?.charset) {
          resolved.meta.unshift({ charset: resolved.charset || "utf-8" });
        }
        if (!resolved.meta.find((m) => m.name === "viewport")?.content) {
          resolved.meta.unshift({ name: "viewport", content: resolved.viewport || "width=device-width, initial-scale=1" });
        }
        resolved.meta = resolved.meta.filter(Boolean);
        resolved.link = resolved.link.filter(Boolean);
        resolved.style = resolved.style.filter(Boolean);
        resolved.script = resolved.script.filter(Boolean);
        resolved.noscript = resolved.noscript.filter(Boolean);
        return resolved;
      }
    },
    layoutTransition: false,
    pageTransition: false,
    keepalive: false,
    rootId: "__nuxt",
    rootTag: "div"
  },
  plugins: [],
  css: {
    $resolve: (val) => (val ?? []).map((c) => c.src || c)
  }
});

const build = defineUntypedSchema({
  builder: {
    $resolve: async (val, get) => {
      if (typeof val === "object") {
        return val;
      }
      const map = {
        vite: "@nuxt/vite-builder",
        webpack: "@nuxt/webpack-builder"
      };
      return map[val] || val || (await get("vite") === false ? map.webpack : map.vite);
    }
  },
  sourcemap: {
    $resolve: async (val, get) => {
      if (typeof val === "boolean") {
        return { server: val, client: val };
      }
      return defu(val, {
        server: true,
        client: await get("dev")
      });
    }
  },
  build: {
    transpile: {
      $resolve: (val) => [].concat(val).filter(Boolean)
    },
    templates: [],
    analyze: {
      $resolve: async (val, get) => {
        if (val !== true) {
          return val ?? false;
        }
        const rootDir = await get("rootDir");
        return {
          template: "treemap",
          projectRoot: rootDir,
          filename: join(rootDir, ".nuxt/stats", "{name}.html")
        };
      }
    }
  }
});

const common = defineUntypedSchema({
  extends: null,
  theme: null,
  rootDir: {
    $resolve: (val) => typeof val === "string" ? resolve(val) : process.cwd()
  },
  workspaceDir: {
    $resolve: async (val, get) => val ? resolve(await get("rootDir"), val) : await findWorkspaceDir(await get("rootDir")).catch(() => get("rootDir"))
  },
  srcDir: {
    $resolve: async (val, get) => resolve(await get("rootDir"), val || ".")
  },
  serverDir: {
    $resolve: async (val, get) => resolve(await get("rootDir"), val || resolve(await get("srcDir"), "server"))
  },
  buildDir: {
    $resolve: async (val, get) => resolve(await get("rootDir"), val || ".nuxt")
  },
  modulesDir: {
    $default: ["node_modules"],
    $resolve: async (val, get) => [
      ...await Promise.all(val.map(async (dir) => resolve(await get("rootDir"), dir))),
      resolve(process.cwd(), "node_modules")
    ]
  },
  dev: Boolean(isDevelopment),
  test: Boolean(isDevelopment),
  debug: {
    $resolve: async (val, get) => val ?? isDebug
  },
  ssr: {
    $resolve: (val) => val ?? true
  },
  modules: [],
  dir: {
    assets: "assets",
    layouts: "layouts",
    middleware: "middleware",
    pages: "pages",
    plugins: "plugins",
    public: {
      $resolve: async (val, get) => val || await get("dir.static") || "public"
    },
    static: {
      $schema: { deprecated: "use `dir.public` option instead" },
      $resolve: async (val, get) => val || await get("dir.public") || "public"
    }
  },
  extensions: {
    $resolve: (val) => [".js", ".jsx", ".mjs", ".ts", ".tsx", ".vue"].concat(val).filter(Boolean)
  },
  alias: {
    $resolve: async (val, get) => ({
      "~~": await get("rootDir"),
      "@@": await get("rootDir"),
      "~": await get("srcDir"),
      "@": await get("srcDir"),
      [await get("dir.assets")]: join(await get("srcDir"), await get("dir.assets")),
      [await get("dir.public")]: join(await get("srcDir"), await get("dir.public")),
      ...val
    })
  },
  ignoreOptions: void 0,
  ignorePrefix: "-",
  ignore: {
    $resolve: async (val, get) => [
      "**/*.stories.{js,ts,jsx,tsx}",
      "**/*.{spec,test}.{js,ts,jsx,tsx}",
      "**/*.d.ts",
      ".output",
      await get("ignorePrefix") && `**/${await get("ignorePrefix")}*.*`
    ].concat(val).filter(Boolean)
  },
  watchers: {
    rewatchOnRawEvents: void 0,
    webpack: {
      aggregateTimeout: 1e3
    },
    chokidar: {
      ignoreInitial: true
    }
  },
  hooks: null,
  runtimeConfig: {
    $resolve: async (val, get) => defu(val, {
      public: {},
      app: {
        baseURL: (await get("app")).baseURL,
        buildAssetsDir: (await get("app")).buildAssetsDir,
        cdnURL: (await get("app")).cdnURL
      }
    })
  },
  appConfig: {}
});

const dev = defineUntypedSchema({
  devServer: {
    https: false,
    port: process.env.NUXT_PORT || process.env.NITRO_PORT || process.env.PORT || 3e3,
    host: process.env.NUXT_HOST || process.env.NITRO_HOST || process.env.HOST || "localhost",
    url: "http://localhost:3000"
  }
});

const experimental = defineUntypedSchema({
  experimental: {
    asyncEntry: {
      $resolve: (val) => val ?? false
    },
    reactivityTransform: false,
    externalVue: true,
    treeshakeClientOnly: true,
    viteNode: {
      $resolve: (val) => {
        val = process.env.EXPERIMENTAL_VITE_NODE ? true : val;
        if (val === true) {
          console.warn("`vite-node` is now enabled by default. You can safely remove `experimental.viteNode` from your config.");
        } else if (val === false) {
          console.warn("`vite-node` is now enabled by default. To disable it, set `vite.devBundler` to `legacy` instead.");
        }
        return val ?? true;
      }
    },
    viteServerDynamicImports: true,
    inlineSSRStyles: {
      async $resolve(val, get) {
        if (val === false || await get("dev") || await get("ssr") === false || await get("builder") === "@nuxt/webpack-builder") {
          return false;
        }
        return val ?? true;
      }
    },
    noScripts: false,
    payloadExtraction: {
      async $resolve(enabled, get) {
        enabled = enabled ?? false;
        if (enabled) {
          console.warn("Using experimental payload extraction for full-static output. You can opt-out by setting `experimental.payloadExtraction` to `false`.");
        }
        return enabled;
      }
    },
    crossOriginPrefetch: false,
    writeEarlyHints: false
  }
});

const generate = defineUntypedSchema({
  generate: {
    routes: [],
    exclude: []
  }
});

const internal = defineUntypedSchema({
  _majorVersion: 3,
  _legacyGenerate: false,
  _start: false,
  _build: false,
  _generate: false,
  _prepare: false,
  _cli: false,
  _requiredModules: {},
  _nuxtConfigFile: void 0,
  _nuxtConfigFiles: [],
  appDir: "",
  _installedModules: [],
  _modules: []
});

const nitro = defineUntypedSchema({
  nitro: {
    routeRules: {
      $resolve: async (val, get) => ({
        ...await get("routeRules") || {},
        ...val || {}
      })
    }
  },
  routeRules: {},
  serverHandlers: [],
  devServerHandlers: []
});

const postcss = defineUntypedSchema({
  postcss: {
    config: false,
    plugins: {
      "postcss-import": {
        $resolve: async (val, get) => val !== false ? defu(val || {}, {
          resolve: createResolver({
            alias: { ...await get("alias") },
            modules: [
              await get("srcDir"),
              await get("rootDir"),
              ...await get("modulesDir")
            ]
          })
        }) : val
      },
      "postcss-url": {},
      autoprefixer: {},
      cssnano: {
        $resolve: async (val, get) => val ?? !(await get("dev") && {
          preset: ["default", {
            minifyFontValues: { removeQuotes: false }
          }]
        })
      }
    }
  }
});

const router = defineUntypedSchema({
  router: {
    options: {}
  }
});

const typescript = defineUntypedSchema({
  typescript: {
    strict: true,
    includeWorkspace: false,
    typeCheck: false,
    tsConfig: {},
    shim: true
  }
});

const vite = defineUntypedSchema({
  vite: {
    root: {
      $resolve: async (val, get) => val ?? await get("srcDir")
    },
    mode: {
      $resolve: async (val, get) => val ?? (await get("dev") ? "development" : "production")
    },
    logLevel: "warn",
    define: {
      $resolve: async (val, get) => ({
        "process.dev": await get("dev"),
        ...val || {}
      })
    },
    resolve: {
      extensions: [".mjs", ".js", ".ts", ".jsx", ".tsx", ".json", ".vue"]
    },
    publicDir: {
      $resolve: async (val, get) => val ?? resolve(await get("srcDir"), (await get("dir")).public)
    },
    vue: {
      isProduction: {
        $resolve: async (val, get) => val ?? !await get("dev")
      },
      template: {
        compilerOptions: {
          $resolve: async (val, get) => val ?? (await get("vue")).compilerOptions
        }
      }
    },
    optimizeDeps: {
      exclude: {
        $resolve: async (val, get) => [
          ...val || [],
          ...(await get("build.transpile")).filter((i) => typeof i === "string"),
          "vue-demi"
        ]
      }
    },
    esbuild: {
      jsxFactory: "h",
      jsxFragment: "Fragment",
      tsconfigRaw: "{}"
    },
    clearScreen: false,
    build: {
      assetsDir: {
        $resolve: async (val, get) => val ?? withoutLeadingSlash((await get("app")).buildAssetsDir)
      },
      emptyOutDir: false
    },
    server: {
      fs: {
        allow: {
          $resolve: async (val, get) => [
            await get("buildDir"),
            await get("srcDir"),
            await get("rootDir"),
            await get("workspaceDir"),
            ...await get("modulesDir"),
            ...val ?? []
          ]
        }
      }
    }
  }
});

const webpack = defineUntypedSchema({
  webpack: {
    analyze: {
      $resolve: async (val, get) => {
        if (val !== true) {
          return val ?? false;
        }
        const rootDir = await get("rootDir");
        return {
          template: "treemap",
          projectRoot: rootDir,
          filename: join(rootDir, ".nuxt/stats", "{name}.html")
        };
      }
    },
    profile: process.argv.includes("--profile"),
    extractCSS: true,
    cssSourceMap: {
      $resolve: async (val, get) => val ?? await get("dev")
    },
    serverURLPolyfill: "url",
    filenames: {
      app: ({ isDev }) => isDev ? `[name].js` : `[contenthash:7].js`,
      chunk: ({ isDev }) => isDev ? `[name].js` : `[contenthash:7].js`,
      css: ({ isDev }) => isDev ? "[name].css" : "css/[contenthash:7].css",
      img: ({ isDev }) => isDev ? "[path][name].[ext]" : "img/[name].[contenthash:7].[ext]",
      font: ({ isDev }) => isDev ? "[path][name].[ext]" : "fonts/[name].[contenthash:7].[ext]",
      video: ({ isDev }) => isDev ? "[path][name].[ext]" : "videos/[name].[contenthash:7].[ext]"
    },
    loaders: {
      $resolve: async (val, get) => {
        const styleLoaders = [
          "css",
          "cssModules",
          "less",
          "sass",
          "scss",
          "stylus",
          "vueStyle"
        ];
        for (const name of styleLoaders) {
          const loader = val[name];
          if (loader && loader.sourceMap === void 0) {
            loader.sourceMap = Boolean(await get("build.cssSourceMap"));
          }
        }
        return val;
      },
      file: { esModule: false },
      fontUrl: { esModule: false, limit: 1e3 },
      imgUrl: { esModule: false, limit: 1e3 },
      pugPlain: {},
      vue: {
        productionMode: { $resolve: async (val, get) => val ?? !await get("dev") },
        transformAssetUrls: {
          video: "src",
          source: "src",
          object: "src",
          embed: "src"
        },
        compilerOptions: { $resolve: async (val, get) => val ?? await get("vue.compilerOptions") }
      },
      css: {
        importLoaders: 0,
        url: {
          filter: (url, resourcePath) => !url.startsWith("/")
        },
        esModule: false
      },
      cssModules: {
        importLoaders: 0,
        url: {
          filter: (url, resourcePath) => !url.startsWith("/")
        },
        esModule: false,
        modules: {
          localIdentName: "[local]_[hash:base64:5]"
        }
      },
      less: {},
      sass: {
        sassOptions: {
          indentedSyntax: true
        }
      },
      scss: {},
      stylus: {},
      vueStyle: {}
    },
    plugins: [],
    terser: {},
    aggressiveCodeRemoval: false,
    optimizeCSS: {
      $resolve: async (val, get) => val ?? (await get("build.extractCSS") ? {} : false)
    },
    optimization: {
      runtimeChunk: "single",
      minimize: { $resolve: async (val, get) => val ?? !await get("dev") },
      minimizer: void 0,
      splitChunks: {
        chunks: "all",
        automaticNameDelimiter: "/",
        cacheGroups: {}
      }
    },
    postcss: {
      execute: void 0,
      postcssOptions: {
        config: {
          $resolve: async (val, get) => val ?? await get("postcss.config")
        },
        plugins: {
          $resolve: async (val, get) => val ?? await get("postcss.plugins")
        }
      },
      sourceMap: void 0,
      implementation: void 0,
      order: ""
    },
    devMiddleware: {
      stats: "none"
    },
    hotMiddleware: {},
    friendlyErrors: true,
    warningIgnoreFilters: []
  }
});

const index = {
  ...adhoc,
  ...app,
  ...build,
  ...common,
  ...dev,
  ...experimental,
  ...generate,
  ...internal,
  ...nitro,
  ...postcss,
  ...router,
  ...typescript,
  ...vite,
  ...webpack
};

export { index as NuxtConfigSchema };
