#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const find_workspace_root_1 = require("../src/utils/find-workspace-root");
const chalk = require("chalk");
const init_local_1 = require("./init-local");
const output_1 = require("../src/utils/output");
const installation_directory_1 = require("../src/utils/installation-directory");
const semver_1 = require("semver");
const strip_indents_1 = require("../src/utils/strip-indents");
const package_json_1 = require("../src/utils/package-json");
const child_process_1 = require("child_process");
function main() {
    var _a;
    const workspace = (0, find_workspace_root_1.findWorkspaceRoot)(process.cwd());
    // new is a special case because there is no local workspace to load
    if (process.argv[2] === 'new' ||
        process.argv[2] === '_migrate' ||
        process.argv[2] === 'init' ||
        (process.argv[2] === 'graph' && !workspace)) {
        process.env.NX_DAEMON = 'false';
        require('nx/src/command-line/nx-commands').commandsObject.argv;
    }
    else {
        if (workspace && workspace.type === 'nx') {
            require('v8-compile-cache');
        }
        // polyfill rxjs observable to avoid issues with multiple version of Observable installed in node_modules
        // https://twitter.com/BenLesh/status/1192478226385428483?s=20
        if (!Symbol.observable)
            Symbol.observable = Symbol('observable polyfill');
        if (!workspace) {
            output_1.output.log({
                title: `The current directory isn't part of an Nx workspace.`,
                bodyLines: [
                    `To create a workspace run:`,
                    chalk.bold.white(`npx create-nx-workspace@latest <workspace name>`),
                    '',
                    `To add Nx to existing workspace run with a workspace-specific nx.json:`,
                    chalk.bold.white(`npx add-nx-to-monorepo@latest`),
                    '',
                    `To add the default nx.json file run:`,
                    chalk.bold.white(`nx init`),
                ],
            });
            output_1.output.note({
                title: `For more information please visit https://nx.dev/`,
            });
            process.exit(1);
        }
        // Make sure that a local copy of Nx exists in workspace
        let localNx;
        try {
            localNx = resolveNx(workspace);
        }
        catch (_b) {
            localNx = null;
        }
        const isLocalInstall = localNx === resolveNx(null);
        const LOCAL_NX_VERSION = localNx
            ? getLocalNxVersion(workspace)
            : null;
        const GLOBAL_NX_VERSION = isLocalInstall
            ? null
            : require('../package.json').version;
        (_a = globalThis.GLOBAL_NX_VERSION) !== null && _a !== void 0 ? _a : (globalThis.GLOBAL_NX_VERSION = GLOBAL_NX_VERSION);
        if (process.argv[2] === '--version') {
            console.log((0, strip_indents_1.stripIndents) `Nx Version:
      - Local: v${LOCAL_NX_VERSION !== null && LOCAL_NX_VERSION !== void 0 ? LOCAL_NX_VERSION : 'Not found'}
      - Global: v${GLOBAL_NX_VERSION !== null && GLOBAL_NX_VERSION !== void 0 ? GLOBAL_NX_VERSION : 'Not found'}`);
            process.exit(0);
        }
        if (!localNx) {
            handleMissingLocalInstallation();
        }
        // this file is already in the local workspace
        if (isLocalInstall) {
            (0, init_local_1.initLocal)(workspace);
        }
        else {
            // Nx is being run from globally installed CLI - hand off to the local
            warnIfUsingOutdatedGlobalInstall(GLOBAL_NX_VERSION, LOCAL_NX_VERSION);
            if (localNx.includes('.nx')) {
                const nxWrapperPath = localNx.replace(/\.nx.*/, '.nx/') + 'nxw.js';
                require(nxWrapperPath);
            }
            else {
                require(localNx);
            }
        }
    }
}
function resolveNx(workspace) {
    // prefer Nx installed in .nx/installation
    try {
        return require.resolve('nx/bin/nx.js', {
            paths: workspace ? [(0, installation_directory_1.getNxInstallationPath)(workspace.dir)] : undefined,
        });
    }
    catch (_a) { }
    // check for root install
    try {
        return require.resolve('nx/bin/nx.js', {
            paths: workspace ? [workspace.dir] : undefined,
        });
    }
    catch (_b) {
        // fallback for old CLI install setup
        return require.resolve('@nrwl/cli/bin/nx.js', {
            paths: workspace ? [workspace.dir] : undefined,
        });
    }
}
function handleMissingLocalInstallation() {
    output_1.output.error({
        title: `Could not find Nx modules in this workspace.`,
        bodyLines: [`Have you run ${chalk.bold.white(`npm/yarn install`)}?`],
    });
    process.exit(1);
}
/**
 * Assumes currently running Nx is global install.
 * Warns if out of date by 1 major version or more.
 */
function warnIfUsingOutdatedGlobalInstall(globalNxVersion, localNxVersion) {
    // Never display this warning if Nx is already running via Nx
    if (process.env.NX_CLI_SET) {
        return;
    }
    const isOutdatedGlobalInstall = globalNxVersion &&
        ((localNxVersion && (0, semver_1.major)(globalNxVersion) < (0, semver_1.major)(localNxVersion)) ||
            (!localNxVersion &&
                getLatestVersionOfNx() &&
                (0, semver_1.major)(globalNxVersion) < (0, semver_1.major)(getLatestVersionOfNx())));
    // Using a global Nx Install
    if (isOutdatedGlobalInstall) {
        const bodyLines = localNxVersion
            ? [
                `Your repository uses a higher version of Nx (${localNxVersion}) than your global CLI version (${globalNxVersion})`,
            ]
            : [];
        bodyLines.push('For more information, see https://nx.dev/more-concepts/global-nx');
        output_1.output.warn({
            title: `Its time to update Nx 🎉`,
            bodyLines,
        });
    }
}
function getLocalNxVersion(workspace) {
    const localNxPackages = ['nx', '@nrwl/tao', '@nrwl/cli'];
    for (const pkg of localNxPackages) {
        try {
            const { packageJson } = (0, package_json_1.readModulePackageJson)('nx', (0, installation_directory_1.getNxRequirePaths)(workspace.dir));
            return packageJson.version;
        }
        catch (_a) { }
    }
}
function _getLatestVersionOfNx() {
    try {
        return (0, child_process_1.execSync)('npm view nx@latest version').toString().trim();
    }
    catch (_a) {
        try {
            return (0, child_process_1.execSync)('pnpm view nx@latest version').toString().trim();
        }
        catch (_b) {
            return null;
        }
    }
}
const getLatestVersionOfNx = ((fn) => {
    let cache = null;
    return () => cache || (cache = fn());
})(_getLatestVersionOfNx);
main();
//# sourceMappingURL=nx.js.map