"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.projectsToRun = exports.runMany = void 0;
const tslib_1 = require("tslib");
const run_command_1 = require("../tasks-runner/run-command");
const command_line_utils_1 = require("../utils/command-line-utils");
const project_graph_utils_1 = require("../utils/project-graph-utils");
const connect_1 = require("./connect");
const perf_hooks_1 = require("perf_hooks");
const project_graph_1 = require("../project-graph/project-graph");
const configuration_1 = require("../config/configuration");
const output_1 = require("../utils/output");
const find_matching_projects_1 = require("../utils/find-matching-projects");
const workspace_configuration_check_1 = require("../utils/workspace-configuration-check");
const dep_graph_1 = require("./dep-graph");
function runMany(args, extraTargetDependencies = {}, extraOptions = { excludeTaskDependencies: false, loadDotEnvFiles: true }) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        perf_hooks_1.performance.mark('command-execution-begins');
        (0, workspace_configuration_check_1.workspaceConfigurationCheck)();
        const nxJson = (0, configuration_1.readNxJson)();
        const { nxArgs, overrides } = (0, command_line_utils_1.splitArgsIntoNxArgsAndOverrides)(args, 'run-many', { printWarnings: true }, nxJson);
        if (nxArgs.verbose) {
            process.env.NX_VERBOSE_LOGGING = 'true';
        }
        yield (0, connect_1.connectToNxCloudIfExplicitlyAsked)(nxArgs);
        const projectGraph = yield (0, project_graph_1.createProjectGraphAsync)({ exitOnError: true });
        const projects = projectsToRun(nxArgs, projectGraph);
        if (nxArgs.graph) {
            const projectNames = projects.map((t) => t.name);
            return yield (0, dep_graph_1.generateGraph)({
                watch: false,
                open: true,
                view: 'tasks',
                all: nxArgs.all,
                targets: nxArgs.targets,
                projects: projectNames,
            }, projectNames);
        }
        else {
            yield (0, run_command_1.runCommand)(projects, projectGraph, { nxJson }, nxArgs, overrides, null, extraTargetDependencies, extraOptions);
        }
    });
}
exports.runMany = runMany;
function projectsToRun(nxArgs, projectGraph) {
    var _a;
    const selectedProjects = new Map();
    const validProjects = runnableForTarget(projectGraph.nodes, nxArgs.targets);
    const invalidProjects = [];
    // --all is default now, if --projects is provided, it'll override the --all
    if (nxArgs.all && nxArgs.projects.length === 0) {
        for (const projectName of validProjects) {
            selectedProjects.set(projectName, projectGraph.nodes[projectName]);
        }
    }
    else {
        const allProjectNames = Object.keys(projectGraph.nodes);
        const matchingProjects = (0, find_matching_projects_1.findMatchingProjects)(nxArgs.projects, allProjectNames, new Set(allProjectNames));
        for (const project of matchingProjects) {
            if (!validProjects.has(project)) {
                invalidProjects.push(project);
            }
            else {
                selectedProjects.set(project, projectGraph.nodes[project]);
            }
        }
        if (invalidProjects.length > 0) {
            output_1.output.warn({
                title: `The following projects do not have a configuration for any of the provided targets ("${nxArgs.targets.join(', ')}")`,
                bodyLines: invalidProjects.map((name) => `- ${name}`),
            });
        }
    }
    const excludedProjects = (0, find_matching_projects_1.findMatchingProjects)((_a = nxArgs.exclude) !== null && _a !== void 0 ? _a : [], Array.from(selectedProjects.keys()), new Set(selectedProjects.keys()));
    for (const excludedProject of excludedProjects) {
        const project = selectedProjects.has(excludedProject);
        if (project) {
            selectedProjects.delete(excludedProject);
        }
    }
    return Array.from(selectedProjects.values());
}
exports.projectsToRun = projectsToRun;
function runnableForTarget(projects, targets) {
    const runnable = new Set();
    for (let projectName in projects) {
        const project = projects[projectName];
        if (targets.find((target) => (0, project_graph_utils_1.projectHasTarget)(project, target))) {
            runnable.add(projectName);
        }
    }
    return runnable;
}
//# sourceMappingURL=run-many.js.map