"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addNxToMonorepo = void 0;
const tslib_1 = require("tslib");
const enquirer_1 = require("enquirer");
const fs_1 = require("fs");
const ignore_1 = require("ignore");
const path_1 = require("path");
const yargsParser = require("yargs-parser");
const fileutils_1 = require("../utils/fileutils");
const output_1 = require("../utils/output");
const package_manager_1 = require("../utils/package-manager");
const path_2 = require("../utils/path");
const utils_1 = require("./utils");
const parsedArgs = yargsParser(process.argv, {
    boolean: ['yes'],
    string: ['cacheable'],
    alias: {
        yes: ['y'],
    },
});
function addNxToMonorepo() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const repoRoot = process.cwd();
        if (!(0, fs_1.existsSync)((0, path_2.joinPathFragments)(repoRoot, 'package.json'))) {
            output_1.output.error({
                title: `Run the command in the folder with a package.json file.`,
            });
            process.exit(1);
        }
        output_1.output.log({ title: `🐳 Nx initialization` });
        const packageJsonFiles = allProjectPackageJsonFiles(repoRoot);
        const scripts = combineAllScriptNames(repoRoot, packageJsonFiles);
        let targetDefaults;
        let cacheableOperations;
        let scriptOutputs = {};
        let useCloud;
        if (parsedArgs.yes !== true && scripts.length > 0) {
            output_1.output.log({
                title: `🧑‍🔧 Please answer the following questions about the scripts found in your workspace in order to generate task runner configuration`,
            });
            targetDefaults = (yield (0, enquirer_1.prompt)([
                {
                    type: 'multiselect',
                    name: 'targetDefaults',
                    message: `Which scripts need to be run in order?  (e.g. before building a project, dependent projects must be built.)`,
                    choices: scripts,
                },
            ])).targetDefaults;
            cacheableOperations = (yield (0, enquirer_1.prompt)([
                {
                    type: 'multiselect',
                    name: 'cacheableOperations',
                    message: 'Which scripts are cacheable? (Produce the same output given the same input, e.g. build, test and lint usually are, serve and start are not)',
                    choices: scripts,
                },
            ])).cacheableOperations;
            for (const scriptName of cacheableOperations) {
                scriptOutputs[scriptName] = (yield (0, enquirer_1.prompt)([
                    {
                        type: 'input',
                        name: scriptName,
                        message: `Does the "${scriptName}" script create any outputs? If not, leave blank, otherwise provide a path relative to a project root (e.g. dist, lib, build, coverage)`,
                    },
                ]))[scriptName];
            }
            useCloud = yield (0, utils_1.askAboutNxCloud)();
        }
        else {
            targetDefaults = [];
            cacheableOperations = parsedArgs.cacheable
                ? parsedArgs.cacheable.split(',')
                : [];
            useCloud = false;
        }
        (0, utils_1.createNxJsonFile)(repoRoot, targetDefaults, cacheableOperations, scriptOutputs);
        (0, utils_1.addDepsToPackageJson)(repoRoot, useCloud);
        output_1.output.log({ title: `📦 Installing dependencies` });
        (0, utils_1.runInstall)(repoRoot);
        if (useCloud) {
            (0, utils_1.initCloud)(repoRoot, 'nx-init-monorepo');
        }
        printFinalMessage();
    });
}
exports.addNxToMonorepo = addNxToMonorepo;
// scanning package.json files
function allProjectPackageJsonFiles(repoRoot) {
    const packageJsonFiles = allPackageJsonFiles(repoRoot, repoRoot);
    return packageJsonFiles.filter((c) => c != 'package.json');
}
function allPackageJsonFiles(repoRoot, dirName) {
    const ignoredGlobs = getIgnoredGlobs(repoRoot);
    const relDirName = (0, path_1.relative)(repoRoot, dirName);
    if (relDirName &&
        (ignoredGlobs.ignores(relDirName) ||
            relDirName.indexOf(`node_modules`) > -1)) {
        return [];
    }
    let res = [];
    try {
        (0, fs_1.readdirSync)(dirName).forEach((c) => {
            const child = (0, path_1.join)(dirName, c);
            if (ignoredGlobs.ignores((0, path_1.relative)(repoRoot, child))) {
                return;
            }
            try {
                const s = (0, fs_1.statSync)(child);
                if (s.isFile() && c == 'package.json') {
                    res.push((0, path_1.relative)(repoRoot, child));
                }
                else if (s.isDirectory()) {
                    res = [...res, ...allPackageJsonFiles(repoRoot, child)];
                }
            }
            catch (_a) { }
        });
    }
    catch (_a) { }
    return res;
}
function getIgnoredGlobs(repoRoot) {
    const ig = (0, ignore_1.default)();
    try {
        ig.add((0, fs_1.readFileSync)(`${repoRoot}/.gitignore`, 'utf-8'));
    }
    catch (_a) { }
    return ig;
}
function combineAllScriptNames(repoRoot, packageJsonFiles) {
    const res = new Set();
    packageJsonFiles.forEach((p) => {
        const packageJson = (0, fileutils_1.readJsonFile)((0, path_1.join)(repoRoot, p));
        Object.keys(packageJson.scripts || {}).forEach((scriptName) => res.add(scriptName));
    });
    return [...res];
}
function printFinalMessage() {
    const pmc = (0, package_manager_1.getPackageManagerCommand)();
    output_1.output.success({
        title: `🎉 Done!`,
        bodyLines: [
            `- Enabled computation caching!`,
            `- Run "${pmc.exec} nx run-many --target=build" to run the build script for every project in the monorepo.`,
            `- Run it again to replay the cached computation.`,
            `- Run "${pmc.exec} nx graph" to see the structure of the monorepo.`,
            `- Learn more at https://nx.dev/recipes/adopting-nx/adding-to-monorepo`,
        ],
    });
}
//# sourceMappingURL=add-nx-to-monorepo.js.map