"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLegacyMigrationFunctionIfApplicable = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const path_1 = require("path");
const semver_1 = require("semver");
const fileutils_1 = require("../../utils/fileutils");
const object_sort_1 = require("../../utils/object-sort");
const output_1 = require("../../utils/output");
const package_json_1 = require("../../utils/package-json");
const package_manager_1 = require("../../utils/package-manager");
const utils_1 = require("../utils");
// map of Angular major versions to Nx versions to use for legacy `nx init` migrations,
// key is major Angular version and value is Nx version to use
const nxAngularLegacyVersionMap = {};
// min major angular version supported in latest Nx
const minMajorAngularVersionSupported = 14;
// version when the Nx CLI changed from @nrwl/tao & @nrwl/cli to nx
const versionWithConsolidatedPackages = '13.9.0';
function getLegacyMigrationFunctionIfApplicable(repoRoot, interactive) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const angularVersion = (0, package_json_1.readModulePackageJson)('@angular/core').packageJson.version;
        const majorAngularVersion = (0, semver_1.major)(angularVersion);
        if (majorAngularVersion >= minMajorAngularVersionSupported) {
            // non-legacy
            return null;
        }
        let legacyMigrationCommand;
        let pkgName;
        let pkgVersion;
        if (majorAngularVersion < 13) {
            // for versions lower than 13, the migration was in @nrwl/workspace:ng-add
            pkgName = '@nrwl/workspace';
            pkgVersion = yield resolvePackageVersion(pkgName, `^${majorAngularVersion}.0.0`);
            legacyMigrationCommand = `ng g ${pkgName}:ng-add --preserveAngularCLILayout`;
        }
        else if (majorAngularVersion < 14) {
            // for v13, the migration was in @nrwl/angular:ng-add
            pkgName = '@nrwl/angular';
            pkgVersion = yield resolvePackageVersion(pkgName, '~14.1.0');
            legacyMigrationCommand = `ng g ${pkgName}:ng-add --preserve-angular-cli-layout`;
        }
        else {
            // use the latest Nx version that supported the Angular version
            pkgName = '@nrwl/angular';
            pkgVersion = yield resolvePackageVersion('nx', nxAngularLegacyVersionMap[majorAngularVersion]);
            legacyMigrationCommand = `nx@${pkgVersion} init ${process.argv
                .slice(2)
                .join(' ')}`;
        }
        return () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            output_1.output.log({ title: '🐳 Nx initialization' });
            const useNxCloud = interactive ? yield (0, utils_1.askAboutNxCloud)() : false;
            output_1.output.log({ title: '📦 Installing dependencies' });
            const pmc = (0, package_manager_1.getPackageManagerCommand)();
            yield installDependencies(repoRoot, pkgName, pkgVersion, useNxCloud, pmc);
            output_1.output.log({ title: '📝 Setting up workspace' });
            (0, child_process_1.execSync)(`${pmc.exec} ${legacyMigrationCommand}`, { stdio: [0, 1, 2] });
            if (useNxCloud) {
                output_1.output.log({ title: '🛠️ Setting up Nx Cloud' });
                (0, utils_1.initCloud)(repoRoot, 'nx-init-angular');
            }
            output_1.output.success({
                title: '🎉 Nx is now enabled in your workspace!',
                bodyLines: [
                    `Execute 'npx nx build' twice to see the computation caching in action.`,
                    'Learn more about the changes done to your workspace at https://nx.dev/recipes/adopting-nx/migration-angular.',
                ],
            });
        });
    });
}
exports.getLegacyMigrationFunctionIfApplicable = getLegacyMigrationFunctionIfApplicable;
function installDependencies(repoRoot, pkgName, pkgVersion, useNxCloud, pmc) {
    var _a, _b;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const json = (0, fileutils_1.readJsonFile)((0, path_1.join)(repoRoot, 'package.json'));
        (_a = json.devDependencies) !== null && _a !== void 0 ? _a : (json.devDependencies = {});
        json.devDependencies['@nrwl/workspace'] = pkgVersion;
        if ((0, semver_1.gte)(pkgVersion, versionWithConsolidatedPackages)) {
            json.devDependencies['nx'] = pkgVersion;
        }
        else {
            json.devDependencies['@nrwl/cli'] = pkgVersion;
            json.devDependencies['@nrwl/tao'] = pkgVersion;
        }
        if (useNxCloud) {
            // get the latest @nrwl/nx-cloud version compatible with the Nx major
            // version being installed
            json.devDependencies['@nrwl/nx-cloud'] = yield resolvePackageVersion('@nrwl/nx-cloud', `^${(0, semver_1.major)(pkgVersion)}.0.0`);
        }
        json.devDependencies = (0, object_sort_1.sortObjectByKeys)(json.devDependencies);
        if (pkgName === '@nrwl/angular') {
            (_b = json.dependencies) !== null && _b !== void 0 ? _b : (json.dependencies = {});
            json.dependencies['@nrwl/angular'] = pkgVersion;
            json.dependencies = (0, object_sort_1.sortObjectByKeys)(json.dependencies);
        }
        (0, fileutils_1.writeJsonFile)(`package.json`, json);
        (0, child_process_1.execSync)(pmc.install, { stdio: [0, 1, 2] });
    });
}
function resolvePackageVersion(packageName, version) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            return yield (0, package_manager_1.resolvePackageVersionUsingRegistry)(packageName, version);
        }
        catch (_a) {
            return yield (0, package_manager_1.resolvePackageVersionUsingInstallation)(packageName, version);
        }
    });
}
//# sourceMappingURL=legacy-angular-versions.js.map