"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sanitizeWrapperScript = exports.getNxWrapperContents = exports.updateGitIgnore = exports.writeMinimalNxJson = exports.generateDotNxSetup = exports.nxWrapperPath = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const path = require("path");
const semver_1 = require("semver");
const tree_1 = require("../../generators/tree");
const json_1 = require("../../generators/utils/json");
const nxWrapperPath = (p = path) => p.join('.nx', 'nxw.js');
exports.nxWrapperPath = nxWrapperPath;
const NODE_MISSING_ERR = 'Nx requires NodeJS to be available. To install NodeJS and NPM, see: https://nodejs.org/en/download/ .';
const NPM_MISSING_ERR = 'Nx requires npm to be available. To install NodeJS and NPM, see: https://nodejs.org/en/download/ .';
const BATCH_SCRIPT_CONTENTS = `set path_to_root=%~dp0
WHERE node >nul 2>nul
IF %ERRORLEVEL% NEQ 0 (ECHO ${NODE_MISSING_ERR}; EXIT 1)
WHERE npm >nul 2>nul
IF %ERRORLEVEL% NEQ 0 (ECHO ${NPM_MISSING_ERR}; EXIT 1)
node ${path.win32.join('%path_to_root%', (0, exports.nxWrapperPath)(path.win32))} %*`;
const SHELL_SCRIPT_CONTENTS = `#!/bin/bash
command -v node >/dev/null 2>&1 || { echo >&2 "${NODE_MISSING_ERR}"; exit 1; }
command -v npm >/dev/null 2>&1 || { echo >&2 "${NPM_MISSING_ERR}"; exit 1; }
path_to_root=$(dirname $BASH_SOURCE)
node ${path.posix.join('$path_to_root', (0, exports.nxWrapperPath)(path.posix))} $@`;
function generateDotNxSetup(version) {
    const host = new tree_1.FsTree(process.cwd(), false);
    writeMinimalNxJson(host, version);
    updateGitIgnore(host);
    host.write((0, exports.nxWrapperPath)(), getNxWrapperContents());
    host.write('nx.bat', BATCH_SCRIPT_CONTENTS);
    host.write('nx', SHELL_SCRIPT_CONTENTS, {
        mode: fs_1.constants.S_IXUSR | fs_1.constants.S_IRUSR | fs_1.constants.S_IWUSR,
    });
    const changes = host.listChanges();
    (0, tree_1.printChanges)(changes);
    (0, tree_1.flushChanges)(host.root, changes);
}
exports.generateDotNxSetup = generateDotNxSetup;
function writeMinimalNxJson(host, version) {
    if (!host.exists('nx.json')) {
        if (!(0, semver_1.valid)(version)) {
            version = (0, child_process_1.execSync)(`npm view nx@${version} version`).toString();
        }
        (0, json_1.writeJson)(host, 'nx.json', {
            tasksRunnerOptions: {
                default: {
                    runner: 'nx/tasks-runners/default',
                    options: {
                        cacheableOperations: [],
                    },
                },
            },
            installation: {
                version: version.trimEnd(),
            },
        });
    }
}
exports.writeMinimalNxJson = writeMinimalNxJson;
function updateGitIgnore(host) {
    var _a;
    const contents = (_a = host.read('.gitignore', 'utf-8')) !== null && _a !== void 0 ? _a : '';
    host.write('.gitignore', [contents, '.nx/installation', '.nx/cache'].join('\n'));
}
exports.updateGitIgnore = updateGitIgnore;
// Gets the sanitized contents for nxw.js
function getNxWrapperContents() {
    return sanitizeWrapperScript((0, fs_1.readFileSync)(path.join(__dirname, 'nxw.js'), 'utf-8'));
}
exports.getNxWrapperContents = getNxWrapperContents;
// Remove any empty comments or comments that start with `//#: ` or eslint-disable comments.
// This removes the sourceMapUrl since it is invalid, as well as any internal comments.
function sanitizeWrapperScript(input) {
    const linesToRemove = [
        // Comments that start with //#
        '\\/\\/# .*',
        // Comments that are empty (often used for newlines between internal comments)
        '\\s*\\/\\/\\s*',
        // Comments that disable an eslint rule.
        '\\/\\/ eslint-disable-next-line.*',
    ];
    const regex = `(${linesToRemove.join('|')})$`;
    return input.replace(new RegExp(regex, 'gm'), '');
}
exports.sanitizeWrapperScript = sanitizeWrapperScript;
//# sourceMappingURL=add-nx-scripts.js.map