"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addNxToCraRepo = void 0;
const tslib_1 = require("tslib");
const child_process_1 = require("child_process");
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const yargsParser = require("yargs-parser");
const fileutils_1 = require("../../utils/fileutils");
const output_1 = require("../../utils/output");
const package_manager_1 = require("../../utils/package-manager");
const check_for_custom_webpack_setup_1 = require("./check-for-custom-webpack-setup");
const check_for_uncommitted_changes_1 = require("./check-for-uncommitted-changes");
const clean_up_files_1 = require("./clean-up-files");
const read_name_from_package_json_1 = require("./read-name-from-package-json");
const rename_js_to_jsx_1 = require("./rename-js-to-jsx");
const setup_e2e_project_1 = require("./setup-e2e-project");
const tsconfig_setup_1 = require("./tsconfig-setup");
const write_craco_config_1 = require("./write-craco-config");
const write_vite_config_1 = require("./write-vite-config");
const write_vite_index_html_1 = require("./write-vite-index-html");
const parsedArgs = yargsParser(process.argv, {
    boolean: ['force', 'e2e', 'nxCloud', 'vite', 'integrated'],
    default: {
        nxCloud: true,
        vite: true,
    },
    configuration: {
        'strip-dashed': true,
    },
});
function addNxToCraRepo() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!parsedArgs.force) {
            (0, check_for_uncommitted_changes_1.checkForUncommittedChanges)();
            (0, check_for_custom_webpack_setup_1.checkForCustomWebpackSetup)();
        }
        output_1.output.log({ title: '✨ Nx initialization' });
        const normalizedOptions = normalizeOptions(parsedArgs);
        yield reorgnizeWorkspaceStructure(normalizedOptions);
    });
}
exports.addNxToCraRepo = addNxToCraRepo;
function addDependencies(pmc, ...deps) {
    const depsArg = deps.join(' ');
    output_1.output.log({ title: `📦 Adding dependencies: ${depsArg}` });
    (0, child_process_1.execSync)(`${pmc.addDev} ${depsArg}`, { stdio: [0, 1, 2] });
}
function normalizeOptions(options) {
    const packageManager = (0, package_manager_1.detectPackageManager)();
    const pmc = (0, package_manager_1.getPackageManagerCommand)(packageManager);
    const appIsJs = !(0, fileutils_1.fileExists)(`tsconfig.json`);
    const reactAppName = (0, read_name_from_package_json_1.readNameFromPackageJson)();
    const packageJson = (0, fileutils_1.readJsonFile)('package.json');
    const deps = Object.assign(Object.assign({}, packageJson.dependencies), packageJson.devDependencies);
    const isCRA5 = /^[^~]?5/.test(deps['react-scripts']);
    const npmVersion = (0, child_process_1.execSync)('npm -v').toString();
    // Should remove this check 04/2023 once Node 14 & npm 6 reach EOL
    const npxYesFlagNeeded = !npmVersion.startsWith('6'); // npm 7 added -y flag to npx
    const isVite = options.vite;
    const isStandalone = !options.integrated;
    return Object.assign(Object.assign({}, options), { packageManager,
        pmc,
        appIsJs,
        reactAppName,
        isCRA5,
        npxYesFlagNeeded,
        isVite,
        isStandalone });
}
function reorgnizeWorkspaceStructure(options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        createTempWorkspace(options);
        moveFilesToTempWorkspace(options);
        yield addBundler(options);
        output_1.output.log({ title: '🧶  Updating .gitignore file' });
        (0, child_process_1.execSync)(`echo "node_modules" >> .gitignore`, { stdio: [0, 1, 2] });
        (0, child_process_1.execSync)(`echo "dist" >> .gitignore`, { stdio: [0, 1, 2] });
        process.chdir('..');
        copyFromTempWorkspaceToRoot();
        cleanUpUnusedFilesAndAddConfigFiles(options);
        output_1.output.log({ title: '🙂 Please be patient, one final step remaining!' });
        output_1.output.log({
            title: '🧶  Adding npm packages to your new Nx workspace',
        });
        addDependencies(options.pmc, '@testing-library/jest-dom', 'eslint-config-react-app', 'web-vitals', 'jest-watch-typeahead');
        if (options.isVite) {
            addDependencies(options.pmc, 'vite', 'vitest', '@vitejs/plugin-react');
        }
        else {
            addDependencies(options.pmc, '@craco/craco', 'cross-env', 'react-scripts', 'tsconfig-paths-webpack-plugin');
        }
        output_1.output.log({ title: '🎉 Done!' });
        output_1.output.note({
            title: 'First time using Nx? Check out this interactive Nx tutorial.',
            bodyLines: [
                `https://nx.dev/react-tutorial/1-code-generation`,
                ` `,
                `Prefer watching videos? Check out this free Nx course on Egghead.io.`,
                `https://egghead.io/playlists/scale-react-development-with-nx-4038`,
            ],
        });
        if (options.isVite) {
            const indexPath = options.isStandalone
                ? 'index.html'
                : (0, path_1.join)('apps', options.reactAppName, 'index.html');
            const oldIndexPath = options.isStandalone
                ? (0, path_1.join)('public', 'index.html')
                : (0, path_1.join)('apps', options.reactAppName, 'public', 'index.html');
            output_1.output.note({
                title: `A new ${indexPath} has been created. Compare it to the previous ${oldIndexPath} file and make any changes needed, then delete the previous file.`,
            });
        }
        output_1.output.note({
            title: 'Or, you can try the commands!',
            bodyLines: [
                options.integrated ? `npx nx serve ${options.reactAppName}` : 'npm start',
                options.integrated
                    ? `npx nx build ${options.reactAppName}`
                    : 'npm run build',
                options.integrated ? `npx nx test ${options.reactAppName}` : `npm test`,
                ` `,
                `https://nx.dev/getting-started/intro#10-try-the-commands`,
            ],
        });
    });
}
function createTempWorkspace(options) {
    (0, child_process_1.execSync)(`npx ${options.npxYesFlagNeeded ? '-y' : ''} create-nx-workspace@latest temp-workspace --appName=${options.reactAppName} --preset=react-monorepo --style=css --bundler=${options.isVite ? 'vite' : 'webpack'} --packageManager=${options.packageManager} ${options.nxCloud ? '--nxCloud' : '--nxCloud=false'}`, { stdio: [0, 1, 2] });
    output_1.output.log({ title: '👋 Welcome to Nx!' });
    output_1.output.log({ title: '🧹 Clearing unused files' });
    (0, fs_extra_1.copySync)((0, path_1.join)('temp-workspace', 'apps', options.reactAppName, 'project.json'), 'project.json');
    (0, fs_extra_1.removeSync)((0, path_1.join)('temp-workspace', 'apps', options.reactAppName));
    (0, fs_extra_1.removeSync)('node_modules');
}
function moveFilesToTempWorkspace(options) {
    output_1.output.log({ title: '🚚 Moving your React app in your new Nx workspace' });
    const requiredCraFiles = [
        'project.json',
        options.isStandalone ? null : 'package.json',
        'src',
        'public',
        options.appIsJs ? null : 'tsconfig.json',
        options.packageManager === 'yarn' ? 'yarn.lock' : null,
        options.packageManager === 'pnpm' ? 'pnpm-lock.yaml' : null,
        options.packageManager === 'npm' ? 'package-lock.json' : null,
    ];
    const optionalCraFiles = ['README.md'];
    const filesToMove = [...requiredCraFiles, ...optionalCraFiles].filter(Boolean);
    filesToMove.forEach((f) => {
        try {
            (0, fs_extra_1.moveSync)(f, options.isStandalone
                ? (0, path_1.join)('temp-workspace', f)
                : (0, path_1.join)('temp-workspace', 'apps', options.reactAppName, f), {
                overwrite: true,
            });
        }
        catch (error) {
            if (requiredCraFiles.includes(f)) {
                throw error;
            }
        }
    });
    process.chdir('temp-workspace');
}
function addBundler(options) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (options.isVite) {
            output_1.output.log({ title: '🧑‍🔧  Setting up Vite' });
            const { addViteCommandsToPackageScripts } = yield Promise.resolve().then(() => require('./add-vite-commands-to-package-scripts'));
            addViteCommandsToPackageScripts(options.reactAppName, options.isStandalone);
            (0, write_vite_config_1.writeViteConfig)(options.reactAppName, options.isStandalone, options.appIsJs);
            (0, write_vite_index_html_1.writeViteIndexHtml)(options.reactAppName, options.isStandalone, options.appIsJs);
            (0, rename_js_to_jsx_1.renameJsToJsx)(options.reactAppName, options.isStandalone);
        }
        else {
            output_1.output.log({ title: '🧑‍🔧  Setting up craco + Webpack' });
            const { addCracoCommandsToPackageScripts } = yield Promise.resolve().then(() => require('./add-craco-commands-to-package-scripts'));
            addCracoCommandsToPackageScripts(options.reactAppName, options.isStandalone);
            (0, write_craco_config_1.writeCracoConfig)(options.reactAppName, options.isCRA5, options.isStandalone);
            output_1.output.log({
                title: '🛬 Skip CRA preflight check since Nx manages the monorepo',
            });
            (0, child_process_1.execSync)(`echo "SKIP_PREFLIGHT_CHECK=true" > .env`, { stdio: [0, 1, 2] });
        }
    });
}
function copyFromTempWorkspaceToRoot() {
    output_1.output.log({ title: '🚚 Folder restructuring.' });
    (0, fs_extra_1.readdirSync)('temp-workspace').forEach((f) => {
        (0, fs_extra_1.moveSync)((0, path_1.join)('temp-workspace', f), f, { overwrite: true });
    });
}
function cleanUpUnusedFilesAndAddConfigFiles(options) {
    var _a, _b;
    output_1.output.log({ title: '🧹  Cleaning up.' });
    (0, clean_up_files_1.cleanUpFiles)(options.reactAppName, options.isStandalone);
    output_1.output.log({ title: "📃 Extend the app's tsconfig.json from the base" });
    (0, tsconfig_setup_1.setupTsConfig)(options.reactAppName, options.isStandalone);
    if (options.e2e && !options.isStandalone) {
        output_1.output.log({ title: '📃 Setup e2e tests' });
        (0, setup_e2e_project_1.setupE2eProject)(options.reactAppName);
    }
    else {
        (0, fs_extra_1.removeSync)((0, path_1.join)('apps', `${options.reactAppName}-e2e`));
        const packageJson = (0, fileutils_1.readJsonFile)('package.json');
        const cypressPkgName = ((_a = packageJson.devDependencies) === null || _a === void 0 ? void 0 : _a['@nrwl/cypress']) ||
            ((_b = packageJson.dependencies) === null || _b === void 0 ? void 0 : _b['@nrwl/cypress'])
            ? '@nrwl/cypress'
            : '@nx/cypress';
        (0, child_process_1.execSync)(`${options.pmc.rm} ${cypressPkgName} eslint-plugin-cypress`);
    }
    if (options.isStandalone) {
        (0, fs_extra_1.removeSync)('apps');
    }
}
//# sourceMappingURL=index.js.map