"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stringifyPnpmLockfile = exports.parsePnpmLockfile = void 0;
const pnpm_normalizer_1 = require("./utils/pnpm-normalizer");
const package_json_1 = require("./utils/package-json");
const object_sort_1 = require("../../../utils/object-sort");
const hashing_impl_1 = require("../../../hasher/hashing-impl");
function parsePnpmLockfile(lockFileContent, builder) {
    const data = (0, pnpm_normalizer_1.parseAndNormalizePnpmLockfile)(lockFileContent);
    // we use key => node map to avoid duplicate work when parsing keys
    const keyMap = new Map();
    addNodes(data, builder, keyMap);
    addDependencies(data, builder, keyMap);
}
exports.parsePnpmLockfile = parsePnpmLockfile;
function addNodes(data, builder, keyMap) {
    const nodes = new Map();
    Object.entries(data.packages).forEach(([key, snapshot]) => {
        var _a, _b, _c;
        const packageName = findPackageName(key, snapshot, data);
        const version = findVersion(key, packageName).split('_')[0];
        // we don't need to keep duplicates, we can just track the keys
        const existingNode = (_a = nodes.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version);
        if (existingNode) {
            keyMap.set(key, existingNode);
            return;
        }
        const node = {
            type: 'npm',
            name: version ? `npm:${packageName}@${version}` : `npm:${packageName}`,
            data: {
                version,
                packageName,
                hash: ((_b = snapshot.resolution) === null || _b === void 0 ? void 0 : _b['integrity']) ||
                    hashing_impl_1.defaultHashing.hashArray(((_c = snapshot.resolution) === null || _c === void 0 ? void 0 : _c['tarball'])
                        ? [snapshot.resolution['tarball']]
                        : [packageName, version]),
            },
        };
        keyMap.set(key, node);
        if (!nodes.has(packageName)) {
            nodes.set(packageName, new Map([[version, node]]));
        }
        else {
            nodes.get(packageName).set(version, node);
        }
    });
    const hoistedDeps = (0, pnpm_normalizer_1.loadPnpmHoistedDepsDefinition)();
    for (const [packageName, versionMap] of nodes.entries()) {
        let hoistedNode;
        if (versionMap.size === 1) {
            hoistedNode = versionMap.values().next().value;
        }
        else {
            const hoistedVersion = getHoistedVersion(hoistedDeps, packageName);
            hoistedNode = versionMap.get(hoistedVersion);
        }
        if (hoistedNode) {
            hoistedNode.name = `npm:${packageName}`;
        }
        versionMap.forEach((node) => {
            builder.addExternalNode(node);
        });
    }
}
function getHoistedVersion(hoistedDependencies, packageName) {
    let version = (0, package_json_1.getHoistedPackageVersion)(packageName);
    if (!version) {
        const key = Object.keys(hoistedDependencies).find((k) => k.startsWith(`/${packageName}/`));
        if (key) {
            version = key.slice(key.lastIndexOf('/') + 1).split('_')[0];
        }
        else {
            // pnpm might not hoist every package
            // similarly those packages will not be available to be used via import
            return;
        }
    }
    return version;
}
function addDependencies(data, builder, keyMap) {
    Object.entries(data.packages).forEach(([key, snapshot]) => {
        const node = keyMap.get(key);
        [snapshot.dependencies, snapshot.optionalDependencies].forEach((section) => {
            if (section) {
                Object.entries(section).forEach(([name, versionRange]) => {
                    const version = findVersion(versionRange, name).split('_')[0];
                    const target = builder.graph.externalNodes[`npm:${name}@${version}`] ||
                        builder.graph.externalNodes[`npm:${name}`];
                    if (target) {
                        builder.addStaticDependency(node.name, target.name);
                    }
                });
            }
        });
    });
}
function stringifyPnpmLockfile(graph, rootLockFileContent, packageJson) {
    const data = (0, pnpm_normalizer_1.parseAndNormalizePnpmLockfile)(rootLockFileContent);
    const output = {
        lockfileVersion: data.lockfileVersion,
        importers: {
            '.': mapRootSnapshot(packageJson, data.packages, graph.externalNodes),
        },
        packages: (0, object_sort_1.sortObjectByKeys)(mapSnapshots(data.packages, graph.externalNodes)),
    };
    return (0, pnpm_normalizer_1.stringifyToPnpmYaml)(output);
}
exports.stringifyPnpmLockfile = stringifyPnpmLockfile;
function mapSnapshots(packages, nodes) {
    const result = {};
    Object.values(nodes).forEach((node) => {
        const matchedKeys = findOriginalKeys(packages, node, {
            returnFullKey: true,
        });
        // the package manager doesn't check for types of dependencies
        // so we can safely set all to prod
        matchedKeys.forEach(([key, snapshot]) => {
            snapshot.dev = false;
            result[key] = snapshot;
        });
    });
    return result;
}
function findOriginalKeys(packages, { data: { packageName, version } }, { returnFullKey } = {}) {
    const matchedKeys = [];
    for (const key of Object.keys(packages)) {
        const snapshot = packages[key];
        // standard package
        if (key.startsWith(`/${packageName}/${version}`)) {
            matchedKeys.push([returnFullKey ? key : key.split('/').pop(), snapshot]);
        }
        // tarball package
        if (key === version) {
            matchedKeys.push([version, snapshot]);
        }
        // alias package
        if (version.startsWith('npm:') &&
            key.startsWith(`/${version.slice(4, version.lastIndexOf('@'))}/${version.slice(version.lastIndexOf('@') + 1)}`)) {
            matchedKeys.push([key, snapshot]);
        }
    }
    return matchedKeys;
}
function mapRootSnapshot(packageJson, packages, nodes) {
    const snapshot = { specifiers: {} };
    [
        'dependencies',
        'optionalDependencies',
        'devDependencies',
        'peerDependencies',
    ].forEach((depType) => {
        if (packageJson[depType]) {
            Object.keys(packageJson[depType]).forEach((packageName) => {
                const version = packageJson[depType][packageName];
                const node = nodes[`npm:${packageName}@${version}`] || nodes[`npm:${packageName}`];
                snapshot.specifiers[packageName] = version;
                // peer dependencies are mapped to dependencies
                let section = depType === 'peerDependencies' ? 'dependencies' : depType;
                snapshot[section] = snapshot[section] || {};
                snapshot[section][packageName] = findOriginalKeys(packages, node)[0][0];
            });
        }
    });
    Object.keys(snapshot).forEach((key) => {
        snapshot[key] = (0, object_sort_1.sortObjectByKeys)(snapshot[key]);
    });
    return snapshot;
}
function findVersion(key, packageName) {
    if (key.startsWith(`/${packageName}/`)) {
        return key.slice(key.lastIndexOf('/') + 1);
    }
    // for alias packages prepend with "npm:"
    if (key.startsWith('/')) {
        const aliasName = key.slice(1, key.lastIndexOf('/'));
        const version = key.slice(key.lastIndexOf('/') + 1);
        return `npm:${aliasName}@${version}`;
    }
    // for tarball package the entire key is the version spec
    return key;
}
function findPackageName(key, value, data) {
    const matchPropValue = (record) => {
        if (!record) {
            return undefined;
        }
        const index = Object.values(record).findIndex((version) => version === key);
        if (index > -1) {
            return Object.keys(record)[index];
        }
    };
    const matchedDependencyName = (snapshot) => {
        return (matchPropValue(snapshot.dependencies) ||
            matchPropValue(snapshot.optionalDependencies) ||
            matchPropValue(snapshot.peerDependencies));
    };
    // snapshot already has a name
    if (value.name) {
        return value.name;
    }
    // it'a a root dependency
    const rootDependencyName = matchedDependencyName(data.importers['.']) ||
        // only root importers have devDependencies
        matchPropValue(data.importers['.'].devDependencies);
    if (rootDependencyName) {
        return rootDependencyName;
    }
    // find a snapshot that has a dependency that points to this snapshot
    const snapshots = Object.values(data.packages);
    for (let i = 0; i < snapshots.length; i++) {
        const snapshot = snapshots[i];
        const dependencyName = matchedDependencyName(snapshot);
        if (dependencyName) {
            return dependencyName;
        }
    }
    // otherwise, it's a standard package
    return key.startsWith('/') ? key.slice(1, key.lastIndexOf('/')) : key;
}
//# sourceMappingURL=pnpm-parser.js.map