"use strict";
/**
 * This file contains the logic to convert pnpm lockfile to a standard format.
 * It will convert inline specifiers to the separate specifiers format and ensure importers are present.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DEPENDENCIES_FIELDS = exports.loadPnpmHoistedDepsDefinition = exports.parseAndNormalizePnpmLockfile = exports.stringifyToPnpmYaml = void 0;
const tslib_1 = require("tslib");
const js_yaml_1 = require("@zkochan/js-yaml");
const fs_1 = require("fs");
const workspace_root_1 = require("../../../../utils/workspace-root");
const LOCKFILE_YAML_FORMAT = {
    blankLines: true,
    lineWidth: 1000,
    noCompatMode: true,
    noRefs: true,
    sortKeys: false,
};
function stringifyToPnpmYaml(lockfile) {
    return (0, js_yaml_1.dump)(lockfile, LOCKFILE_YAML_FORMAT);
}
exports.stringifyToPnpmYaml = stringifyToPnpmYaml;
function parseAndNormalizePnpmLockfile(content) {
    const lockFileData = (0, js_yaml_1.load)(content);
    return revertFromInlineSpecifiersFormatIfNecessary(convertFromLockfileFileMutable(lockFileData));
}
exports.parseAndNormalizePnpmLockfile = parseAndNormalizePnpmLockfile;
function loadPnpmHoistedDepsDefinition() {
    var _a, _b;
    const fullPath = `${workspace_root_1.workspaceRoot}/node_modules/.modules.yaml`;
    if ((0, fs_1.existsSync)(fullPath)) {
        const content = (0, fs_1.readFileSync)(fullPath, 'utf-8');
        return (_b = (_a = (0, js_yaml_1.load)(content)) === null || _a === void 0 ? void 0 : _a.hoistedDependencies) !== null && _b !== void 0 ? _b : {};
    }
    else {
        throw new Error(`Could not find ".modules.yaml" at "${fullPath}"`);
    }
}
exports.loadPnpmHoistedDepsDefinition = loadPnpmHoistedDepsDefinition;
/**
 * THE FOLLOWING CODE IS COPIED FROM @pnpm/lockfile-file for convenience
 */
function isMutableLockfile(lockfileFile) {
    return typeof lockfileFile['importers'] === 'undefined';
}
/**
 * Reverts changes from the "forceSharedFormat" write option if necessary.
 */
function convertFromLockfileFileMutable(lockfileFile) {
    var _a;
    if (isMutableLockfile(lockfileFile)) {
        lockfileFile.importers = {
            '.': Object.assign(Object.assign({ specifiers: (_a = lockfileFile['specifiers']) !== null && _a !== void 0 ? _a : {} }, (lockfileFile['dependenciesMeta'] && {
                dependenciesMeta: lockfileFile['dependenciesMeta'],
            })), (lockfileFile['publishDirectory'] && {
                publishDirectory: lockfileFile['publishDirectory'],
            })),
        };
        delete lockfileFile.specifiers;
        for (const depType of exports.DEPENDENCIES_FIELDS) {
            if (lockfileFile[depType] != null) {
                lockfileFile.importers['.'][depType] = lockfileFile[depType];
                delete lockfileFile[depType];
            }
        }
        return lockfileFile;
    }
    else {
        return lockfileFile;
    }
}
const INLINE_SPECIFIERS_FORMAT_LOCKFILE_VERSION_SUFFIX = '-inlineSpecifiers';
function isInlineSpecifierLockfile(lockfile) {
    return (typeof lockfile.lockfileVersion === 'string' &&
        lockfile.lockfileVersion.endsWith(INLINE_SPECIFIERS_FORMAT_LOCKFILE_VERSION_SUFFIX));
}
function revertFromInlineSpecifiersFormatIfNecessary(lockFile) {
    if (isInlineSpecifierLockfile(lockFile)) {
        const { lockfileVersion, importers } = lockFile, rest = tslib_1.__rest(lockFile, ["lockfileVersion", "importers"]);
        const originalVersionStr = lockfileVersion.replace(INLINE_SPECIFIERS_FORMAT_LOCKFILE_VERSION_SUFFIX, '');
        const originalVersion = Number(originalVersionStr);
        if (isNaN(originalVersion)) {
            throw new Error(`Unable to revert lockfile from inline specifiers format. Invalid version parsed: ${originalVersionStr}`);
        }
        const mappedImporters = {};
        Object.entries(importers).forEach(([key, value]) => {
            mappedImporters[key] = revertProjectSnapshot(value);
        });
        return Object.assign(Object.assign({}, rest), { lockfileVersion: originalVersion, importers: mappedImporters });
    }
    return lockFile;
}
function revertProjectSnapshot(from) {
    const specifiers = {};
    function moveSpecifiers(from) {
        const resolvedDependencies = {};
        for (const [depName, { specifier, version }] of Object.entries(from)) {
            const existingValue = specifiers[depName];
            if (existingValue != null && existingValue !== specifier) {
                throw new Error(`Project snapshot lists the same dependency more than once with conflicting versions: ${depName}`);
            }
            specifiers[depName] = specifier;
            resolvedDependencies[depName] = version;
        }
        return resolvedDependencies;
    }
    const dependencies = from.dependencies
        ? moveSpecifiers(from.dependencies)
        : null;
    const devDependencies = from.devDependencies
        ? moveSpecifiers(from.devDependencies)
        : null;
    const optionalDependencies = from.optionalDependencies
        ? moveSpecifiers(from.optionalDependencies)
        : null;
    return Object.assign(Object.assign({}, from), { specifiers,
        dependencies,
        devDependencies,
        optionalDependencies });
}
exports.DEPENDENCIES_FIELDS = [
    'optionalDependencies',
    'dependencies',
    'devDependencies',
];
//# sourceMappingURL=pnpm-normalizer.js.map