"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createPackageJson = void 0;
const fileutils_1 = require("../../../utils/fileutils");
const object_sort_1 = require("../../../utils/object-sort");
const fs_1 = require("fs");
const workspace_root_1 = require("../../../utils/workspace-root");
const hasher_1 = require("../../../hasher/hasher");
const configuration_1 = require("../../../config/configuration");
/**
 * Creates a package.json in the output directory for support to install dependencies within containers.
 *
 * If a package.json exists in the project, it will reuse that.
 * If isProduction flag is set, it wil  remove devDependencies and optional peerDependencies
 */
function createPackageJson(projectName, graph, options = {}) {
    var _a;
    const projectNode = graph.nodes[projectName];
    const { selfInputs, dependencyInputs } = options.target
        ? (0, hasher_1.getTargetInputs)((0, configuration_1.readNxJson)(), projectNode, options.target)
        : { selfInputs: [], dependencyInputs: [] };
    const npmDeps = {
        dependencies: {},
        peerDependencies: {},
        peerDependenciesMeta: {},
    };
    const seen = new Set();
    (_a = options.helperDependencies) === null || _a === void 0 ? void 0 : _a.forEach((dep) => {
        seen.add(dep);
        npmDeps.dependencies[graph.externalNodes[dep].data.packageName] =
            graph.externalNodes[dep].data.version;
        recursivelyCollectPeerDependencies(dep, graph, npmDeps, seen);
    });
    findAllNpmDeps(projectNode, graph, npmDeps, seen, dependencyInputs, selfInputs);
    // default package.json if one does not exist
    let packageJson = {
        name: projectName,
        version: '0.0.1',
    };
    if ((0, fs_1.existsSync)(`${graph.nodes[projectName].data.root}/package.json`)) {
        try {
            packageJson = (0, fileutils_1.readJsonFile)(`${graph.nodes[projectName].data.root}/package.json`);
            // for standalone projects we don't want to include all the root dependencies
            if (graph.nodes[projectName].data.root === '.') {
                packageJson = {
                    name: packageJson.name,
                    version: packageJson.version,
                };
            }
        }
        catch (e) { }
    }
    const rootPackageJson = (0, fileutils_1.readJsonFile)(`${options.root || workspace_root_1.workspaceRoot}/package.json`);
    Object.entries(npmDeps.dependencies).forEach(([packageName, version]) => {
        var _a, _b, _c, _d, _e, _f;
        if (((_a = rootPackageJson.devDependencies) === null || _a === void 0 ? void 0 : _a[packageName]) &&
            !((_b = packageJson.dependencies) === null || _b === void 0 ? void 0 : _b[packageName]) &&
            !((_c = packageJson.peerDependencies) === null || _c === void 0 ? void 0 : _c[packageName])) {
            // don't store dev dependencies for production
            if (!options.isProduction) {
                (_d = packageJson.devDependencies) !== null && _d !== void 0 ? _d : (packageJson.devDependencies = {});
                packageJson.devDependencies[packageName] = version;
            }
        }
        else {
            if (!((_e = packageJson.peerDependencies) === null || _e === void 0 ? void 0 : _e[packageName])) {
                (_f = packageJson.dependencies) !== null && _f !== void 0 ? _f : (packageJson.dependencies = {});
                packageJson.dependencies[packageName] = version;
            }
        }
    });
    Object.entries(npmDeps.peerDependencies).forEach(([packageName, version]) => {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        if (!((_a = packageJson.peerDependencies) === null || _a === void 0 ? void 0 : _a[packageName])) {
            if ((_b = rootPackageJson.dependencies) === null || _b === void 0 ? void 0 : _b[packageName]) {
                (_c = packageJson.dependencies) !== null && _c !== void 0 ? _c : (packageJson.dependencies = {});
                packageJson.dependencies[packageName] = version;
                return;
            }
            const isOptionalPeer = (_d = npmDeps.peerDependenciesMeta[packageName]) === null || _d === void 0 ? void 0 : _d.optional;
            if (!isOptionalPeer) {
                if (!options.isProduction ||
                    ((_e = rootPackageJson.dependencies) === null || _e === void 0 ? void 0 : _e[packageName])) {
                    (_f = packageJson.peerDependencies) !== null && _f !== void 0 ? _f : (packageJson.peerDependencies = {});
                    packageJson.peerDependencies[packageName] = version;
                }
            }
            else if (!options.isProduction) {
                // add peer optional dependencies if not in production
                (_g = packageJson.peerDependencies) !== null && _g !== void 0 ? _g : (packageJson.peerDependencies = {});
                packageJson.peerDependencies[packageName] = version;
                (_h = packageJson.peerDependenciesMeta) !== null && _h !== void 0 ? _h : (packageJson.peerDependenciesMeta = {});
                packageJson.peerDependenciesMeta[packageName] = {
                    optional: true,
                };
            }
        }
    });
    packageJson.devDependencies && (packageJson.devDependencies = (0, object_sort_1.sortObjectByKeys)(packageJson.devDependencies));
    packageJson.dependencies && (packageJson.dependencies = (0, object_sort_1.sortObjectByKeys)(packageJson.dependencies));
    packageJson.peerDependencies && (packageJson.peerDependencies = (0, object_sort_1.sortObjectByKeys)(packageJson.peerDependencies));
    packageJson.peerDependenciesMeta && (packageJson.peerDependenciesMeta = (0, object_sort_1.sortObjectByKeys)(packageJson.peerDependenciesMeta));
    return packageJson;
}
exports.createPackageJson = createPackageJson;
function findAllNpmDeps(projectNode, graph, npmDeps, seen, dependencyPatterns, rootPatterns) {
    if (seen.has(projectNode.name))
        return;
    seen.add(projectNode.name);
    const projectFiles = (0, hasher_1.filterUsingGlobPatterns)(projectNode.data.root, projectNode.data.files, rootPatterns !== null && rootPatterns !== void 0 ? rootPatterns : dependencyPatterns);
    const projectDependencies = new Set();
    projectFiles.forEach((fileData) => { var _a; return (_a = fileData.dependencies) === null || _a === void 0 ? void 0 : _a.forEach((dep) => projectDependencies.add(dep.target)); });
    for (const dep of projectDependencies) {
        const node = graph.externalNodes[dep];
        if (seen.has(dep)) {
            // if it's in peerDependencies, move it to regular dependencies
            // since this is a direct dependency of the project
            if (node && npmDeps.peerDependencies[node.data.packageName]) {
                npmDeps.dependencies[node.data.packageName] = node.data.version;
                delete npmDeps.peerDependencies[node.data.packageName];
            }
        }
        else {
            if (node) {
                seen.add(dep);
                npmDeps.dependencies[node.data.packageName] = node.data.version;
                recursivelyCollectPeerDependencies(node.name, graph, npmDeps, seen);
            }
            else {
                findAllNpmDeps(graph.nodes[dep], graph, npmDeps, seen, dependencyPatterns);
            }
        }
    }
}
function recursivelyCollectPeerDependencies(projectName, graph, npmDeps, seen) {
    const npmPackage = graph.externalNodes[projectName];
    if (!npmPackage) {
        return npmDeps;
    }
    const packageName = npmPackage.data.packageName;
    try {
        const packageJson = require(`${packageName}/package.json`);
        if (!packageJson.peerDependencies) {
            return npmDeps;
        }
        Object.keys(packageJson.peerDependencies)
            .map((dependencyName) => `npm:${dependencyName}`)
            .map((dependency) => graph.externalNodes[dependency])
            .filter(Boolean)
            .forEach((node) => {
            if (!seen.has(node.name)) {
                seen.add(node.name);
                npmDeps.peerDependencies[node.data.packageName] = node.data.version;
                if (packageJson.peerDependenciesMeta &&
                    packageJson.peerDependenciesMeta[node.data.packageName] &&
                    packageJson.peerDependenciesMeta[node.data.packageName].optional) {
                    npmDeps.peerDependenciesMeta[node.data.packageName] = {
                        optional: true,
                    };
                }
                recursivelyCollectPeerDependencies(node.name, graph, npmDeps, seen);
            }
        });
        return npmDeps;
    }
    catch (e) {
        return npmDeps;
    }
}
//# sourceMappingURL=create-package-json.js.map