"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildExplicitDependencies = void 0;
const project_graph_1 = require("../../../../config/project-graph");
const path_1 = require("path");
const build_explicit_typescript_and_package_json_dependencies_1 = require("./build-explicit-typescript-and-package-json-dependencies");
const os = require("os");
function buildExplicitDependencies(jsPluginConfig, ctx, builder) {
    let totalNumOfFilesToProcess = totalNumberOfFilesToProcess(ctx);
    // using workers has an overhead, so we only do it when the number of
    // files we need to process is >= 100 and there are more than 2 CPUs
    // to be able to use at least 2 workers (1 worker per CPU and
    // 1 CPU for the main thread)
    if (totalNumOfFilesToProcess < 100 || getNumberOfWorkers() <= 2) {
        return buildExplicitDependenciesWithoutWorkers(jsPluginConfig, ctx, builder);
    }
    else {
        return buildExplicitDependenciesUsingWorkers(jsPluginConfig, ctx, totalNumOfFilesToProcess, builder);
    }
}
exports.buildExplicitDependencies = buildExplicitDependencies;
function totalNumberOfFilesToProcess(ctx) {
    let totalNumOfFilesToProcess = 0;
    Object.values(ctx.filesToProcess).forEach((t) => (totalNumOfFilesToProcess += t.length));
    return totalNumOfFilesToProcess;
}
function splitFilesIntoBins(ctx, totalNumOfFilesToProcess, numberOfWorkers) {
    // we want to have numberOfWorkers * 5 bins
    const filesPerBin = Math.round(totalNumOfFilesToProcess / numberOfWorkers / 5) + 1;
    const bins = [];
    let currentProjectFileMap = {};
    let currentNumberOfFiles = 0;
    for (const source of Object.keys(ctx.filesToProcess)) {
        for (const f of Object.values(ctx.filesToProcess[source])) {
            if (!currentProjectFileMap[source])
                currentProjectFileMap[source] = [];
            currentProjectFileMap[source].push(f);
            currentNumberOfFiles++;
            if (currentNumberOfFiles >= filesPerBin) {
                bins.push(currentProjectFileMap);
                currentProjectFileMap = {};
                currentNumberOfFiles = 0;
            }
        }
    }
    bins.push(currentProjectFileMap);
    return bins;
}
function createWorkerPool(numberOfWorkers) {
    const res = [];
    for (let i = 0; i < numberOfWorkers; ++i) {
        res.push(new (require('worker_threads').Worker)((0, path_1.join)(__dirname, './project-graph-worker.js'), {
            env: process.env,
        }));
    }
    return res;
}
function buildExplicitDependenciesWithoutWorkers(jsPluginConfig, ctx, builder) {
    (0, build_explicit_typescript_and_package_json_dependencies_1.buildExplicitTypescriptAndPackageJsonDependencies)(jsPluginConfig, ctx.nxJsonConfiguration, ctx.projectsConfigurations, builder.graph, ctx.filesToProcess).forEach((r) => {
        if (r.type === project_graph_1.DependencyType.static) {
            builder.addStaticDependency(r.sourceProjectName, r.targetProjectName, r.sourceProjectFile);
        }
        else {
            builder.addDynamicDependency(r.sourceProjectName, r.targetProjectName, r.sourceProjectFile);
        }
    });
}
function buildExplicitDependenciesUsingWorkers(jsPluginConfig, ctx, totalNumOfFilesToProcess, builder) {
    const numberOfWorkers = Math.min(totalNumOfFilesToProcess, getNumberOfWorkers());
    const bins = splitFilesIntoBins(ctx, totalNumOfFilesToProcess, numberOfWorkers);
    const workers = createWorkerPool(numberOfWorkers);
    let numberOfExpectedResponses = bins.length;
    return new Promise((res, reject) => {
        for (let w of workers) {
            w.on('message', (explicitDependencies) => {
                explicitDependencies.forEach((r) => {
                    builder.addExplicitDependency(r.sourceProjectName, r.sourceProjectFile, r.targetProjectName);
                });
                if (bins.length > 0) {
                    w.postMessage({ filesToProcess: bins.shift() });
                }
                // we processed all the bins
                if (--numberOfExpectedResponses === 0) {
                    for (let w of workers) {
                        w.terminate();
                    }
                    res(null);
                }
            });
            w.on('error', reject);
            w.on('exit', (code) => {
                if (code !== 0) {
                    reject(new Error(`Unable to complete project graph creation. Worker stopped with exit code: ${code}`));
                }
            });
            w.postMessage({
                nxJsonConfiguration: ctx.nxJsonConfiguration,
                projectsConfigurations: ctx.projectsConfigurations,
                projectGraph: builder.graph,
                jsPluginConfig,
            });
            w.postMessage({ filesToProcess: bins.shift() });
        }
    });
}
function getNumberOfWorkers() {
    return process.env.NX_PROJECT_GRAPH_MAX_WORKERS
        ? +process.env.NX_PROJECT_GRAPH_MAX_WORKERS
        : Math.min(os.cpus().length - 1, 8); // This is capped for cases in CI where `os.cpus()` returns way more CPUs than the resources that are allocated
}
//# sourceMappingURL=build-dependencies.js.map