"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeImplicitDependencies = exports.buildWorkspaceProjectNodes = void 0;
const tslib_1 = require("tslib");
const path_1 = require("path");
const fs_1 = require("fs");
const workspace_root_1 = require("../../utils/workspace-root");
const nx_plugin_1 = require("../../utils/nx-plugin");
const project_graph_utils_1 = require("../../utils/project-graph-utils");
const fileutils_1 = require("../../utils/fileutils");
const find_matching_projects_1 = require("../../utils/find-matching-projects");
const logger_1 = require("../../utils/logger");
const workspaces_1 = require("../../config/workspaces");
function buildWorkspaceProjectNodes(ctx, builder, nxJson) {
    var _a, _b;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const toAdd = [];
        const projects = Object.keys(ctx.workspace.projects);
        const projectsSet = new Set(projects);
        for (const key of projects) {
            const p = ctx.workspace.projects[key];
            const projectRoot = (0, path_1.join)(workspace_root_1.workspaceRoot, p.root);
            if ((0, fs_1.existsSync)((0, path_1.join)(projectRoot, 'package.json'))) {
                p.targets = (0, project_graph_utils_1.mergeNpmScriptsWithTargets)(projectRoot, p.targets);
                try {
                    const { nx } = (0, fileutils_1.readJsonFile)((0, path_1.join)(projectRoot, 'package.json'));
                    if (nx === null || nx === void 0 ? void 0 : nx.tags) {
                        p.tags = [...(p.tags || []), ...nx.tags];
                    }
                    if (nx === null || nx === void 0 ? void 0 : nx.implicitDependencies) {
                        p.implicitDependencies = [
                            ...(p.implicitDependencies || []),
                            ...nx.implicitDependencies,
                        ];
                    }
                    if (nx === null || nx === void 0 ? void 0 : nx.namedInputs) {
                        p.namedInputs = Object.assign(Object.assign({}, (p.namedInputs || {})), nx.namedInputs);
                    }
                }
                catch (_c) {
                    // ignore json parser errors
                }
            }
            p.implicitDependencies = normalizeImplicitDependencies(key, p.implicitDependencies, projects, projectsSet);
            p.targets = (0, nx_plugin_1.mergePluginTargetsWithNxTargets)(p.root, p.targets, yield (0, nx_plugin_1.loadNxPlugins)(ctx.workspace.plugins));
            p.targets = normalizeProjectTargets(p, nxJson.targetDefaults, key);
            // TODO: remove in v16
            const projectType = p.projectType === 'application'
                ? key.endsWith('-e2e') || key === 'e2e'
                    ? 'e2e'
                    : 'app'
                : 'lib';
            const tags = ((_b = (_a = ctx.workspace.projects) === null || _a === void 0 ? void 0 : _a[key]) === null || _b === void 0 ? void 0 : _b.tags) || [];
            toAdd.push({
                name: key,
                type: projectType,
                data: Object.assign(Object.assign({}, p), { tags, files: ctx.fileMap[key] }),
            });
        }
        // Sort by root directory length (do we need this?)
        toAdd.sort((a, b) => {
            if (!a.data.root)
                return -1;
            if (!b.data.root)
                return -1;
            return a.data.root.length > b.data.root.length ? -1 : 1;
        });
        toAdd.forEach((n) => {
            builder.addNode({
                name: n.name,
                type: n.type,
                data: n.data,
            });
        });
    });
}
exports.buildWorkspaceProjectNodes = buildWorkspaceProjectNodes;
/**
 * Apply target defaults and normalization
 */
function normalizeProjectTargets(project, targetDefaults, projectName) {
    var _a;
    const targets = project.targets;
    for (const target in targets) {
        const executor = ((_a = targets[target].executor) !== null && _a !== void 0 ? _a : targets[target].command)
            ? 'nx:run-commands'
            : null;
        const defaults = (0, workspaces_1.readTargetDefaultsForTarget)(target, targetDefaults, executor);
        if (defaults) {
            targets[target] = (0, workspaces_1.mergeTargetConfigurations)(project, target, defaults);
        }
        const config = targets[target];
        if (config.command) {
            if (config.executor) {
                throw new Error(`${logger_1.NX_PREFIX} ${projectName}: ${target} should not have executor and command both configured.`);
            }
            else {
                targets[target] = Object.assign(Object.assign({}, targets[target]), { executor, options: Object.assign(Object.assign({}, config.options), { command: config.command }) });
                delete config.command;
            }
        }
    }
    return targets;
}
function normalizeImplicitDependencies(source, implicitDependencies, projectNames, projectsSet) {
    if (!(implicitDependencies === null || implicitDependencies === void 0 ? void 0 : implicitDependencies.length)) {
        return implicitDependencies !== null && implicitDependencies !== void 0 ? implicitDependencies : [];
    }
    const matches = (0, find_matching_projects_1.findMatchingProjects)(implicitDependencies, projectNames, projectsSet);
    return (matches
        .filter((x) => x !== source)
        // implicit dependencies that start with ! should hang around, to be processed by
        // implicit-project-dependencies.ts after explicit deps are added to graph.
        .concat(implicitDependencies.filter((x) => x.startsWith('!'))));
}
exports.normalizeImplicitDependencies = normalizeImplicitDependencies;
//# sourceMappingURL=workspace-projects.js.map