"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildProjectGraphUsingProjectFileMap = exports.buildProjectGraph = void 0;
const tslib_1 = require("tslib");
const workspace_root_1 = require("../utils/workspace-root");
const path_1 = require("path");
const perf_hooks_1 = require("perf_hooks");
const assert_workspace_validity_1 = require("../utils/assert-workspace-validity");
const nx_deps_cache_1 = require("./nx-deps-cache");
const build_dependencies_1 = require("./build-dependencies");
const build_nodes_1 = require("./build-nodes");
const nx_plugin_1 = require("../utils/nx-plugin");
const file_hasher_1 = require("../hasher/file-hasher");
const file_map_utils_1 = require("./file-map-utils");
const typescript_1 = require("../utils/typescript");
const fileutils_1 = require("../utils/fileutils");
const logger_1 = require("../utils/logger");
const project_graph_builder_1 = require("./project-graph-builder");
const configuration_1 = require("../config/configuration");
const workspaces_1 = require("../config/workspaces");
const fs_1 = require("fs");
function buildProjectGraph() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const projectConfigurations = new workspaces_1.Workspaces(workspace_root_1.workspaceRoot).readProjectsConfigurations();
        const { projectFileMap, allWorkspaceFiles } = (0, file_map_utils_1.createProjectFileMap)(projectConfigurations, file_hasher_1.defaultFileHasher.allFileData());
        const cacheEnabled = process.env.NX_CACHE_PROJECT_GRAPH !== 'false';
        let cache = cacheEnabled ? (0, nx_deps_cache_1.readCache)() : null;
        return (yield buildProjectGraphUsingProjectFileMap(projectConfigurations, projectFileMap, allWorkspaceFiles, cache, cacheEnabled)).projectGraph;
    });
}
exports.buildProjectGraph = buildProjectGraph;
function buildProjectGraphUsingProjectFileMap(projectsConfigurations, projectFileMap, allWorkspaceFiles, cache, shouldWriteCache) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const nxJson = (0, configuration_1.readNxJson)();
        const projectGraphVersion = '5.1';
        (0, assert_workspace_validity_1.assertWorkspaceValidity)(projectsConfigurations, nxJson);
        const packageJsonDeps = readCombinedDeps();
        const rootTsConfig = readRootTsConfig();
        let filesToProcess;
        let cachedFileData;
        const useCacheData = cache &&
            !(0, nx_deps_cache_1.shouldRecomputeWholeGraph)(cache, packageJsonDeps, projectsConfigurations, nxJson, rootTsConfig);
        if (useCacheData) {
            const fromCache = (0, nx_deps_cache_1.extractCachedFileData)(projectFileMap, cache);
            filesToProcess = fromCache.filesToProcess;
            cachedFileData = fromCache.cachedFileData;
        }
        else {
            filesToProcess = projectFileMap;
            cachedFileData = {};
        }
        const context = createContext(projectsConfigurations, nxJson, projectFileMap, filesToProcess);
        let projectGraph = yield buildProjectGraphUsingContext(nxJson, context, cachedFileData, projectGraphVersion, useCacheData ? cache : null);
        const projectGraphCache = (0, nx_deps_cache_1.createCache)(nxJson, packageJsonDeps, projectGraph, rootTsConfig);
        if (shouldWriteCache) {
            (0, nx_deps_cache_1.writeCache)(projectGraphCache);
        }
        projectGraph.allWorkspaceFiles = allWorkspaceFiles;
        return {
            projectGraph,
            projectGraphCache,
        };
    });
}
exports.buildProjectGraphUsingProjectFileMap = buildProjectGraphUsingProjectFileMap;
function readCombinedDeps() {
    const installationPackageJsonPath = (0, path_1.join)(workspace_root_1.workspaceRoot, '.nx', 'installation', 'package.json');
    const installationPackageJson = (0, fs_1.existsSync)(installationPackageJsonPath)
        ? (0, fileutils_1.readJsonFile)(installationPackageJsonPath)
        : {};
    const rootPackageJsonPath = (0, path_1.join)(workspace_root_1.workspaceRoot, 'package.json');
    const rootPackageJson = (0, fs_1.existsSync)(rootPackageJsonPath)
        ? (0, fileutils_1.readJsonFile)(rootPackageJsonPath)
        : {};
    return Object.assign(Object.assign(Object.assign(Object.assign({}, rootPackageJson.dependencies), rootPackageJson.devDependencies), installationPackageJson.dependencies), installationPackageJson.devDependencies);
}
function buildProjectGraphUsingContext(nxJson, ctx, cachedFileData, projectGraphVersion, cache) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        perf_hooks_1.performance.mark('build project graph:start');
        const builder = new project_graph_builder_1.ProjectGraphBuilder(cache
            ? {
                nodes: cache.nodes,
                externalNodes: cache.externalNodes,
                dependencies: cache.dependencies,
            }
            : null);
        builder.setVersion(projectGraphVersion);
        yield (0, build_nodes_1.buildWorkspaceProjectNodes)(ctx, builder, nxJson);
        const initProjectGraph = builder.getUpdatedProjectGraph();
        const r = yield updateProjectGraphWithPlugins(ctx, initProjectGraph);
        const updatedBuilder = new project_graph_builder_1.ProjectGraphBuilder(r);
        for (const proj of Object.keys(cachedFileData)) {
            for (const f of updatedBuilder.graph.nodes[proj].data.files) {
                const cached = cachedFileData[proj][f.file];
                if (cached && cached.dependencies) {
                    f.dependencies = [...cached.dependencies];
                }
            }
        }
        (0, build_dependencies_1.buildImplicitProjectDependencies)(ctx, updatedBuilder);
        const finalGraph = updatedBuilder.getUpdatedProjectGraph();
        perf_hooks_1.performance.mark('build project graph:end');
        perf_hooks_1.performance.measure('build project graph', 'build project graph:start', 'build project graph:end');
        return finalGraph;
    });
}
function createContext(projectsConfigurations, nxJson, fileMap, filesToProcess) {
    const projects = Object.keys(projectsConfigurations.projects).reduce((map, projectName) => {
        map[projectName] = Object.assign({}, projectsConfigurations.projects[projectName]);
        return map;
    }, {});
    return {
        nxJsonConfiguration: nxJson,
        projectsConfigurations,
        workspace: Object.assign(Object.assign(Object.assign({}, projectsConfigurations), nxJson), { projects }),
        fileMap,
        filesToProcess,
    };
}
function updateProjectGraphWithPlugins(context, initProjectGraph) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const plugins = (yield (0, nx_plugin_1.loadNxPlugins)(context.nxJsonConfiguration.plugins)).filter((x) => !!x.processProjectGraph);
        let graph = initProjectGraph;
        for (const plugin of plugins) {
            try {
                graph = yield plugin.processProjectGraph(graph, context);
            }
            catch (e) {
                const message = `Failed to process the project graph with "${plugin.name}". This will error in the future!`;
                if (process.env.NX_VERBOSE_LOGGING === 'true') {
                    console.error(e);
                    logger_1.logger.error(message);
                    return graph;
                }
                else {
                    logger_1.logger.warn(message);
                    logger_1.logger.warn(`Run with NX_VERBOSE_LOGGING=true to see the error.`);
                }
            }
        }
        return graph;
    });
}
function readRootTsConfig() {
    try {
        const tsConfigPath = (0, typescript_1.getRootTsConfigPath)();
        if (tsConfigPath) {
            return (0, fileutils_1.readJsonFile)(tsConfigPath, { expectComments: true });
        }
    }
    catch (e) {
        return {};
    }
}
//# sourceMappingURL=build-project-graph.js.map