"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTsNodeCompilerOptions = exports.registerTsConfigPaths = exports.registerTranspiler = exports.registerTsProject = void 0;
const path_1 = require("path");
const logger_1 = require("./logger");
const swcNodeInstalled = packageIsInstalled('@swc-node/register');
const tsNodeInstalled = packageIsInstalled('ts-node/register');
let ts;
/**
 * Optionally, if swc-node and tsconfig-paths are available in the current workspace, apply the require
 * register hooks so that .ts files can be used for writing custom workspace projects.
 *
 * If ts-node and tsconfig-paths are not available, the user can still provide an index.js file in
 * the root of their project and the fundamentals will still work (but
 * workspace path mapping will not, for example).
 *
 * @returns cleanup function
 */
const registerTsProject = (path, configFilename = 'tsconfig.json') => {
    const tsConfigPath = (0, path_1.join)(path, configFilename);
    const compilerOptions = readCompilerOptions(tsConfigPath);
    const cleanupFunctions = [
        registerTsConfigPaths(tsConfigPath),
        registerTranspiler(compilerOptions),
    ];
    return () => {
        for (const fn of cleanupFunctions) {
            fn();
        }
    };
};
exports.registerTsProject = registerTsProject;
/**
 * Register ts-node or swc-node given a set of compiler options.
 *
 * Note: Several options require enums from typescript. To avoid importing typescript,
 * use import type + raw values
 *
 * @returns cleanup method
 */
function registerTranspiler(compilerOptions) {
    // Function to register transpiler that returns cleanup function
    let registerTranspiler;
    if (swcNodeInstalled) {
        // These are requires to prevent it from registering when it shouldn't
        const { register } = require('@swc-node/register/register');
        registerTranspiler = () => register(compilerOptions);
    }
    else {
        // We can fall back on ts-node if its available
        if (tsNodeInstalled) {
            const { register } = require('ts-node');
            // ts-node doesn't provide a cleanup method
            registerTranspiler = () => {
                const service = register({
                    transpileOnly: true,
                    compilerOptions: getTsNodeCompilerOptions(compilerOptions),
                });
                // Don't warn if a faster transpiler is enabled
                if (!service.options.transpiler && !service.options.swc) {
                    warnTsNodeUsage();
                }
                return () => { };
            };
        }
    }
    if (registerTranspiler) {
        return registerTranspiler();
    }
    else {
        warnNoTranspiler();
        return () => { };
    }
}
exports.registerTranspiler = registerTranspiler;
/**
 * @param tsConfigPath Adds the paths from a tsconfig file into node resolutions
 * @returns cleanup function
 */
function registerTsConfigPaths(tsConfigPath) {
    try {
        /**
         * Load the ts config from the source project
         */
        const tsconfigPaths = require('tsconfig-paths');
        const tsConfigResult = tsconfigPaths.loadConfig(tsConfigPath);
        /**
         * Register the custom workspace path mappings with node so that workspace libraries
         * can be imported and used within project
         */
        if (tsConfigResult.resultType === 'success') {
            return tsconfigPaths.register({
                baseUrl: tsConfigResult.absoluteBaseUrl,
                paths: tsConfigResult.paths,
            });
        }
    }
    catch (err) {
        warnNoTsconfigPaths();
    }
    return () => { };
}
exports.registerTsConfigPaths = registerTsConfigPaths;
function readCompilerOptions(tsConfigPath) {
    if (swcNodeInstalled) {
        const { readDefaultTsConfig, } = require('@swc-node/register/read-default-tsconfig');
        return readDefaultTsConfig(tsConfigPath);
    }
    else {
        return readCompilerOptionsWithTypescript(tsConfigPath);
    }
}
function readCompilerOptionsWithTypescript(tsConfigPath) {
    if (!ts) {
        ts = require('typescript');
    }
    const { readConfigFile, parseJsonConfigFileContent, sys } = ts;
    const jsonContent = readConfigFile(tsConfigPath, sys.readFile);
    const { options } = parseJsonConfigFileContent(jsonContent.config, sys, (0, path_1.dirname)(tsConfigPath));
    // This property is returned in compiler options for some reason, but not part of the typings.
    // ts-node fails on unknown props, so we have to remove it.
    delete options.configFilePath;
    return options;
}
function warnTsNodeUsage() {
    logger_1.logger.warn((0, logger_1.stripIndent)(`${logger_1.NX_PREFIX} Falling back to ts-node for local typescript execution. This may be a little slower.
  - To fix this, ensure @swc-node/register and @swc/core have been installed`));
}
function warnNoTsconfigPaths() {
    logger_1.logger.warn((0, logger_1.stripIndent)(`${logger_1.NX_PREFIX} Unable to load tsconfig-paths, workspace libraries may be inaccessible.
  - To fix this, install tsconfig-paths with npm/yarn/pnpm`));
}
function warnNoTranspiler() {
    logger_1.logger.warn((0, logger_1.stripIndent)(`${logger_1.NX_PREFIX} Unable to locate swc-node or ts-node. Nx will be unable to run local ts files without transpiling.
  - To fix this, ensure @swc-node/register and @swc/core have been installed`));
}
function packageIsInstalled(m) {
    try {
        const p = require.resolve(m);
        return true;
    }
    catch (_a) {
        return false;
    }
}
/**
 * ts-node requires string values for enum based typescript options.
 * `register`'s signature just types the field as `object`, so we
 * unfortunately do not get any kind of type safety on this.
 */
function getTsNodeCompilerOptions(compilerOptions) {
    var _a;
    if (!ts) {
        ts = require('typescript');
    }
    const flagMap = {
        module: 'ModuleKind',
        target: 'ScriptTarget',
        moduleDetection: 'ModuleDetectionKind',
        newLine: 'NewLineKind',
        moduleResolution: 'ModuleResolutionKind',
        importsNotUsedAsValues: 'ImportsNotUsedAsValues',
    };
    const result = Object.assign({}, compilerOptions);
    for (const flag in flagMap) {
        if (compilerOptions[flag]) {
            result[flag] = ts[flagMap[flag]][compilerOptions[flag]];
        }
    }
    delete result.pathsBasePath;
    delete result.configFilePath;
    // instead of mapping to enum value we just remove it as it shouldn't ever need to be set for ts-node
    delete result.jsx;
    // lib option is in the format `lib.es2022.d.ts`, so we need to remove the leading `lib.` and trailing `.d.ts` to make it valid
    result.lib = (_a = result.lib) === null || _a === void 0 ? void 0 : _a.map((value) => {
        return value.replace(/^lib\./, '').replace(/\.d\.ts$/, '');
    });
    if (result.moduleResolution) {
        result.moduleResolution =
            result.moduleResolution === 'NodeJs'
                ? 'node'
                : result.moduleResolution.toLowerCase();
    }
    return result;
}
exports.getTsNodeCompilerOptions = getTsNodeCompilerOptions;
//# sourceMappingURL=register.js.map