/**
 * A CSS selector.
 */
export declare type Selector = string;
/**
 * A hash fragment.
 */
export declare type Hash = string;
/**
 * Specifies an element that is the target of a side effect (e.g. scroll into view, focus). This is
 * either the element itself or a selector that will return the element when passed to querySelector().
 */
export declare type Target = Element | Selector;
/**
 * The scroll position as x and y coordinates.
 */
export declare type ScrollPosition = {
    readonly x: number;
    readonly y: number;
};
/**
 * Maps from a URL hash fragment to a target element.
 *
 * Supports "#", "#top" and element IDs. The empty string returns undefined.
 *
 * Useful for scrolling to the element referred to by the hash fragment
 * in a URL (which browsers do natively, but single page apps often don't).
 *
 * See https://github.com/rafrex/react-router-hash-link (only manages scroll, not focus)
 * See https://github.com/ReactTraining/react-router/issues/394
 * See https://www.w3.org/TR/html5/single-page.html#scroll-to-the-fragment
 *
 * @param hash the hash fragment, e.g. "#", "#top" or "#my-heading-id"
 */
export declare const elementFromHash: (hash: string) => Element | undefined;
/**
 * Set the document title.
 * See https://www.w3.org/TR/UNDERSTANDING-WCAG20/navigation-mechanisms-title.html
 *
 * @param title The new document title.
 */
export declare const setTitle: (title: string) => void;
/**
 * True if the specified element is within the viewport, false otherwise.
 * See https://stackoverflow.com/questions/123999/how-to-tell-if-a-dom-element-is-visible-in-the-current-viewport/7557433#7557433
 *
 * @param element the element to test
 */
export declare const isInViewport: (element: Element) => boolean;
export declare const elementFromTarget: (target: Target, parent?: ParentNode) => Element | undefined;
export declare const getScrollPosition: () => ScrollPosition;
export declare const setScrollPosition: (scrollPosition: ScrollPosition) => void;
/**
 * Focuses an element, setting `tabindex="-1"` if necessary.
 *
 * If you specify the preventScroll focus option (which has poor
 * browser support) this function will use a window.scrollTo()
 * hack to emulate the preventScroll option's behavior.
 *
 * See: https://developer.paciellogroup.com/blog/2014/08/using-the-tabindex-attribute/
 * See: https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus#Browser_compatibility
 * See: https://github.com/whatwg/html/issues/834
 * See: https://stackoverflow.com/questions/4963053/focus-to-input-without-scrolling/6610501
 * See: https://github.com/calvellido/focus-options-polyfill
 *
 * @param target the element to focus.
 * @param options focus options.
 */
export declare const focusElement: (target: Target, options?: FocusOptions | undefined) => Promise<boolean>;
/**
 * Scrolls an element into view if it is not currently visible.
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * @param target the element to scroll into view
 * @param options controls how the scroll is executed
 */
export declare const scrollIntoViewIfRequired: (target: Target, options?: ScrollIntoViewOptions | undefined, inViewport?: ((element: Element) => boolean) | undefined) => void;
/**
 * Focuses a specified element and then scrolls it (or another element) into view (if required).
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 */
export declare const focusAndScrollIntoViewIfRequired: (focusTarget: Target, scrollIntoViewTarget: Target, focusOptions?: FocusOptions | undefined, scrollIntoViewOptions?: ScrollIntoViewOptions | undefined) => Promise<boolean>;
/**
 * Resets focus after a SPA page navigation.
 *
 * See: https://github.com/ReactTraining/react-router/issues/5210
 *
 * @param primaryFocusTarget a CSS selector for your primary focus target,
 * e.g. `[main h1]`. This is the element that will receive focus after SPA
 * navigation. If this element does not exist the document body will be used
 * as a fallback.
 * @param focusSelector a CSS selector for the element to focus. If this
 * element does not exist the primaryFocusTarget will be used as a fallback.
 */
export declare const resetFocus: (primaryFocusTarget: string, focusTarget?: string | Element | undefined, focusOptions?: FocusOptions | undefined, scrollIntoViewOptions?: ScrollIntoViewOptions | undefined) => Promise<boolean>;
/**
 * Make an announcement to screen reader users. Useful for page navigation events.
 *
 * See https://almerosteyn.com/2017/03/accessible-react-navigation
 * See https://getbootstrap.com/docs/4.3/utilities/screen-readers/
 * See https://github.com/twbs/bootstrap/blob/ff29c1224c20b8fcf2d1e7c28426470f1dc3e40d/scss/mixins/_screen-reader.scss#L6
 *
 * @param message the message to announce to screen reader users, e.g. "navigated to about page".
 * @param announcementsDivId a DOM ID of the visually hidden announcements element, e.g. "announcements".
 */
export declare const announce: (message: string, announcementsDivId?: string) => void;
/**
 * Focuses and scrolls into view the first invalid form element inside
 * a given form.
 *
 * For IE support you might want to use the closest() polyfill from https://developer.mozilla.org/en-US/docs/Web/API/Element/closest#Polyfill
 *
 * @param formSelector a CSS selector that uniquely identifies the form to focus, e.g. `#search-form`.
 * @param invalidElementSelector the CSS selector that is used to identify invalid elements within a form, e.g. `[aria-invalid="true"]`.
 * @param formGroupSelector a CSS selector passed to the closest() method of an invalid form input that identifies the element
 *                          that contains both the form input and its label. This form group element will be scrolled into view
 *                          so that both the input and its label are visible.
 */
export declare const focusInvalidForm: (formSelector: string, invalidElementSelector: string, formGroupSelector: string, focusOptions?: FocusOptions | undefined, scrollIntoViewOptions?: ScrollIntoViewOptions | undefined) => Promise<boolean>;
