/**
 * A CSS selector.
 */
export declare type Selector = string;
/**
 * A hash fragment.
 */
export declare type Hash = string;
/**
 * ARIA live region politeness values.
 * See https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/ARIA_Live_Regions
 */
export declare type AriaLivePoliteness = "off" | "polite" | "assertive";
/**
 * Specifies an element that is the target of a side effect (e.g. scroll into view, focus). This is
 * either the element itself or a selector that will return the element when passed to querySelector().
 */
export declare type Target = Element | Selector;
/**
 * The scroll position as x and y coordinates.
 */
export declare type ScrollPosition = {
    readonly x: number;
    readonly y: number;
};
/**
 * Maps from a URL hash fragment to a target element.
 *
 * Supports "#", "#top" and element IDs. The empty string returns undefined.
 *
 * Useful for scrolling to the element referred to by the hash fragment
 * in a URL (which browsers do natively, but single page apps often don't).
 *
 * See https://github.com/rafrex/react-router-hash-link (only manages scroll, not focus)
 * See https://github.com/ReactTraining/react-router/issues/394
 * See https://www.w3.org/TR/html5/single-page.html#scroll-to-the-fragment
 *
 * @param hash the hash fragment, including the leading # character, e.g. "#", "#top" or "#my-heading-id"
 */
export declare const elementFromHash: (hash: string) => Element | undefined;
/**
 * Set the document title.
 * See https://www.w3.org/TR/UNDERSTANDING-WCAG20/navigation-mechanisms-title.html
 *
 * @param title The new document title.
 */
export declare const setTitle: (title: string) => void;
/**
 * True if the specified element is within the viewport, false otherwise.
 * See https://stackoverflow.com/questions/123999/how-to-tell-if-a-dom-element-is-visible-in-the-current-viewport/7557433#7557433
 *
 * @param element the element to test
 */
export declare const isInViewport: (element: Element) => boolean;
export declare const elementFromTarget: (target: Target, parent?: ParentNode) => Element | undefined;
export declare const getScrollPosition: () => ScrollPosition;
/**
 * Scrolls the window to the given scroll position.
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * @param scrollPosition the scroll position to scroll to
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const setScrollPosition: (scrollPosition: ScrollPosition, smoothScroll?: boolean) => void;
/**
 * Focuses an element, setting `tabindex="-1"` if necessary.
 *
 * @param target the element to focus.
 * @param preventScroll true if the browser should not scroll the target element into view, false otherwise.
 */
export declare const focusElement: (target: Target, preventScroll?: boolean) => Promise<boolean>;
/**
 * True if the user prefers reduced motion, false otherwise.
 *
 * See https://css-tricks.com/introduction-reduced-motion-media-query/
 */
export declare const prefersReducedMotion: () => boolean;
/**
 * Scrolls an element into view.
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * @param element the element to scroll into view
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const scrollIntoView: (element: Element, smoothScroll?: boolean) => void;
/**
 * Scrolls an element into view if it is not currently visible.
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * @param target the element to scroll into view
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const scrollIntoViewIfRequired: (target: Target, smoothScroll?: boolean, inViewport?: (element: Element) => boolean) => void;
/**
 * Focuses a specified element and then scrolls it (or another element) into view (if required).
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * @param focusTarget the element to focus
 * @param scrollIntoViewTarget the element to scroll into view
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const focusAndScrollIntoViewIfRequired: (focusTarget: Target, scrollIntoViewTarget: Target, smoothScroll?: boolean) => Promise<boolean>;
/**
 * Resets focus and scroll position after a SPA page navigation.
 *
 * Will attempt to move focus to the focusTarget, primaryFocusTarget,
 * document element and finally document body, in that order. If any of
 * those elements do not exist or cannot be focused, will attempt to
 * focus the next fallback element.
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * See: https://github.com/ReactTraining/react-router/issues/5210
 *
 * @param primaryFocusTarget a CSS selector for your primary focus target,
 * e.g. `main h1`.
 * @param focusTarget the element to focus, e.g. the element identified by
 * the hash fragment of the URL.
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const resetFocus: (primaryFocusTarget: string, focusTarget?: string | Element | undefined, smoothScroll?: boolean) => Promise<boolean>;
/**
 * Make an announcement to screen reader users. Useful for page navigation events.
 *
 * See https://almerosteyn.com/2017/03/accessible-react-navigation
 * See https://getbootstrap.com/docs/4.3/utilities/screen-readers/
 * See https://github.com/twbs/bootstrap/blob/ff29c1224c20b8fcf2d1e7c28426470f1dc3e40d/scss/mixins/_screen-reader.scss#L6
 *
 * @param message the message to announce to screen reader users, e.g. "navigated to about page".
 * @param announcementsDivId a DOM ID of the visually hidden announcements element, e.g. "announcements".
 */
export declare const announce: (message: string, announcementsDivId?: string, setMessageTimeout?: number, clearMessageTimeout?: number, politeness?: AriaLivePoliteness) => void;
/**
 * Focuses and scrolls into view the first invalid form element inside
 * a given form.
 *
 * Call this function after you have validated a form and identified errors.
 *
 * See https://webaim.org/techniques/formvalidation/
 *
 * For smooth scrolling behavior you might want to use the smoothscroll
 * polyfill http://iamdustan.com/smoothscroll/
 *
 * If the user has indicated that they prefer reduced motion, the smoothScroll value will be ignored.
 *
 * For IE support you might want to use the closest() polyfill from https://developer.mozilla.org/en-US/docs/Web/API/Element/closest#Polyfill
 *
 * @param formSelector a CSS selector that uniquely identifies the form to focus, e.g. `#search-form`.
 * @param invalidElementSelector the CSS selector that is used to identify invalid elements within a form, e.g. `[aria-invalid="true"]`.
 * @param formGroupSelector a CSS selector passed to the `closest()` method of an invalid form input that identifies the element
 *                          that contains both the form input and its label. This form group element will be scrolled into view
 *                          so that both the input and its label are visible.
 * @param smoothScroll true for smooth scrolling, false otherwise
 */
export declare const focusInvalidForm: (formSelector: string, invalidElementSelector: string, formGroupSelector: string, smoothScroll?: boolean) => Promise<boolean>;
