'use strict';

describe('AccessToken', function() {

  var result, $location, Storage, IdToken, AccessToken, date;

  var fragment = 'access_token=token&token_type=bearer&expires_in=7200&state=/path&extra=stuff';
  var fragmentForever = 'access_token=token&token_type=bearer&state=/path&extra=stuff';
  var fragmentImmediate = 'access_token=token&token_type=bearer&expires_in=0&state=/path&extra=stuff';
  var fragmentWithIdToken = 'access_token=token&token_type=bearer'+
      '&id_token=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiYWRtaW4iOnRydWV9.TJVA95OrM7E2cBab30RMHrHDcEfxjoYZgeFONFh7HgQ&expires_in=3600';
  var denied   = 'error=access_denied&error_description=error';
  var expires_at = '2014-08-17T17:38:37.584Z';
  var newExpiresIn = 9600;
  var token    = { access_token: 'token', token_type: 'bearer', expires_in: 7200, state: '/path', expires_at: expires_at };

  beforeEach(module('oauth'));

  beforeEach(inject(function($injector) { $location = $injector.get('$location'); }));
  beforeEach(inject(function($injector) { Storage = $injector.get('Storage'); }));
  beforeEach(inject(function($injector) { IdToken = $injector.get('IdToken'); }));
  beforeEach(inject(function($injector) { AccessToken = $injector.get('AccessToken'); }));


  describe('#set', function() {

    describe('when sets the access token', function() {

      beforeEach(function() {
        $location.hash(fragment);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets the access token', function() {
        expect(result.access_token).toEqual('token');
      });

      it('sets #expires_at', function() {
        var expected_date = new Date();
        expected_date.setSeconds(expected_date.getSeconds() + 7200 - 60);
        expect(parseInt(result.expires_at/100)).toBe(parseInt(expected_date/100)); // 10 ms
      });
    });

    describe('when sets the access token without an expiry', function() {

      beforeEach(function() {
        $location.hash(fragmentForever);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets #expires_at', function() {
        expect(result.expires_at).toBe(null);
      });
    });

    describe('when sets the access token with an expiry of 0 (immediate)', function() {

      beforeEach(function() {
        $location.hash(fragmentImmediate);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets #expires_at', function() {
        var expected_date = new Date();
        expected_date.setSeconds(expected_date.getSeconds() - 60);
        expect(parseInt(result.expires_at/100)).toBe(parseInt(expected_date/100)); // 10 ms
      });
    });

    describe('with the access token in the fragment URI', function() {

      beforeEach(function() {
        $location.hash(fragment);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets the access token', function() {
        expect(result.access_token).toEqual('token');
      });

      it('removes the fragment string', function() {
        expect($location.hash()).toEqual('extra=stuff');
      });

      it('stores the token in the session', function() {
        Storage.get('token');
        expect(result.access_token).toEqual('token');
      });
    });

    describe('with the access token stored in the session', function() {

      beforeEach(function() {
        Storage.set('token', token);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets the access token from session', function() {
        expect(result.access_token).toEqual('token');
      });
    });

    describe('with the denied message in the fragment URI', function() {

      beforeEach(function() {
        $location.hash(denied);
      });

      beforeEach(function() {
        result = AccessToken.set();
      });

      it('sets the access token', function() {
        expect(result.error).toEqual('access_denied');
      });

      it('removes the fragment string', function() {
        expect($location.hash()).toEqual('');
      });

      it('stores the error message in the session', function() {
        Storage.get('token');
        expect(result.error).toBe('access_denied');
      });
    });

    describe('with id token and access token in the fragment', function() {
      beforeEach(function() {
        $location.hash(fragmentWithIdToken);
      });

      describe('when all validation passes', function() {
        beforeEach(function() {
          //specific validation mechanism are tested in id-token spec
          spyOn(IdToken, 'validateIdToken').and.returnValue(true);
          spyOn(IdToken, 'validateAccessToken').and.returnValue(true);
          result = AccessToken.set();
        });

        it('sets the access token', function() {
          expect(result.access_token).toEqual('token');
        });

        it('sets the id token', function() {
          expect(result.id_token).toEqual('eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiYWRtaW4iOnRydWV9.TJVA95OrM7E2cBab30RMHrHDcEfxjoYZgeFONFh7HgQ');
        });

        it('populates id token claims', function() {
          expect(result.id_token_claims).toBeDefined();
          expect(result.id_token_claims.name).toEqual('John Doe');
        });
      });

      describe('when id token validation fails', function(){
        beforeEach(function() {
          //specific validation mechanism are tested in id-token spec
          spyOn(IdToken, 'validateIdToken').and.returnValue(false);
          spyOn(IdToken, 'validateAccessToken').and.returnValue(true);
          result = AccessToken.set();
        });

        it('erases access token and id token', function() {
          expect(result.access_token).toEqual(null);
          expect(result.id_token).toEqual(null);
        });

        it('sets the error', function() {
          expect(result.error).toEqual('Failed to validate token:');
        })
      });

      describe('when access token validation fails', function(){
        beforeEach(function() {
          //specific validation mechanism are tested in id-token spec
          spyOn(IdToken, 'validateIdToken').and.returnValue(true);
          spyOn(IdToken, 'validateAccessToken').and.returnValue(false);
          result = AccessToken.set();
        });

        it('erases access token and id token', function() {
          expect(result.access_token).toEqual(null);
          expect(result.id_token).toEqual(null);
        });

        it('sets the error', function() {
          expect(result.error).toEqual('Failed to validate token:');
        })
      });

    })
  });


  describe('#get', function() {

    beforeEach(function() {
      $location.hash(fragment);
    });

    beforeEach(function() {
      AccessToken.set();
    });

    beforeEach(function() {
      result = AccessToken.get();
    });

    it('sets the access token', function() {
      expect(result.access_token).toEqual('token');
    });
  });


  describe('#destroy', function() {

    beforeEach(function() {
      $location.hash(fragment);
    });

    beforeEach(function() {
      AccessToken.set();
    });

    beforeEach(function() {
      result = AccessToken.destroy();
    });

    it('sets the access token', function() {
      expect(result).toBeNull();
    });

    it('reset the cache', function() {
      expect(Storage.get('token')).toBeUndefined();
    });
  });


  describe('#expired', function() {

    beforeEach(function() {
      $location.hash(fragment);
    });

    beforeEach(function() {
      AccessToken.set();
    });

    describe('when not expired', function() {

      beforeEach(function() {
        result = AccessToken.expired();
      });

      it('returns false', function() {
        expect(result).toBe(false);
      });
    });

    describe('when expired', function() {

      beforeEach(function() {
        date = new Date();
        date.setTime(date.getTime() + 86400000);
      });

      beforeEach(function() {
        Timecop.install();
        Timecop.travel(date); // go to the future (one day)
      });

      beforeEach(function() {
        result = AccessToken.expired();
      });

      afterEach(function() {
        Timecop.uninstall();
      });

      it('returns true', function() {
        expect(result).toBe(true);
      });
    });
  });

  describe('#updateExpiry', function () {
    beforeEach(function () {
      $location.hash('');
      Storage.set('token', token);
      AccessToken.set();
    });


    it('updates the expiry to a new time', function () {
      AccessToken.updateExpiry(newExpiresIn);
      expect(AccessToken.expired()).toBeFalsy();
      var newExpiresAt = new Date();
      newExpiresAt.setSeconds(newExpiresAt.getSeconds() + newExpiresIn - 60);
      expect(AccessToken.get().expires_at).toEqual(newExpiresAt);
    });
  });

  describe('#sessionExpired', function() {
    describe('with the access token stored in the session', function() {

      beforeEach(function() {
        //It is an invalid test to have oAuth hash AND be getting token from session
        //if hash is in URL it should always be used, cuz its coming from oAuth2 provider re-direct
        $location.hash('');
        Storage.set('token', token);
        result = AccessToken.set().expires_at;
      });

      it('rehydrates the expires_at value', function() {
        expect(result).toEqual(expires_at);
      });
    });
  });
});
