"use strict";

exports.generateRequestFnFromLinks = generateRequestFnFromLinks;
exports.sendRequest = sendRequest;

var _transactions = require("./transactions");

var _http = require("../http");

var _errors = require("../errors");

const parseInsufficientAuthenticationError = header => {
  if (!header) {
    throw new _errors.AuthSdkError('Missing header string');
  }

  return header.split(',').map(part => part.trim()).map(part => part.split('=')).reduce((acc, curr) => {
    // unwrap quotes from value
    acc[curr[0]] = curr[1].replace(/^"(.*)"$/, '$1');
    return acc;
  }, {});
};
/* eslint-disable complexity */


async function sendRequest(oktaAuth, options) {
  const {
    accessToken: accessTokenObj,
    idToken: idTokenObj
  } = oktaAuth.tokenManager.getTokensSync();
  const idToken = idTokenObj === null || idTokenObj === void 0 ? void 0 : idTokenObj.idToken;
  const accessToken = options.accessToken || (accessTokenObj === null || accessTokenObj === void 0 ? void 0 : accessTokenObj.accessToken);
  const {
    issuer
  } = oktaAuth.options;
  const {
    url,
    method,
    payload
  } = options;
  const requestUrl = url.startsWith(issuer) ? url : `${issuer}${url}`;

  if (!accessToken) {
    throw new _errors.AuthSdkError('AccessToken is required to request MyAccount API endpoints.');
  }

  let res;

  try {
    res = await (0, _http.httpRequest)(oktaAuth, {
      headers: {
        'Accept': '*/*;okta-version=1.0.0'
      },
      accessToken,
      url: requestUrl,
      method,
      ...(payload && {
        args: payload
      })
    });
  } catch (err) {
    var _errorResp$headers;

    const errorResp = err.xhr;

    if (idToken && (errorResp === null || errorResp === void 0 ? void 0 : errorResp.status) === 403 && !!(errorResp !== null && errorResp !== void 0 && (_errorResp$headers = errorResp.headers) !== null && _errorResp$headers !== void 0 && _errorResp$headers['www-authenticate'])) {
      var _errorResp$headers2;

      const {
        error,
        // eslint-disable-next-line camelcase
        error_description,
        // eslint-disable-next-line camelcase
        max_age
      } = parseInsufficientAuthenticationError(errorResp === null || errorResp === void 0 ? void 0 : (_errorResp$headers2 = errorResp.headers) === null || _errorResp$headers2 === void 0 ? void 0 : _errorResp$headers2['www-authenticate']);

      if (error === 'insufficient_authentication_context') {
        const insufficientAuthenticationError = new _errors.AuthApiError({
          errorSummary: error,
          // eslint-disable-next-line camelcase
          errorCauses: [{
            errorSummary: error_description
          }]
        }, errorResp, // eslint-disable-next-line camelcase
        {
          max_age: +max_age
        });
        throw insufficientAuthenticationError;
      } else {
        throw err;
      }
    } else {
      throw err;
    }
  }

  const map = {
    EmailTransaction: _transactions.EmailTransaction,
    EmailStatusTransaction: _transactions.EmailStatusTransaction,
    EmailChallengeTransaction: _transactions.EmailChallengeTransaction,
    ProfileTransaction: _transactions.ProfileTransaction,
    ProfileSchemaTransaction: _transactions.ProfileSchemaTransaction,
    PhoneTransaction: _transactions.PhoneTransaction
  };
  const TransactionClass = map[options.transactionClassName] || _transactions.BaseTransaction;

  if (Array.isArray(res)) {
    return res.map(item => new TransactionClass(oktaAuth, {
      res: item,
      accessToken
    }));
  }

  return new TransactionClass(oktaAuth, {
    res,
    accessToken
  });
}
/* eslint-enable complexity */


function generateRequestFnFromLinks({
  oktaAuth,
  accessToken,
  methodName,
  links,
  transactionClassName
}) {
  for (const method of ['GET', 'POST', 'PUT', 'DELETE']) {
    if (method.toLowerCase() === methodName) {
      const link = links.self;
      return async payload => sendRequest(oktaAuth, {
        accessToken,
        url: link.href,
        method,
        payload,
        transactionClassName
      });
    }
  }

  const link = links[methodName];

  if (!link) {
    throw new _errors.AuthSdkError(`No link is found with methodName: ${methodName}`);
  }

  return async payload => sendRequest(oktaAuth, {
    accessToken,
    url: link.href,
    method: link.hints.allow[0],
    payload,
    transactionClassName
  });
}
//# sourceMappingURL=request.js.map