/*!
 * Copyright (c) 2015-present, Okta, Inc. and/or its affiliates. All rights reserved.
 * The Okta software accompanied by this notice is provided pursuant to the Apache License, Version 2.0 (the "License.")
 *
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 */

import { removeNils } from '../../util/object.js';
import { isString } from '../../util/types.js';
import { toQueryString } from '../../util/url.js';
import AuthSdkError from '../../errors/AuthSdkError.js';

function convertTokenParamsToOAuthParams(tokenParams) {
    if (!tokenParams.clientId) {
        throw new AuthSdkError('A clientId must be specified in the OktaAuth constructor to get a token');
    }
    if (isString(tokenParams.responseType) && tokenParams.responseType.indexOf(' ') !== -1) {
        throw new AuthSdkError('Multiple OAuth responseTypes must be defined as an array');
    }
    var oauthParams = {
        'client_id': tokenParams.clientId,
        'code_challenge': tokenParams.codeChallenge,
        'code_challenge_method': tokenParams.codeChallengeMethod,
        'display': tokenParams.display,
        'idp': tokenParams.idp,
        'idp_scope': tokenParams.idpScope,
        'login_hint': tokenParams.loginHint,
        'max_age': tokenParams.maxAge,
        'nonce': tokenParams.nonce,
        'prompt': tokenParams.prompt,
        'redirect_uri': tokenParams.redirectUri,
        'response_mode': tokenParams.responseMode,
        'response_type': tokenParams.responseType,
        'sessionToken': tokenParams.sessionToken,
        'state': tokenParams.state,
    };
    oauthParams = removeNils(oauthParams);
    ['idp_scope', 'response_type'].forEach(function (mayBeArray) {
        if (Array.isArray(oauthParams[mayBeArray])) {
            oauthParams[mayBeArray] = oauthParams[mayBeArray].join(' ');
        }
    });
    if (tokenParams.responseType.indexOf('id_token') !== -1 &&
        tokenParams.scopes.indexOf('openid') === -1) {
        throw new AuthSdkError('openid scope must be specified in the scopes argument when requesting an id_token');
    }
    else {
        oauthParams.scope = tokenParams.scopes.join(' ');
    }
    return oauthParams;
}
function buildAuthorizeParams(tokenParams) {
    var oauthQueryParams = convertTokenParamsToOAuthParams(tokenParams);
    return toQueryString(Object.assign(Object.assign({}, oauthQueryParams), (tokenParams.extraParams && Object.assign({}, tokenParams.extraParams))));
}

export { buildAuthorizeParams, convertTokenParamsToOAuthParams };
//# sourceMappingURL=authorize.js.map
