/**
 * @module ol/geom/Polygon
 */
import {extend} from '../array.js';
import {closestSquaredDistanceXY, getCenter} from '../extent.js';
import GeometryLayout from '../geom/GeometryLayout.js';
import GeometryType from '../geom/GeometryType.js';
import LinearRing from '../geom/LinearRing.js';
import Point from '../geom/Point.js';
import SimpleGeometry from '../geom/SimpleGeometry.js';
import {offset as sphereOffset} from '../sphere.js';
import {linearRings as linearRingsArea} from '../geom/flat/area.js';
import {assignClosestArrayPoint, arrayMaxSquaredDelta} from '../geom/flat/closest.js';
import {linearRingsContainsXY} from '../geom/flat/contains.js';
import {deflateCoordinatesArray} from '../geom/flat/deflate.js';
import {inflateCoordinatesArray} from '../geom/flat/inflate.js';
import {getInteriorPointOfArray} from '../geom/flat/interiorpoint.js';
import {intersectsLinearRingArray} from '../geom/flat/intersectsextent.js';
import {linearRingIsOriented, orientLinearRings} from '../geom/flat/orient.js';
import {quantizeArray} from '../geom/flat/simplify.js';
import {modulo} from '../math.js';

/**
 * @classdesc
 * Polygon geometry.
 *
 * @api
 */
var Polygon = (function (SimpleGeometry) {
  function Polygon(coordinates, opt_layout, opt_ends) {

    SimpleGeometry.call(this);

    /**
     * @type {Array<number>}
     * @private
     */
    this.ends_ = [];

    /**
     * @private
     * @type {number}
     */
    this.flatInteriorPointRevision_ = -1;

    /**
     * @private
     * @type {module:ol/coordinate~Coordinate}
     */
    this.flatInteriorPoint_ = null;

    /**
     * @private
     * @type {number}
     */
    this.maxDelta_ = -1;

    /**
     * @private
     * @type {number}
     */
    this.maxDeltaRevision_ = -1;

    /**
     * @private
     * @type {number}
     */
    this.orientedRevision_ = -1;

    /**
     * @private
     * @type {Array<number>}
     */
    this.orientedFlatCoordinates_ = null;

    if (opt_layout !== undefined && opt_ends) {
      this.setFlatCoordinates(opt_layout, coordinates);
      this.ends_ = opt_ends;
    } else {
      this.setCoordinates(coordinates, opt_layout);
    }

  }

  if ( SimpleGeometry ) Polygon.__proto__ = SimpleGeometry;
  Polygon.prototype = Object.create( SimpleGeometry && SimpleGeometry.prototype );
  Polygon.prototype.constructor = Polygon;

  /**
   * Append the passed linear ring to this polygon.
   * @param {module:ol/geom/LinearRing} linearRing Linear ring.
   * @api
   */
  Polygon.prototype.appendLinearRing = function appendLinearRing (linearRing) {
    if (!this.flatCoordinates) {
      this.flatCoordinates = linearRing.getFlatCoordinates().slice();
    } else {
      extend(this.flatCoordinates, linearRing.getFlatCoordinates());
    }
    this.ends_.push(this.flatCoordinates.length);
    this.changed();
  };

  /**
   * Make a complete copy of the geometry.
   * @return {!module:ol/geom/Polygon} Clone.
   * @override
   * @api
   */
  Polygon.prototype.clone = function clone () {
    return new Polygon(this.flatCoordinates.slice(), this.layout, this.ends_.slice());
  };

  /**
   * @inheritDoc
   */
  Polygon.prototype.closestPointXY = function closestPointXY (x, y, closestPoint, minSquaredDistance) {
    if (minSquaredDistance < closestSquaredDistanceXY(this.getExtent(), x, y)) {
      return minSquaredDistance;
    }
    if (this.maxDeltaRevision_ != this.getRevision()) {
      this.maxDelta_ = Math.sqrt(arrayMaxSquaredDelta(
        this.flatCoordinates, 0, this.ends_, this.stride, 0));
      this.maxDeltaRevision_ = this.getRevision();
    }
    return assignClosestArrayPoint(
      this.flatCoordinates, 0, this.ends_, this.stride,
      this.maxDelta_, true, x, y, closestPoint, minSquaredDistance);
  };

  /**
   * @inheritDoc
   */
  Polygon.prototype.containsXY = function containsXY (x, y) {
    return linearRingsContainsXY(this.getOrientedFlatCoordinates(), 0, this.ends_, this.stride, x, y);
  };

  /**
   * Return the area of the polygon on projected plane.
   * @return {number} Area (on projected plane).
   * @api
   */
  Polygon.prototype.getArea = function getArea () {
    return linearRingsArea(this.getOrientedFlatCoordinates(), 0, this.ends_, this.stride);
  };

  /**
   * Get the coordinate array for this geometry.  This array has the structure
   * of a GeoJSON coordinate array for polygons.
   *
   * @param {boolean=} opt_right Orient coordinates according to the right-hand
   *     rule (counter-clockwise for exterior and clockwise for interior rings).
   *     If `false`, coordinates will be oriented according to the left-hand rule
   *     (clockwise for exterior and counter-clockwise for interior rings).
   *     By default, coordinate orientation will depend on how the geometry was
   *     constructed.
   * @return {Array<Array<module:ol/coordinate~Coordinate>>} Coordinates.
   * @override
   * @api
   */
  Polygon.prototype.getCoordinates = function getCoordinates (opt_right) {
    var flatCoordinates;
    if (opt_right !== undefined) {
      flatCoordinates = this.getOrientedFlatCoordinates().slice();
      orientLinearRings(
        flatCoordinates, 0, this.ends_, this.stride, opt_right);
    } else {
      flatCoordinates = this.flatCoordinates;
    }

    return inflateCoordinatesArray(
      flatCoordinates, 0, this.ends_, this.stride);
  };

  /**
   * @return {Array<number>} Ends.
   */
  Polygon.prototype.getEnds = function getEnds () {
    return this.ends_;
  };

  /**
   * @return {Array<number>} Interior point.
   */
  Polygon.prototype.getFlatInteriorPoint = function getFlatInteriorPoint () {
    if (this.flatInteriorPointRevision_ != this.getRevision()) {
      var flatCenter = getCenter(this.getExtent());
      this.flatInteriorPoint_ = getInteriorPointOfArray(
        this.getOrientedFlatCoordinates(), 0, this.ends_, this.stride,
        flatCenter, 0);
      this.flatInteriorPointRevision_ = this.getRevision();
    }
    return this.flatInteriorPoint_;
  };

  /**
   * Return an interior point of the polygon.
   * @return {module:ol/geom/Point} Interior point as XYM coordinate, where M is the
   * length of the horizontal intersection that the point belongs to.
   * @api
   */
  Polygon.prototype.getInteriorPoint = function getInteriorPoint () {
    return new Point(this.getFlatInteriorPoint(), GeometryLayout.XYM);
  };

  /**
   * Return the number of rings of the polygon,  this includes the exterior
   * ring and any interior rings.
   *
   * @return {number} Number of rings.
   * @api
   */
  Polygon.prototype.getLinearRingCount = function getLinearRingCount () {
    return this.ends_.length;
  };

  /**
   * Return the Nth linear ring of the polygon geometry. Return `null` if the
   * given index is out of range.
   * The exterior linear ring is available at index `0` and the interior rings
   * at index `1` and beyond.
   *
   * @param {number} index Index.
   * @return {module:ol/geom/LinearRing} Linear ring.
   * @api
   */
  Polygon.prototype.getLinearRing = function getLinearRing (index) {
    if (index < 0 || this.ends_.length <= index) {
      return null;
    }
    return new LinearRing(this.flatCoordinates.slice(
      index === 0 ? 0 : this.ends_[index - 1], this.ends_[index]), this.layout);
  };

  /**
   * Return the linear rings of the polygon.
   * @return {Array<module:ol/geom/LinearRing>} Linear rings.
   * @api
   */
  Polygon.prototype.getLinearRings = function getLinearRings () {
    var layout = this.layout;
    var flatCoordinates = this.flatCoordinates;
    var ends = this.ends_;
    var linearRings = [];
    var offset = 0;
    for (var i = 0, ii = ends.length; i < ii; ++i) {
      var end = ends[i];
      var linearRing = new LinearRing(flatCoordinates.slice(offset, end), layout);
      linearRings.push(linearRing);
      offset = end;
    }
    return linearRings;
  };

  /**
   * @return {Array<number>} Oriented flat coordinates.
   */
  Polygon.prototype.getOrientedFlatCoordinates = function getOrientedFlatCoordinates () {
    if (this.orientedRevision_ != this.getRevision()) {
      var flatCoordinates = this.flatCoordinates;
      if (linearRingIsOriented(
        flatCoordinates, 0, this.ends_, this.stride)) {
        this.orientedFlatCoordinates_ = flatCoordinates;
      } else {
        this.orientedFlatCoordinates_ = flatCoordinates.slice();
        this.orientedFlatCoordinates_.length =
            orientLinearRings(
              this.orientedFlatCoordinates_, 0, this.ends_, this.stride);
      }
      this.orientedRevision_ = this.getRevision();
    }
    return this.orientedFlatCoordinates_;
  };

  /**
   * @inheritDoc
   */
  Polygon.prototype.getSimplifiedGeometryInternal = function getSimplifiedGeometryInternal (squaredTolerance) {
    var simplifiedFlatCoordinates = [];
    var simplifiedEnds = [];
    simplifiedFlatCoordinates.length = quantizeArray(
      this.flatCoordinates, 0, this.ends_, this.stride,
      Math.sqrt(squaredTolerance),
      simplifiedFlatCoordinates, 0, simplifiedEnds);
    return new Polygon(simplifiedFlatCoordinates, GeometryLayout.XY, simplifiedEnds);
  };

  /**
   * @inheritDoc
   * @api
   */
  Polygon.prototype.getType = function getType () {
    return GeometryType.POLYGON;
  };

  /**
   * @inheritDoc
   * @api
   */
  Polygon.prototype.intersectsExtent = function intersectsExtent (extent) {
    return intersectsLinearRingArray(
      this.getOrientedFlatCoordinates(), 0, this.ends_, this.stride, extent);
  };

  /**
   * Set the coordinates of the polygon.
   * @param {!Array<Array<module:ol/coordinate~Coordinate>>} coordinates Coordinates.
   * @param {module:ol/geom/GeometryLayout=} opt_layout Layout.
   * @override
   * @api
   */
  Polygon.prototype.setCoordinates = function setCoordinates (coordinates, opt_layout) {
    this.setLayout(opt_layout, coordinates, 2);
    if (!this.flatCoordinates) {
      this.flatCoordinates = [];
    }
    var ends = deflateCoordinatesArray(
      this.flatCoordinates, 0, coordinates, this.stride, this.ends_);
    this.flatCoordinates.length = ends.length === 0 ? 0 : ends[ends.length - 1];
    this.changed();
  };

  return Polygon;
}(SimpleGeometry));


export default Polygon;


/**
 * Create an approximation of a circle on the surface of a sphere.
 * @param {module:ol/coordinate~Coordinate} center Center (`[lon, lat]` in degrees).
 * @param {number} radius The great-circle distance from the center to
 *     the polygon vertices.
 * @param {number=} opt_n Optional number of vertices for the resulting
 *     polygon. Default is `32`.
 * @param {number=} opt_sphereRadius Optional radius for the sphere (defaults to
 *     the Earth's mean radius using the WGS84 ellipsoid).
 * @return {module:ol/geom/Polygon} The "circular" polygon.
 * @api
 */
export function circular(center, radius, opt_n, opt_sphereRadius) {
  var n = opt_n ? opt_n : 32;
  /** @type {Array<number>} */
  var flatCoordinates = [];
  for (var i = 0; i < n; ++i) {
    extend(flatCoordinates, sphereOffset(center, radius, 2 * Math.PI * i / n, opt_sphereRadius));
  }
  flatCoordinates.push(flatCoordinates[0], flatCoordinates[1]);
  return new Polygon(flatCoordinates, GeometryLayout.XY, [flatCoordinates.length]);
}


/**
 * Create a polygon from an extent. The layout used is `XY`.
 * @param {module:ol/extent~Extent} extent The extent.
 * @return {module:ol/geom/Polygon} The polygon.
 * @api
 */
export function fromExtent(extent) {
  var minX = extent[0];
  var minY = extent[1];
  var maxX = extent[2];
  var maxY = extent[3];
  var flatCoordinates =
      [minX, minY, minX, maxY, maxX, maxY, maxX, minY, minX, minY];
  return new Polygon(flatCoordinates, GeometryLayout.XY, [flatCoordinates.length]);
}


/**
 * Create a regular polygon from a circle.
 * @param {module:ol/geom/Circle} circle Circle geometry.
 * @param {number=} opt_sides Number of sides of the polygon. Default is 32.
 * @param {number=} opt_angle Start angle for the first vertex of the polygon in
 *     radians. Default is 0.
 * @return {module:ol/geom/Polygon} Polygon geometry.
 * @api
 */
export function fromCircle(circle, opt_sides, opt_angle) {
  var sides = opt_sides ? opt_sides : 32;
  var stride = circle.getStride();
  var layout = circle.getLayout();
  var center = circle.getCenter();
  var arrayLength = stride * (sides + 1);
  var flatCoordinates = new Array(arrayLength);
  for (var i = 0; i < arrayLength; i += stride) {
    flatCoordinates[i] = 0;
    flatCoordinates[i + 1] = 0;
    for (var j = 2; j < stride; j++) {
      flatCoordinates[i + j] = center[j];
    }
  }
  var ends = [flatCoordinates.length];
  var polygon = new Polygon(flatCoordinates, layout, ends);
  makeRegular(polygon, center, circle.getRadius(), opt_angle);
  return polygon;
}


/**
 * Modify the coordinates of a polygon to make it a regular polygon.
 * @param {module:ol/geom/Polygon} polygon Polygon geometry.
 * @param {module:ol/coordinate~Coordinate} center Center of the regular polygon.
 * @param {number} radius Radius of the regular polygon.
 * @param {number=} opt_angle Start angle for the first vertex of the polygon in
 *     radians. Default is 0.
 */
export function makeRegular(polygon, center, radius, opt_angle) {
  var flatCoordinates = polygon.getFlatCoordinates();
  var stride = polygon.getStride();
  var sides = flatCoordinates.length / stride - 1;
  var startAngle = opt_angle ? opt_angle : 0;
  for (var i = 0; i <= sides; ++i) {
    var offset = i * stride;
    var angle = startAngle + (modulo(i, sides) * 2 * Math.PI / sides);
    flatCoordinates[offset] = center[0] + (radius * Math.cos(angle));
    flatCoordinates[offset + 1] = center[1] + (radius * Math.sin(angle));
  }
  polygon.changed();
}

//# sourceMappingURL=Polygon.js.map