/**
 * @module ol/interaction/DragZoom
 */
import {easeOut} from '../easing.js';
import {shiftKeyOnly} from '../events/condition.js';
import {createOrUpdateFromCoordinates, getBottomLeft, getCenter, getTopRight, scaleFromCenter} from '../extent.js';
import DragBox from '../interaction/DragBox.js';


/**
 * @typedef {Object} Options
 * @property {string} [className='ol-dragzoom'] CSS class name for styling the
 * box.
 * @property {module:ol/events/condition~Condition} [condition] A function that
 * takes an {@link module:ol/MapBrowserEvent~MapBrowserEvent} and returns a
 * boolean to indicate whether that event should be handled.
 * Default is {@link module:ol/events/condition~shiftKeyOnly}.
 * @property {number} [duration=200] Animation duration in milliseconds.
 * @property {boolean} [out=false] Use interaction for zooming out.
 */


/**
 * @classdesc
 * Allows the user to zoom the map by clicking and dragging on the map,
 * normally combined with an {@link module:ol/events/condition} that limits
 * it to when a key, shift by default, is held down.
 *
 * To change the style of the box, use CSS and the `.ol-dragzoom` selector, or
 * your custom one configured with `className`.
 * @api
 */
var DragZoom = (function (DragBox) {
  function DragZoom(opt_options) {
    var options = opt_options ? opt_options : {};

    var condition = options.condition ? options.condition : shiftKeyOnly;

    DragBox.call(this, {
      condition: condition,
      className: options.className || 'ol-dragzoom',
      onBoxEnd: onBoxEnd
    });

    /**
     * @private
     * @type {number}
     */
    this.duration_ = options.duration !== undefined ? options.duration : 200;

    /**
     * @private
     * @type {boolean}
     */
    this.out_ = options.out !== undefined ? options.out : false;
  }

  if ( DragBox ) DragZoom.__proto__ = DragBox;
  DragZoom.prototype = Object.create( DragBox && DragBox.prototype );
  DragZoom.prototype.constructor = DragZoom;

  return DragZoom;
}(DragBox));


/**
 * @this {module:ol/interaction/DragZoom}
 */
function onBoxEnd() {
  var map = this.getMap();
  var view = /** @type {!module:ol/View} */ (map.getView());
  var size = /** @type {!module:ol/size~Size} */ (map.getSize());
  var extent = this.getGeometry().getExtent();

  if (this.out_) {
    var mapExtent = view.calculateExtent(size);
    var boxPixelExtent = createOrUpdateFromCoordinates([
      map.getPixelFromCoordinate(getBottomLeft(extent)),
      map.getPixelFromCoordinate(getTopRight(extent))]);
    var factor = view.getResolutionForExtent(boxPixelExtent, size);

    scaleFromCenter(mapExtent, 1 / factor);
    extent = mapExtent;
  }

  var resolution = view.constrainResolution(
    view.getResolutionForExtent(extent, size));

  var center = getCenter(extent);
  center = view.constrainCenter(center);

  view.animate({
    resolution: resolution,
    center: center,
    duration: this.duration_,
    easing: easeOut
  });
}


export default DragZoom;

//# sourceMappingURL=DragZoom.js.map