/**
 * @module ol/render/canvas
 */
import {getFontFamilies} from '../css.js';
import {createCanvasContext2D} from '../dom.js';
import {clear} from '../obj.js';
import LRUCache from '../structs/LRUCache.js';
import {create as createTransform} from '../transform.js';


/**
 * @typedef {Object} FillState
 * @property {module:ol/colorlike~ColorLike} fillStyle
 */


/**
 * @typedef {Object} FillStrokeState
 * @property {module:ol/colorlike~ColorLike} [currentFillStyle]
 * @property {module:ol/colorlike~ColorLike} [currentStrokeStyle]
 * @property {string} [currentLineCap]
 * @property {Array<number>} currentLineDash
 * @property {number} [currentLineDashOffset]
 * @property {string} [currentLineJoin]
 * @property {number} [currentLineWidth]
 * @property {number} [currentMiterLimit]
 * @property {number} [lastStroke]
 * @property {module:ol/colorlike~ColorLike} [fillStyle]
 * @property {module:ol/colorlike~ColorLike} [strokeStyle]
 * @property {string} [lineCap]
 * @property {Array<number>} lineDash
 * @property {number} [lineDashOffset]
 * @property {string} [lineJoin]
 * @property {number} [lineWidth]
 * @property {number} [miterLimit]
 */


/**
 * @typedef {Object} StrokeState
 * @property {string} lineCap
 * @property {Array<number>} lineDash
 * @property {number} lineDashOffset
 * @property {string} lineJoin
 * @property {number} lineWidth
 * @property {number} miterLimit
 * @property {module:ol/colorlike~ColorLike} strokeStyle
 */


/**
 * @typedef {Object} TextState
 * @property {string} font
 * @property {string} [textAlign]
 * @property {string} textBaseline
 */


/**
 * Container for decluttered replay instructions that need to be rendered or
 * omitted together, i.e. when styles render both an image and text, or for the
 * characters that form text along lines. The basic elements of this array are
 * `[minX, minY, maxX, maxY, count]`, where the first four entries are the
 * rendered extent of the group in pixel space. `count` is the number of styles
 * in the group, i.e. 2 when an image and a text are grouped, or 1 otherwise.
 * In addition to these four elements, declutter instruction arrays (i.e. the
 * arguments to {@link module:ol/render/canvas~drawImage} are appended to the array.
 * @typedef {Array<*>} DeclutterGroup
 */


/**
 * @const
 * @type {string}
 */
export var defaultFont = '10px sans-serif';


/**
 * @const
 * @type {module:ol/color~Color}
 */
export var defaultFillStyle = [0, 0, 0, 1];


/**
 * @const
 * @type {string}
 */
export var defaultLineCap = 'round';


/**
 * @const
 * @type {Array<number>}
 */
export var defaultLineDash = [];


/**
 * @const
 * @type {number}
 */
export var defaultLineDashOffset = 0;


/**
 * @const
 * @type {string}
 */
export var defaultLineJoin = 'round';


/**
 * @const
 * @type {number}
 */
export var defaultMiterLimit = 10;


/**
 * @const
 * @type {module:ol/color~Color}
 */
export var defaultStrokeStyle = [0, 0, 0, 1];


/**
 * @const
 * @type {string}
 */
export var defaultTextAlign = 'center';


/**
 * @const
 * @type {string}
 */
export var defaultTextBaseline = 'middle';


/**
 * @const
 * @type {Array<number>}
 */
export var defaultPadding = [0, 0, 0, 0];


/**
 * @const
 * @type {number}
 */
export var defaultLineWidth = 1;


/**
 * The label cache for text rendering. To change the default cache size of 2048
 * entries, use {@link module:ol/structs/LRUCache#setSize}.
 * @type {module:ol/structs/LRUCache<HTMLCanvasElement>}
 * @api
 */
export var labelCache = new LRUCache();


/**
 * @type {!Object<string, number>}
 */
export var checkedFonts = {};


/**
 * @type {CanvasRenderingContext2D}
 */
var measureContext = null;


/**
 * @type {!Object<string, number>}
 */
export var textHeights = {};


/**
 * Clears the label cache when a font becomes available.
 * @param {string} fontSpec CSS font spec.
 */
export var checkFont = (function() {
  var retries = 60;
  var checked = checkedFonts;
  var size = '32px ';
  var referenceFonts = ['monospace', 'serif'];
  var len = referenceFonts.length;
  var text = 'wmytzilWMYTZIL@#/&?$%10\uF013';
  var interval, referenceWidth;

  function isAvailable(font) {
    var context = getMeasureContext();
    // Check weight ranges according to
    // https://developer.mozilla.org/en-US/docs/Web/CSS/font-weight#Fallback_weights
    for (var weight = 100; weight <= 700; weight += 300) {
      var fontWeight = weight + ' ';
      var available = true;
      for (var i = 0; i < len; ++i) {
        var referenceFont = referenceFonts[i];
        context.font = fontWeight + size + referenceFont;
        referenceWidth = context.measureText(text).width;
        if (font != referenceFont) {
          context.font = fontWeight + size + font + ',' + referenceFont;
          var width = context.measureText(text).width;
          // If width and referenceWidth are the same, then the fallback was used
          // instead of the font we wanted, so the font is not available.
          available = available && width != referenceWidth;
        }
      }
      if (available) {
        // Consider font available when it is available in one weight range.
        //FIXME With this we miss rare corner cases, so we should consider
        //FIXME checking availability for each requested weight range.
        return true;
      }
    }
    return false;
  }

  function check() {
    var done = true;
    for (var font in checked) {
      if (checked[font] < retries) {
        if (isAvailable(font)) {
          checked[font] = retries;
          clear(textHeights);
          // Make sure that loaded fonts are picked up by Safari
          measureContext = null;
          labelCache.clear();
        } else {
          ++checked[font];
          done = false;
        }
      }
    }
    if (done) {
      clearInterval(interval);
      interval = undefined;
    }
  }

  return function(fontSpec) {
    var fontFamilies = getFontFamilies(fontSpec);
    if (!fontFamilies) {
      return;
    }
    for (var i = 0, ii = fontFamilies.length; i < ii; ++i) {
      var fontFamily = fontFamilies[i];
      if (!(fontFamily in checked)) {
        checked[fontFamily] = retries;
        if (!isAvailable(fontFamily)) {
          checked[fontFamily] = 0;
          if (interval === undefined) {
            interval = setInterval(check, 32);
          }
        }
      }
    }
  };
})();


/**
 * @return {CanvasRenderingContext2D} Measure context.
 */
function getMeasureContext() {
  if (!measureContext) {
    measureContext = createCanvasContext2D(1, 1);
  }
  return measureContext;
}


/**
 * @param {string} font Font to use for measuring.
 * @return {module:ol/size~Size} Measurement.
 */
export var measureTextHeight = (function() {
  var span;
  var heights = textHeights;
  return function(font) {
    var height = heights[font];
    if (height == undefined) {
      if (!span) {
        span = document.createElement('span');
        span.textContent = 'M';
        span.style.margin = span.style.padding = '0 !important';
        span.style.position = 'absolute !important';
        span.style.left = '-99999px !important';
      }
      span.style.font = font;
      document.body.appendChild(span);
      height = heights[font] = span.offsetHeight;
      document.body.removeChild(span);
    }
    return height;
  };
})();


/**
 * @param {string} font Font.
 * @param {string} text Text.
 * @return {number} Width.
 */
export function measureTextWidth(font, text) {
  var measureContext = getMeasureContext();
  if (font != measureContext.font) {
    measureContext.font = font;
  }
  return measureContext.measureText(text).width;
}


/**
 * @param {CanvasRenderingContext2D} context Context.
 * @param {number} rotation Rotation.
 * @param {number} offsetX X offset.
 * @param {number} offsetY Y offset.
 */
export function rotateAtOffset(context, rotation, offsetX, offsetY) {
  if (rotation !== 0) {
    context.translate(offsetX, offsetY);
    context.rotate(rotation);
    context.translate(-offsetX, -offsetY);
  }
}


export var resetTransform = createTransform();


/**
 * @param {CanvasRenderingContext2D} context Context.
 * @param {module:ol/transform~Transform|null} transform Transform.
 * @param {number} opacity Opacity.
 * @param {HTMLImageElement|HTMLCanvasElement|HTMLVideoElement} image Image.
 * @param {number} originX Origin X.
 * @param {number} originY Origin Y.
 * @param {number} w Width.
 * @param {number} h Height.
 * @param {number} x X.
 * @param {number} y Y.
 * @param {number} scale Scale.
 */
export function drawImage(context,
  transform, opacity, image, originX, originY, w, h, x, y, scale) {
  var alpha;
  if (opacity != 1) {
    alpha = context.globalAlpha;
    context.globalAlpha = alpha * opacity;
  }
  if (transform) {
    context.setTransform.apply(context, transform);
  }

  context.drawImage(image, originX, originY, w, h, x, y, w * scale, h * scale);

  if (alpha) {
    context.globalAlpha = alpha;
  }
  if (transform) {
    context.setTransform.apply(context, resetTransform);
  }
}

//# sourceMappingURL=canvas.js.map