/**
 * @module ol/webgl/Context
 */
import {getUid} from '../util.js';
import {EXTENSIONS as WEBGL_EXTENSIONS} from '../webgl.js';
import Disposable from '../Disposable.js';
import {includes} from '../array.js';
import {listen, unlistenAll} from '../events.js';
import {clear} from '../obj.js';
import {ARRAY_BUFFER, ELEMENT_ARRAY_BUFFER, TEXTURE_2D, TEXTURE_WRAP_S, TEXTURE_WRAP_T} from '../webgl.js';
import ContextEventType from '../webgl/ContextEventType.js';


/**
 * @typedef {Object} BufferCacheEntry
 * @property {import("./Buffer.js").default} buf
 * @property {WebGLBuffer} buffer
 */


/**
 * @classdesc
 * A WebGL context for accessing low-level WebGL capabilities.
 */
var WebGLContext = /*@__PURE__*/(function (Disposable) {
  function WebGLContext(canvas, gl) {
    Disposable.call(this);

    /**
     * @private
     * @type {HTMLCanvasElement}
     */
    this.canvas_ = canvas;

    /**
     * @private
     * @type {WebGLRenderingContext}
     */
    this.gl_ = gl;

    /**
     * @private
     * @type {!Object<string, BufferCacheEntry>}
     */
    this.bufferCache_ = {};

    /**
     * @private
     * @type {!Object<string, WebGLShader>}
     */
    this.shaderCache_ = {};

    /**
     * @private
     * @type {!Object<string, WebGLProgram>}
     */
    this.programCache_ = {};

    /**
     * @private
     * @type {WebGLProgram}
     */
    this.currentProgram_ = null;

    /**
     * @private
     * @type {WebGLFramebuffer}
     */
    this.hitDetectionFramebuffer_ = null;

    /**
     * @private
     * @type {WebGLTexture}
     */
    this.hitDetectionTexture_ = null;

    /**
     * @private
     * @type {WebGLRenderbuffer}
     */
    this.hitDetectionRenderbuffer_ = null;

    /**
     * @type {boolean}
     */
    this.hasOESElementIndexUint = includes(WEBGL_EXTENSIONS, 'OES_element_index_uint');

    // use the OES_element_index_uint extension if available
    if (this.hasOESElementIndexUint) {
      gl.getExtension('OES_element_index_uint');
    }

    listen(this.canvas_, ContextEventType.LOST,
      this.handleWebGLContextLost, this);
    listen(this.canvas_, ContextEventType.RESTORED,
      this.handleWebGLContextRestored, this);

  }

  if ( Disposable ) WebGLContext.__proto__ = Disposable;
  WebGLContext.prototype = Object.create( Disposable && Disposable.prototype );
  WebGLContext.prototype.constructor = WebGLContext;

  /**
   * Just bind the buffer if it's in the cache. Otherwise create
   * the WebGL buffer, bind it, populate it, and add an entry to
   * the cache.
   * @param {number} target Target.
   * @param {import("./Buffer.js").default} buf Buffer.
   */
  WebGLContext.prototype.bindBuffer = function bindBuffer (target, buf) {
    var gl = this.getGL();
    var arr = buf.getArray();
    var bufferKey = getUid(buf);
    if (bufferKey in this.bufferCache_) {
      var bufferCacheEntry = this.bufferCache_[bufferKey];
      gl.bindBuffer(target, bufferCacheEntry.buffer);
    } else {
      var buffer = gl.createBuffer();
      gl.bindBuffer(target, buffer);
      var /** @type {ArrayBufferView} */ arrayBuffer;
      if (target == ARRAY_BUFFER) {
        arrayBuffer = new Float32Array(arr);
      } else if (target == ELEMENT_ARRAY_BUFFER) {
        arrayBuffer = this.hasOESElementIndexUint ?
          new Uint32Array(arr) : new Uint16Array(arr);
      }
      gl.bufferData(target, arrayBuffer, buf.getUsage());
      this.bufferCache_[bufferKey] = {
        buf: buf,
        buffer: buffer
      };
    }
  };

  /**
   * @param {import("./Buffer.js").default} buf Buffer.
   */
  WebGLContext.prototype.deleteBuffer = function deleteBuffer (buf) {
    var gl = this.getGL();
    var bufferKey = getUid(buf);
    var bufferCacheEntry = this.bufferCache_[bufferKey];
    if (!gl.isContextLost()) {
      gl.deleteBuffer(bufferCacheEntry.buffer);
    }
    delete this.bufferCache_[bufferKey];
  };

  /**
   * @inheritDoc
   */
  WebGLContext.prototype.disposeInternal = function disposeInternal () {
    unlistenAll(this.canvas_);
    var gl = this.getGL();
    if (!gl.isContextLost()) {
      for (var key in this.bufferCache_) {
        gl.deleteBuffer(this.bufferCache_[key].buffer);
      }
      for (var key$1 in this.programCache_) {
        gl.deleteProgram(this.programCache_[key$1]);
      }
      for (var key$2 in this.shaderCache_) {
        gl.deleteShader(this.shaderCache_[key$2]);
      }
      // delete objects for hit-detection
      gl.deleteFramebuffer(this.hitDetectionFramebuffer_);
      gl.deleteRenderbuffer(this.hitDetectionRenderbuffer_);
      gl.deleteTexture(this.hitDetectionTexture_);
    }
  };

  /**
   * @return {HTMLCanvasElement} Canvas.
   */
  WebGLContext.prototype.getCanvas = function getCanvas () {
    return this.canvas_;
  };

  /**
   * Get the WebGL rendering context
   * @return {WebGLRenderingContext} The rendering context.
   * @api
   */
  WebGLContext.prototype.getGL = function getGL () {
    return this.gl_;
  };

  /**
   * Get the frame buffer for hit detection.
   * @return {WebGLFramebuffer} The hit detection frame buffer.
   */
  WebGLContext.prototype.getHitDetectionFramebuffer = function getHitDetectionFramebuffer () {
    if (!this.hitDetectionFramebuffer_) {
      this.initHitDetectionFramebuffer_();
    }
    return this.hitDetectionFramebuffer_;
  };

  /**
   * Get shader from the cache if it's in the cache. Otherwise, create
   * the WebGL shader, compile it, and add entry to cache.
   * @param {import("./Shader.js").default} shaderObject Shader object.
   * @return {WebGLShader} Shader.
   */
  WebGLContext.prototype.getShader = function getShader (shaderObject) {
    var shaderKey = getUid(shaderObject);
    if (shaderKey in this.shaderCache_) {
      return this.shaderCache_[shaderKey];
    } else {
      var gl = this.getGL();
      var shader = gl.createShader(shaderObject.getType());
      gl.shaderSource(shader, shaderObject.getSource());
      gl.compileShader(shader);
      this.shaderCache_[shaderKey] = shader;
      return shader;
    }
  };

  /**
   * Get the program from the cache if it's in the cache. Otherwise create
   * the WebGL program, attach the shaders to it, and add an entry to the
   * cache.
   * @param {import("./Fragment.js").default} fragmentShaderObject Fragment shader.
   * @param {import("./Vertex.js").default} vertexShaderObject Vertex shader.
   * @return {WebGLProgram} Program.
   */
  WebGLContext.prototype.getProgram = function getProgram (fragmentShaderObject, vertexShaderObject) {
    var programKey = getUid(fragmentShaderObject) + '/' + getUid(vertexShaderObject);
    if (programKey in this.programCache_) {
      return this.programCache_[programKey];
    } else {
      var gl = this.getGL();
      var program = gl.createProgram();
      gl.attachShader(program, this.getShader(fragmentShaderObject));
      gl.attachShader(program, this.getShader(vertexShaderObject));
      gl.linkProgram(program);
      this.programCache_[programKey] = program;
      return program;
    }
  };

  /**
   * FIXME empty description for jsdoc
   */
  WebGLContext.prototype.handleWebGLContextLost = function handleWebGLContextLost () {
    clear(this.bufferCache_);
    clear(this.shaderCache_);
    clear(this.programCache_);
    this.currentProgram_ = null;
    this.hitDetectionFramebuffer_ = null;
    this.hitDetectionTexture_ = null;
    this.hitDetectionRenderbuffer_ = null;
  };

  /**
   * FIXME empty description for jsdoc
   */
  WebGLContext.prototype.handleWebGLContextRestored = function handleWebGLContextRestored () {
  };

  /**
   * Creates a 1x1 pixel framebuffer for the hit-detection.
   * @private
   */
  WebGLContext.prototype.initHitDetectionFramebuffer_ = function initHitDetectionFramebuffer_ () {
    var gl = this.gl_;
    var framebuffer = gl.createFramebuffer();
    gl.bindFramebuffer(gl.FRAMEBUFFER, framebuffer);

    var texture = createEmptyTexture(gl, 1, 1);
    var renderbuffer = gl.createRenderbuffer();
    gl.bindRenderbuffer(gl.RENDERBUFFER, renderbuffer);
    gl.renderbufferStorage(gl.RENDERBUFFER, gl.DEPTH_COMPONENT16, 1, 1);
    gl.framebufferTexture2D(
      gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, texture, 0);
    gl.framebufferRenderbuffer(gl.FRAMEBUFFER, gl.DEPTH_ATTACHMENT,
      gl.RENDERBUFFER, renderbuffer);

    gl.bindTexture(gl.TEXTURE_2D, null);
    gl.bindRenderbuffer(gl.RENDERBUFFER, null);
    gl.bindFramebuffer(gl.FRAMEBUFFER, null);

    this.hitDetectionFramebuffer_ = framebuffer;
    this.hitDetectionTexture_ = texture;
    this.hitDetectionRenderbuffer_ = renderbuffer;
  };

  /**
   * Use a program.  If the program is already in use, this will return `false`.
   * @param {WebGLProgram} program Program.
   * @return {boolean} Changed.
   * @api
   */
  WebGLContext.prototype.useProgram = function useProgram (program) {
    if (program == this.currentProgram_) {
      return false;
    } else {
      var gl = this.getGL();
      gl.useProgram(program);
      this.currentProgram_ = program;
      return true;
    }
  };

  return WebGLContext;
}(Disposable));


/**
 * @param {WebGLRenderingContext} gl WebGL rendering context.
 * @param {number=} opt_wrapS wrapS.
 * @param {number=} opt_wrapT wrapT.
 * @return {WebGLTexture} The texture.
 */
function createTextureInternal(gl, opt_wrapS, opt_wrapT) {
  var texture = gl.createTexture();
  gl.bindTexture(gl.TEXTURE_2D, texture);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
  gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);

  if (opt_wrapS !== undefined) {
    gl.texParameteri(
      TEXTURE_2D, TEXTURE_WRAP_S, opt_wrapS);
  }
  if (opt_wrapT !== undefined) {
    gl.texParameteri(
      TEXTURE_2D, TEXTURE_WRAP_T, opt_wrapT);
  }

  return texture;
}


/**
 * @param {WebGLRenderingContext} gl WebGL rendering context.
 * @param {number} width Width.
 * @param {number} height Height.
 * @param {number=} opt_wrapS wrapS.
 * @param {number=} opt_wrapT wrapT.
 * @return {WebGLTexture} The texture.
 */
export function createEmptyTexture(gl, width, height, opt_wrapS, opt_wrapT) {
  var texture = createTextureInternal(gl, opt_wrapS, opt_wrapT);
  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, width, height, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
  return texture;
}


/**
 * @param {WebGLRenderingContext} gl WebGL rendering context.
 * @param {HTMLCanvasElement|HTMLImageElement|HTMLVideoElement} image Image.
 * @param {number=} opt_wrapS wrapS.
 * @param {number=} opt_wrapT wrapT.
 * @return {WebGLTexture} The texture.
 */
export function createTexture(gl, image, opt_wrapS, opt_wrapT) {
  var texture = createTextureInternal(gl, opt_wrapS, opt_wrapT);
  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, image);
  return texture;
}

export default WebGLContext;

//# sourceMappingURL=Context.js.map